"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEsSearchIterable = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Creates an `AsyncIterable` that can be used to iterate (ex. via `for..await..of`) over all the data
 * matching the search query. The search request to ES will use `search_after`, thus can iterate over
 * datasets above 10k items as well.
 *
 * @param options
 *
 * @example
 *
 *  const yourFn = async () => {
 *    const dataIterable = createEsSearchIterable({
 *      esClient,
 *      searchRequest: {
 *        index: 'some-index',
 *        sort: [
 *          {
 *            created: { order: 'asc' }
 *          }
 *        ]
 *      }
 *    });
 *
 *    for await (const data of dataIterable) {
 *      // data === your search results
 *    }
 *  }
 */
const createEsSearchIterable = ({
  esClient,
  searchRequest: {
    size = 1000,
    index,
    ...searchOptions
  },
  resultsMapper,
  usePointInTime = true
}) => {
  const keepAliveValue = '5m';
  let done = false;
  let value;
  let searchAfterValue;
  let pointInTime = usePointInTime ? esClient.openPointInTime({
    index,
    ignore_unavailable: true,
    keep_alive: keepAliveValue
  }) : Promise.resolve({
    id: ''
  });
  const createIteratorResult = () => {
    return {
      done,
      value
    };
  };
  const setValue = searchResponse => {
    value = resultsMapper ? resultsMapper(searchResponse) : searchResponse;
  };
  const setDone = async () => {
    done = true;
    if (usePointInTime) {
      const pitId = (await pointInTime).id;
      if (pitId) {
        await esClient.closePointInTime({
          id: pitId
        });
      }
    }
  };
  const fetchData = async () => {
    var _searchResult$pit_id;
    const pitId = (await pointInTime).id;
    const searchResult = await esClient.search({
      ...searchOptions,
      size,
      ...(usePointInTime ? {
        pit: {
          id: pitId,
          keep_alive: keepAliveValue
        }
      } : {
        index
      }),
      search_after: searchAfterValue
    }).catch(e => {
      Error.captureStackTrace(e);
      throw e;
    });
    const searchHits = searchResult.hits.hits;
    const lastSearchHit = searchHits[searchHits.length - 1];
    if (searchHits.length === 0) {
      await setDone();
      return;
    }
    searchAfterValue = lastSearchHit.sort;
    pointInTime = Promise.resolve({
      id: (_searchResult$pit_id = searchResult.pit_id) !== null && _searchResult$pit_id !== void 0 ? _searchResult$pit_id : ''
    });
    setValue(searchResult);

    // If (for some reason) we don't have a `searchAfterValue`,
    // then throw an error, or else we'll keep looping forever
    if (!searchAfterValue) {
      await setDone();
      throw new Error(`Unable to store 'search_after' value. Last 'SearchHit' did not include a 'sort' property \n(did you forget to set the 'sort' attribute on your SearchRequest?)':\n${JSON.stringify(lastSearchHit)}`);
    }
  };
  return {
    [Symbol.asyncIterator]() {
      return {
        async next() {
          if (!done) {
            await fetchData();
          }
          return createIteratorResult();
        },
        async return() {
          done = true;
          return createIteratorResult();
        }
      };
    }
  };
};
exports.createEsSearchIterable = createEsSearchIterable;