"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NOT_SUGGESTED_TYPES = void 0;
exports.getFieldsByTypeHelper = getFieldsByTypeHelper;
exports.getPolicyHelper = getPolicyHelper;
exports.getSourcesHelper = getSourcesHelper;
var _esqlAst = require("@kbn/esql-ast");
var _helpers = require("./helpers");
var _query_string_utils = require("./query_string_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const NOT_SUGGESTED_TYPES = exports.NOT_SUGGESTED_TYPES = ['unsupported'];
const cache = new Map();

// Function to check if a key exists in the cache, ignoring case
function checkCacheInsensitive(keyToCheck) {
  for (const key of cache.keys()) {
    if (key.toLowerCase() === keyToCheck.toLowerCase()) {
      return true; // Or return the value associated with the key if needed: return cache.get(key);
    }
  }
  return false;
}

// Function to get a value from the cache, ignoring case
function getValueInsensitive(keyToCheck) {
  for (const key of cache.keys()) {
    if (key.toLowerCase() === keyToCheck.toLowerCase()) {
      return cache.get(key);
    }
  }
  return undefined;
}

/**
 * Given a query, this function will compute the available fields and cache them
 * for the next time the same query is used.
 * @param queryText
 */
async function cacheFieldsForQuery(queryText) {
  const existsInCache = checkCacheInsensitive(queryText);
  if (existsInCache) {
    // this is already in the cache
    return;
  }
  const queryTextWithoutLastPipe = (0, _query_string_utils.removeLastPipe)(queryText);
  // retrieve the user defined fields from the query without an extra call
  const fieldsAvailableAfterPreviousCommand = getValueInsensitive(queryTextWithoutLastPipe);
  if (fieldsAvailableAfterPreviousCommand && fieldsAvailableAfterPreviousCommand !== null && fieldsAvailableAfterPreviousCommand !== void 0 && fieldsAvailableAfterPreviousCommand.length) {
    const {
      root
    } = (0, _esqlAst.parse)(queryText);
    const availableFields = await (0, _helpers.getCurrentQueryAvailableFields)(queryText, root.commands, fieldsAvailableAfterPreviousCommand);
    cache.set(queryText, availableFields);
  }
}
function getFieldsByTypeHelper(queryText, resourceRetriever) {
  const getFields = async () => {
    // in some cases (as in the case of ROW or SHOW) the query is not set
    if (!queryText) {
      return;
    }
    const [sourceCommand, ...partialQueries] = (0, _query_string_utils.processPipes)(queryText);

    // retrieve the index fields from ES ONLY if the source command is not in the cache
    const existsInCache = getValueInsensitive(sourceCommand);
    if (!existsInCache) {
      const fieldsWithMetadata = await (0, _helpers.getFieldsFromES)(sourceCommand, resourceRetriever);
      cache.set(sourceCommand, fieldsWithMetadata);
    }

    // build fields cache for every partial query
    for (const query of partialQueries) {
      await cacheFieldsForQuery(query);
    }
  };
  return {
    getFieldsByType: async (expectedType = 'any', ignored = []) => {
      const types = Array.isArray(expectedType) ? expectedType : [expectedType];
      await getFields();
      const queryTextForCacheSearch = (0, _query_string_utils.toSingleLine)(queryText);
      const cachedFields = getValueInsensitive(queryTextForCacheSearch);
      return (cachedFields === null || cachedFields === void 0 ? void 0 : cachedFields.filter(({
        name,
        type
      }) => {
        const ts = Array.isArray(type) ? type : [type];
        return !ignored.includes(name) && (types[0] === 'any' ||
        // if the type is 'any' we don't need to check the type
        ts.some(t => types.includes(t))) && !NOT_SUGGESTED_TYPES.includes(type);
      })) || [];
    },
    getFieldsMap: async () => {
      await getFields();
      const queryTextForCacheSearch = (0, _query_string_utils.toSingleLine)(queryText);
      const cachedFields = getValueInsensitive(queryTextForCacheSearch);
      const cacheCopy = new Map();
      cachedFields === null || cachedFields === void 0 ? void 0 : cachedFields.forEach(field => cacheCopy.set(field.name, field));
      return cacheCopy;
    }
  };
}
function getPolicyHelper(resourceRetriever) {
  const getPolicies = async () => {
    var _resourceRetriever$ge;
    return (await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ge = resourceRetriever.getPolicies) === null || _resourceRetriever$ge === void 0 ? void 0 : _resourceRetriever$ge.call(resourceRetriever))) || [];
  };
  return {
    getPolicies: async () => {
      const policies = await getPolicies();
      return policies;
    },
    getPolicyMetadata: async policyName => {
      const policies = await getPolicies();
      return policies.find(({
        name
      }) => name === policyName);
    }
  };
}
function getSourcesHelper(resourceRetriever) {
  return async () => {
    var _resourceRetriever$ge2;
    return (await (resourceRetriever === null || resourceRetriever === void 0 ? void 0 : (_resourceRetriever$ge2 = resourceRetriever.getSources) === null || _resourceRetriever$ge2 === void 0 ? void 0 : _resourceRetriever$ge2.call(resourceRetriever))) || [];
  };
}