"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.whereCompleteItem = exports.getPosition = exports.getDateHistogramCompletionItem = exports.byCompleteItem = void 0;
var _esqlAst = require("@kbn/esql-ast");
var _i18n = require("@kbn/i18n");
var _helpers = require("../../../shared/helpers");
var _factories = require("../../factories");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Position of the caret in the sort command:
*
* ```
* STATS [column1 =] expression1[, ..., [columnN =] expressionN] [BY [column1 =] grouping_expression1[, ..., grouping_expressionN]]
        |           |          |                                    |           |                   |
        |           |          expression_complete                  |           |                   grouping_expression_complete
        |           expression_after_assignment                     |           grouping_expression_after_assignment
        expression_without_assignment                               grouping_expression_without_assignment

* ```
*/

const getPosition = (innerText, command) => {
  const lastCommandArg = command.args[command.args.length - 1];
  if ((0, _helpers.isOptionItem)(lastCommandArg) && lastCommandArg.name === 'by') {
    // in the BY clause

    const lastOptionArg = lastCommandArg.args[lastCommandArg.args.length - 1];
    if ((0, _helpers.isAssignment)(lastOptionArg) && !(0, _helpers.isAssignmentComplete)(lastOptionArg)) {
      return 'grouping_expression_after_assignment';
    }

    // check if the cursor follows a comma or the BY keyword
    // optionally followed by a fragment of a word
    // e.g. ", field/"
    if (/\,\s+\S*$/.test(innerText) || (0, _helpers.noCaseCompare)((0, _helpers.findPreviousWord)(innerText), 'by')) {
      return 'grouping_expression_without_assignment';
    } else {
      return 'grouping_expression_complete';
    }
  }
  if ((0, _helpers.isAssignment)(lastCommandArg) && !(0, _helpers.isAssignmentComplete)(lastCommandArg)) {
    return 'expression_after_assignment';
  }
  if ((0, _helpers.getLastNonWhitespaceChar)(innerText) === ',' || /stats\s+\S*$/i.test(innerText)) {
    return 'expression_without_assignment';
  }
  if ((0, _esqlAst.isFunctionExpression)(lastCommandArg) && lastCommandArg.name === 'where') {
    return 'after_where';
  }
  return 'expression_complete';
};
exports.getPosition = getPosition;
const byCompleteItem = exports.byCompleteItem = {
  label: 'BY',
  text: 'BY ',
  kind: 'Reference',
  detail: 'By',
  sortText: '1',
  command: _factories.TRIGGER_SUGGESTION_COMMAND
};
const whereCompleteItem = exports.whereCompleteItem = {
  label: 'WHERE',
  text: 'WHERE ',
  kind: 'Reference',
  detail: 'Where',
  sortText: '1',
  command: _factories.TRIGGER_SUGGESTION_COMMAND
};
const getDateHistogramCompletionItem = (histogramBarTarget = 50) => ({
  label: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.addDateHistogram', {
    defaultMessage: 'Add date histogram'
  }),
  text: `BUCKET($0, ${histogramBarTarget}, ${_factories.TIME_SYSTEM_PARAMS.join(', ')})`,
  asSnippet: true,
  kind: 'Issue',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.autocomplete.addDateHistogramDetail', {
    defaultMessage: 'Add date histogram using bucket()'
  }),
  sortText: '1',
  command: _factories.TRIGGER_SUGGESTION_COMMAND
});
exports.getDateHistogramCompletionItem = getDateHistogramCompletionItem;