"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.onSuggestion = exports.getPosition = exports.asSuggestion = exports.Position = void 0;
exports.suggest = suggest;
var _i18n = require("@kbn/i18n");
var _esql_types = require("../../../shared/esql_types");
var _helpers = require("../../../shared/helpers");
var _complete_items = require("../../complete_items");
var _factories = require("../../factories");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
let Position = exports.Position = /*#__PURE__*/function (Position) {
  Position["VALUE"] = "value";
  Position["AFTER_VALUE"] = "after_value";
  Position["ON_COLUMN"] = "on_column";
  Position["AFTER_ON_CLAUSE"] = "after_on_clause";
  Position["AS_TYPE_COLUMN"] = "as_type_clause";
  Position["AS_P_VALUE_COLUMN"] = "as_p_value_column";
  Position["AFTER_AS_CLAUSE"] = "after_as_clause";
  return Position;
}({});
const getPosition = (innerText, command) => {
  if (command.args.length < 2) {
    if (innerText.match(/CHANGE_POINT\s+\S*$/i)) {
      return Position.VALUE;
    }
    if (innerText.match(/CHANGE_POINT\s+\S+\s*\S*$/i)) {
      return Position.AFTER_VALUE;
    }
  }
  const lastArg = command.args[command.args.length - 1];
  if (innerText.match(/on\s+\S*$/i)) {
    return Position.ON_COLUMN;
  }
  if ((0, _helpers.isSingleItem)(lastArg) && lastArg.name === 'on') {
    if (innerText.match(/on\s+\S+\s+$/i)) {
      return Position.AFTER_ON_CLAUSE;
    }
  }
  if (innerText.match(/as\s+$/i)) {
    return Position.AS_TYPE_COLUMN;
  }
  if ((0, _helpers.isSingleItem)(lastArg) && lastArg.name === 'as') {
    if (innerText.match(/as\s+\S+,\s*\S*$/i)) {
      return Position.AS_P_VALUE_COLUMN;
    }
    if (innerText.match(/as\s+\S+,\s*\S+\s+$/i)) {
      return Position.AFTER_AS_CLAUSE;
    }
  }
};
exports.getPosition = getPosition;
const onSuggestion = exports.onSuggestion = {
  label: 'ON',
  text: 'ON ',
  kind: 'Reference',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definitions.onDoc', {
    defaultMessage: 'On'
  }),
  sortText: '1',
  command: _factories.TRIGGER_SUGGESTION_COMMAND
};
const asSuggestion = exports.asSuggestion = {
  label: 'AS',
  text: 'AS ',
  kind: 'Reference',
  detail: _i18n.i18n.translate('kbn-esql-validation-autocomplete.esql.definitions.asDoc', {
    defaultMessage: 'As'
  }),
  sortText: '2',
  command: _factories.TRIGGER_SUGGESTION_COMMAND
};
async function suggest({
  innerText,
  command,
  getColumnsByType
}) {
  const pos = getPosition(innerText, command);
  switch (pos) {
    case Position.VALUE:
      const numericFields = await getColumnsByType(_esql_types.ESQL_NUMBER_TYPES, [], {
        advanceCursor: true,
        openSuggestions: true
      });
      const lastWord = (0, _helpers.findFinalWord)(innerText);
      if (lastWord !== '') {
        numericFields.forEach(fieldSuggestion => {
          fieldSuggestion.rangeToReplace = {
            start: innerText.length - lastWord.length + 1,
            end: innerText.length + 1
          };
        });
      }
      return numericFields;
    case Position.AFTER_VALUE:
      {
        return [onSuggestion, asSuggestion, _complete_items.pipeCompleteItem];
      }
    case Position.ON_COLUMN:
      {
        const onFields = await getColumnsByType('any', [], {
          advanceCursor: true,
          openSuggestions: true
        });
        return onFields;
      }
    case Position.AFTER_ON_CLAUSE:
      return [asSuggestion, _complete_items.pipeCompleteItem];
    case Position.AS_TYPE_COLUMN:
      {
        // add comma and space
        return (0, _factories.buildUserDefinedColumnsDefinitions)(['changePointType']).map(v => ({
          ...v,
          text: v.text + ', ',
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        }));
      }
    case Position.AS_P_VALUE_COLUMN:
      {
        return (0, _factories.buildUserDefinedColumnsDefinitions)(['pValue']).map(v => ({
          ...v,
          command: _factories.TRIGGER_SUGGESTION_COMMAND
        }));
      }
    case Position.AFTER_AS_CLAUSE:
      {
        return [_complete_items.pipeCompleteItem];
      }
    default:
      return [];
  }
}