"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.removeEmptyValueBoosts = exports.removeBoostStateProps = exports.parseBoostCenter = exports.normalizeBoostValues = exports.filterIfTerm = void 0;
var _lodash = require("lodash");
var _types = require("../../../shared/schema/types");
var _types2 = require("./types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// If the user hasn't entered a filter, then we can skip filtering the array entirely
const filterIfTerm = (array, filterTerm) => {
  return filterTerm === '' ? array : array.filter(item => item.includes(filterTerm));
};
exports.filterIfTerm = filterIfTerm;
const removeBoostStateProps = searchSettings => {
  const {
    precision_enabled: precisionEnabled,
    ...updatedSettings
  } = (0, _lodash.cloneDeep)(searchSettings);
  const {
    boosts
  } = updatedSettings;
  const keys = Object.keys(boosts);
  keys.forEach(key => boosts[key].forEach(boost => delete boost.newBoost));
  return updatedSettings;
};
exports.removeBoostStateProps = removeBoostStateProps;
const parseBoostCenter = (fieldType, value) => {
  // Leave non-numeric fields alone
  if (fieldType === _types.SchemaType.Number) {
    const floatValue = parseFloat(value);
    return isNaN(floatValue) ? value : floatValue;
  }
  return value;
};
exports.parseBoostCenter = parseBoostCenter;
const toArray = v => Array.isArray(v) ? v : [v];
const toString = v1 => String(v1);
const normalizeBoostValue = boost => {
  if (!Object.hasOwn(boost, 'value')) {
    // Can't simply do `return boost` here as TS can't infer the correct type
    return (0, _lodash.omit)(boost, 'value');
  }
  return {
    ...boost,
    type: boost.type,
    value: toArray(boost.value).map(toString)
  };
};

// Data may have been set to invalid types directly via the public App Search API. Since these are invalid, we don't want to deal
// with them as valid types in the UI. For that reason, we stringify all values here, as the data comes in.
// Additionally, values can be in single values or in arrays.
const normalizeBoostValues = boosts => Object.entries(boosts).reduce((newBoosts, [fieldName, boostList]) => {
  return {
    ...newBoosts,
    [fieldName]: boostList.map(normalizeBoostValue)
  };
}, {});

// Our model allows for empty values to be added to boosts. However, the server will not accept
// empty strings in values. To avoid that, we filter out empty values before sending them to the server.

// I.e., the server will not accept any of the following, so we need to filter them out
// value: undefined
// value: []
// value: ['']
// value: ['foo', '']
exports.normalizeBoostValues = normalizeBoostValues;
const removeEmptyValueBoosts = boosts => {
  // before:
  //   { foo: { values: ['a', '', '   '] } }
  //   { foo: { values: [''] } }
  // after:
  //   { foo: { values: ['a'] } }
  const filterEmptyValueBoosts = fieldBoosts => {
    return fieldBoosts.filter(boost => {
      if (boost.type !== _types2.BoostType.Value) return true;
      const valueBoost = boost;
      const filteredValues = valueBoost.value.filter(value => value.trim() !== '');
      if (filteredValues.length) {
        boost.value = filteredValues;
        return true;
      } else {
        return false;
      }
    });
  };
  return Object.entries(boosts).reduce((acc, [fieldName, fieldBoosts]) => {
    const updatedBoosts = filterEmptyValueBoosts(fieldBoosts);
    return updatedBoosts.length ? {
      ...acc,
      [fieldName]: updatedBoosts
    } : acc;
  }, {});
};
exports.removeEmptyValueBoosts = removeEmptyValueBoosts;