"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isModelAlreadyExistsError = exports.getStructuredToolForIndexEntry = exports.getKBVectorSearchQuery = void 0;
var _zod = require("@kbn/zod");
var _tools = require("@langchain/core/tools");
var _elasticsearch = require("@elastic/elasticsearch");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isModelAlreadyExistsError = error => {
  return error instanceof _elasticsearch.errors.ResponseError && (error.body.error.type === 'resource_not_found_exception' || error.body.error.type === 'status_exception');
};

/**
 * Returns an Elasticsearch query DSL that performs a vector search against the Knowledge Base for the given query/user/filter. Searches only for DocumentEntries, not IndexEntries as they have no content.
 *
 * @param filter - Optional filter to apply to the search
 * @param kbResource - Specific resource tag to filter for, e.g. 'esql' or 'user'
 * @param query - The search query provided by the user
 * @param required - Whether to only include required entries
 * @param user - The authenticated user
 * @returns
 */
exports.isModelAlreadyExistsError = isModelAlreadyExistsError;
const getKBVectorSearchQuery = ({
  filter,
  kbResource,
  query,
  required,
  user
}) => {
  const resourceFilter = kbResource ? [{
    term: {
      kb_resource: kbResource
    }
  }] : [];
  const requiredFilter = required ? [{
    term: {
      required
    }
  }] : [];
  const userFilter = {
    should: [{
      nested: {
        path: 'users',
        query: {
          bool: {
            minimum_should_match: 1,
            should: [{
              match: user.profile_uid ? {
                'users.id': user.profile_uid
              } : {
                'users.name': user.username
              }
            }]
          }
        }
      }
    }, {
      bool: {
        must_not: [{
          nested: {
            path: 'users',
            query: {
              bool: {
                filter: {
                  exists: {
                    field: 'users'
                  }
                }
              }
            }
          }
        }]
      }
    }]
  };
  let semanticTextFilter = [];
  if (query) {
    semanticTextFilter = [{
      semantic: {
        field: 'semantic_text',
        query
      }
    }];
  }
  return {
    bool: {
      must: [...semanticTextFilter, ...requiredFilter, ...resourceFilter],
      ...userFilter,
      filter,
      minimum_should_match: 1
    }
  };
};

/**
 * Returns a StructuredTool for a given IndexEntry
 */
exports.getKBVectorSearchQuery = getKBVectorSearchQuery;
const getStructuredToolForIndexEntry = ({
  indexEntry,
  esClient,
  contentReferencesStore,
  logger
}) => {
  var _indexEntry$inputSche;
  const inputSchema = (_indexEntry$inputSche = indexEntry.inputSchema) === null || _indexEntry$inputSche === void 0 ? void 0 : _indexEntry$inputSche.reduce((prev, input) => {
    const fieldType = input.fieldType === 'string' ? _zod.z.string() : input.fieldType === 'number' ? _zod.z.number() : input.fieldType === 'boolean' ? _zod.z.boolean() : _zod.z.any();
    return {
      ...prev,
      [input.fieldName]: fieldType.describe(input.description)
    };
  }, {});
  return new _tools.DynamicStructuredTool({
    name: indexEntry.name
    // Replace invalid characters with an empty string
    .replace(/[^a-zA-Z0-9_]/g, '')
    // Ensure it starts with a letter. If not, prepend 'a'
    .replace(/^[^a-zA-Z]/, 'a'),
    description: indexEntry.description,
    schema: _zod.z.object({
      query: _zod.z.string().describe(indexEntry.queryDescription),
      ...inputSchema
    }),
    func: async (input, _, cbManager) => {
      var _indexEntry$inputSche2, _indexEntry$inputSche3;
      logger.debug(() => `Generated ${indexEntry.name} Tool:input\n ${JSON.stringify(input, null, 2)}`);

      // Generate filters for inputSchema fields
      const filter = (_indexEntry$inputSche2 = (_indexEntry$inputSche3 = indexEntry.inputSchema) === null || _indexEntry$inputSche3 === void 0 ? void 0 : _indexEntry$inputSche3.reduce(
      // @ts-expect-error upgrade typescript v5.9.3
      (prev, i) => [...prev, {
        term: {
          [`${i.fieldName}`]: input === null || input === void 0 ? void 0 : input[i.fieldName]
        }
      }], [])) !== null && _indexEntry$inputSche2 !== void 0 ? _indexEntry$inputSche2 : [];
      const params = {
        index: indexEntry.index,
        size: 100,
        query: {
          bool: {
            must: [{
              semantic: {
                field: indexEntry.field,
                query: input.query
              }
            }],
            filter
          }
        },
        highlight: {
          fields: {
            [indexEntry.field]: {
              type: 'semantic',
              number_of_fragments: 2,
              order: 'score'
            }
          }
        }
      };
      try {
        const result = await esClient.search(params);
        const kbDocs = result.hits.hits.map(hit => {
          var _hit$highlight;
          const reference = contentReferencesStore.add(p => createReference(p.id, hit));
          if (indexEntry.outputFields && indexEntry.outputFields.length > 0) {
            return indexEntry.outputFields.reduce((prev, field) => {
              // @ts-expect-error
              return {
                ...prev,
                [field]: hit._source[field]
              };
            }, {
              citation: (0, _elasticAssistantCommon.contentReferenceBlock)(reference)
            });
          }
          return {
            text: (_hit$highlight = hit.highlight) === null || _hit$highlight === void 0 ? void 0 : _hit$highlight[indexEntry.field].join('\n --- \n'),
            citation: (0, _elasticAssistantCommon.contentReferenceBlock)(reference)
          };
        });
        logger.debug(() => `Similarity Search Params:\n ${JSON.stringify(params)}`);
        logger.debug(() => `Similarity Search Results:\n ${JSON.stringify(result)}`);
        logger.debug(() => `Similarity Text Extract Results:\n ${JSON.stringify(kbDocs)}`);
        const prunedResult = JSON.stringify(kbDocs).substring(0, 20000);
        return `###\nBelow are all relevant documents in JSON format:\n${prunedResult}###`;
      } catch (e) {
        logger.error(`Error performing IndexEntry KB Similarity Search: ${e.message}`);
        return `I'm sorry, but I was unable to find any information in the knowledge base. Perhaps this error would be useful to deliver to the user. Be sure to print it below your response and in a codeblock so it is rendered nicely: ${e.message}`;
      }
    },
    tags: ['knowledge-base']
    // TODO: Remove after ZodAny is fixed https://github.com/langchain-ai/langchainjs/blob/main/langchain-core/src/tools.ts
  });
};
exports.getStructuredToolForIndexEntry = getStructuredToolForIndexEntry;
const createReference = (id, hit) => {
  const hitIndex = hit._index;
  const hitId = hit._id;
  const esqlQuery = `FROM ${hitIndex} ${hitId ? `METADATA _id\n | WHERE _id == "${hitId}"` : ''}`;
  let timerange;
  const source = hit._source;
  if ('@timestamp' in source && (0, _lodash.isString)(source['@timestamp']) && hitId) {
    timerange = {
      from: source['@timestamp'],
      to: source['@timestamp']
    };
  }
  return (0, _elasticAssistantCommon.esqlQueryReference)({
    id,
    query: esqlQuery,
    label: `Index: ${hit._index}`,
    timerange
  });
};