"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_ITEMS_PER_PAGE = void 0;
exports.getApmTraceError = getApmTraceError;
exports.getTraceItems = getTraceItems;
exports.requiredFields = exports.optionalFields = void 0;
var _server = require("@kbn/observability-plugin/server");
var _lodash = require("lodash");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
var _get_linked_children = require("../span_links/get_linked_children");
var _document_type = require("../../../common/document_type");
var _rollup = require("../../../common/rollup");
var _get_trace_docs_per_page = require("./get_trace_docs_per_page");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const requiredFields = exports.requiredFields = (0, _as_mutable_array.asMutableArray)([_apm.TIMESTAMP_US, _apm.TRACE_ID, _apm.SERVICE_NAME, _apm.ERROR_ID, _apm.ERROR_GROUP_ID, _apm.PROCESSOR_EVENT]);
const optionalFields = exports.optionalFields = (0, _as_mutable_array.asMutableArray)([_apm.PARENT_ID, _apm.TRANSACTION_ID, _apm.SPAN_ID, _apm.SPAN_DESTINATION_SERVICE_RESOURCE, _apm.ERROR_CULPRIT, _apm.ERROR_LOG_MESSAGE, _apm.ERROR_EXC_MESSAGE, _apm.ERROR_EXC_HANDLED, _apm.ERROR_EXC_TYPE]);
async function getTraceItems({
  traceId,
  config,
  apmEventClient,
  start,
  end,
  maxTraceItemsFromUrlParam,
  logger
}) {
  const maxTraceItems = maxTraceItemsFromUrlParam !== null && maxTraceItemsFromUrlParam !== void 0 ? maxTraceItemsFromUrlParam : config.ui.maxTraceItems;
  const errorResponsePromise = getApmTraceError({
    apmEventClient,
    traceId,
    start,
    end
  });
  const traceResponsePromise = getTraceDocsPaginated({
    apmEventClient,
    maxTraceItems,
    traceId,
    start,
    end,
    logger
  });
  const [errorDocs, traceResponse, spanLinksCountById] = await Promise.all([errorResponsePromise, traceResponsePromise, (0, _get_linked_children.getSpanLinksCountById)({
    traceId,
    apmEventClient,
    start,
    end
  })]);
  const traceDocsTotal = traceResponse.total;
  const exceedsMax = traceDocsTotal > maxTraceItems;
  const traceDocs = traceResponse.hits.map(({
    hit
  }) => hit);
  return {
    exceedsMax,
    traceDocs,
    errorDocs,
    spanLinksCountById,
    traceDocsTotal,
    maxTraceItems
  };
}
const MAX_ITEMS_PER_PAGE = exports.MAX_ITEMS_PER_PAGE = 10000; // 10000 is the max allowed by ES
const excludedLogLevels = ['debug', 'info', 'warning'];
async function getApmTraceError({
  apmEventClient,
  traceId,
  start,
  end
}) {
  const response = await apmEventClient.search('get_errors_docs', {
    apm: {
      sources: [{
        documentType: _document_type.ApmDocumentType.ErrorEvent,
        rollupInterval: _rollup.RollupInterval.None
      }]
    },
    body: {
      track_total_hits: false,
      size: 1000,
      query: {
        bool: {
          filter: [{
            term: {
              [_apm.TRACE_ID]: traceId
            }
          }, ...(0, _server.rangeQuery)(start, end)],
          must_not: {
            terms: {
              [_apm.ERROR_LOG_LEVEL]: excludedLogLevels
            }
          }
        }
      },
      fields: [...requiredFields, ...optionalFields],
      _source: [_apm.ERROR_LOG_MESSAGE, _apm.ERROR_EXC_MESSAGE, _apm.ERROR_EXC_HANDLED, _apm.ERROR_EXC_TYPE]
    }
  });
  return response.hits.hits.map(hit => {
    var _event$parent$id, _event$parent, _event$span, _event$error, _errorSource$error$ex, _errorSource$error$ex2;
    const errorSource = 'error' in hit._source ? hit._source : undefined;
    const event = (0, _utils.unflattenKnownApmEventFields)(hit.fields, requiredFields);
    const waterfallErrorEvent = {
      ...event,
      parent: {
        ...(event === null || event === void 0 ? void 0 : event.parent),
        id: (_event$parent$id = event === null || event === void 0 ? void 0 : (_event$parent = event.parent) === null || _event$parent === void 0 ? void 0 : _event$parent.id) !== null && _event$parent$id !== void 0 ? _event$parent$id : event === null || event === void 0 ? void 0 : (_event$span = event.span) === null || _event$span === void 0 ? void 0 : _event$span.id
      },
      error: {
        ...((_event$error = event.error) !== null && _event$error !== void 0 ? _event$error : {}),
        exception: ((_errorSource$error$ex = errorSource === null || errorSource === void 0 ? void 0 : (_errorSource$error$ex2 = errorSource.error.exception) === null || _errorSource$error$ex2 === void 0 ? void 0 : _errorSource$error$ex2.length) !== null && _errorSource$error$ex !== void 0 ? _errorSource$error$ex : 0) > 0 ? errorSource === null || errorSource === void 0 ? void 0 : errorSource.error.exception : (event === null || event === void 0 ? void 0 : event.error.exception) && [event.error.exception],
        log: errorSource === null || errorSource === void 0 ? void 0 : errorSource.error.log
      }
    };
    return waterfallErrorEvent;
  });
}
async function getTraceDocsPaginated({
  apmEventClient,
  maxTraceItems,
  traceId,
  start,
  end,
  hits = [],
  searchAfter,
  logger
}) {
  var _last;
  const response = await (0, _get_trace_docs_per_page.getTraceDocsPerPage)({
    apmEventClient,
    maxTraceItems,
    traceId,
    start,
    end,
    searchAfter
  });
  const mergedHits = [...hits, ...response.hits];
  logger.debug(`Paginating traces: retrieved: ${response.hits.length}, (total: ${mergedHits.length} of ${response.total}), maxTraceItems: ${maxTraceItems}`);
  if (mergedHits.length >= maxTraceItems || mergedHits.length >= response.total || mergedHits.length === 0 || response.hits.length < MAX_ITEMS_PER_PAGE) {
    return {
      hits: mergedHits,
      total: response.total
    };
  }
  return getTraceDocsPaginated({
    apmEventClient,
    maxTraceItems,
    traceId,
    start,
    end,
    hits: mergedHits,
    searchAfter: (_last = (0, _lodash.last)(response.hits)) === null || _last === void 0 ? void 0 : _last.sort,
    logger
  });
}