"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WaterfallLegendType = void 0;
exports.buildTraceTree = buildTraceTree;
exports.convertTreeToList = void 0;
exports.generateLegendsAndAssignColorsToWaterfall = generateLegendsAndAssignColorsToWaterfall;
exports.getClockSkew = getClockSkew;
exports.getOrderedWaterfallItems = getOrderedWaterfallItems;
exports.getOrphanItemsIds = getOrphanItemsIds;
exports.getSpanItem = getSpanItem;
exports.getTransactionItem = getTransactionItem;
exports.getWaterfall = getWaterfall;
exports.reparentOrphanItems = reparentOrphanItems;
exports.updateTraceTreeNode = void 0;
var _eui = require("@elastic/eui");
var _common = require("@kbn/observability-plugin/common");
var _lodash = require("lodash");
var _agent_name = require("../../../../../../../../common/agent_name");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ROOT_ID = 'root';
let WaterfallLegendType = exports.WaterfallLegendType = /*#__PURE__*/function (WaterfallLegendType) {
  WaterfallLegendType["ServiceName"] = "serviceName";
  WaterfallLegendType["SpanType"] = "spanType";
  return WaterfallLegendType;
}({});
function getLegendValues(transactionOrSpan) {
  return {
    [WaterfallLegendType.ServiceName]: transactionOrSpan.service.name,
    [WaterfallLegendType.SpanType]: transactionOrSpan.processor.event === _common.ProcessorEvent.span ? transactionOrSpan.span.subtype || transactionOrSpan.span.type : ''
  };
}
function getTransactionItem(transaction, linkedChildrenCount = 0) {
  var _transaction$parent, _transaction$span$lin, _transaction$span, _transaction$span$lin2;
  return {
    docType: 'transaction',
    doc: transaction,
    id: transaction.transaction.id,
    parentId: (_transaction$parent = transaction.parent) === null || _transaction$parent === void 0 ? void 0 : _transaction$parent.id,
    duration: transaction.transaction.duration.us,
    offset: 0,
    skew: 0,
    legendValues: getLegendValues(transaction),
    color: '',
    spanLinksCount: {
      linkedParents: (_transaction$span$lin = (_transaction$span = transaction.span) === null || _transaction$span === void 0 ? void 0 : (_transaction$span$lin2 = _transaction$span.links) === null || _transaction$span$lin2 === void 0 ? void 0 : _transaction$span$lin2.length) !== null && _transaction$span$lin !== void 0 ? _transaction$span$lin : 0,
      linkedChildren: linkedChildrenCount
    }
  };
}
function getSpanItem(span, linkedChildrenCount = 0) {
  var _span$parent, _span$span$links$leng, _span$span$links;
  return {
    docType: 'span',
    doc: span,
    id: span.span.id,
    parentId: (_span$parent = span.parent) === null || _span$parent === void 0 ? void 0 : _span$parent.id,
    duration: span.span.duration.us,
    offset: 0,
    skew: 0,
    legendValues: getLegendValues(span),
    color: '',
    spanLinksCount: {
      linkedParents: (_span$span$links$leng = (_span$span$links = span.span.links) === null || _span$span$links === void 0 ? void 0 : _span$span$links.length) !== null && _span$span$links$leng !== void 0 ? _span$span$links$leng : 0,
      linkedChildren: linkedChildrenCount
    }
  };
}
function getErrorItem(error, items, entryWaterfallTransaction) {
  var _entryWaterfallTransa;
  const entryTimestamp = (_entryWaterfallTransa = entryWaterfallTransaction === null || entryWaterfallTransaction === void 0 ? void 0 : entryWaterfallTransaction.doc.timestamp.us) !== null && _entryWaterfallTransa !== void 0 ? _entryWaterfallTransa : 0;
  const parent = items.find(waterfallItem => {
    var _error$parent;
    return waterfallItem.id === ((_error$parent = error.parent) === null || _error$parent === void 0 ? void 0 : _error$parent.id);
  });
  const errorItem = {
    docType: 'error',
    doc: error,
    id: error.error.id,
    parent,
    parentId: parent === null || parent === void 0 ? void 0 : parent.id,
    offset: error.timestamp.us - entryTimestamp,
    skew: 0,
    color: ''
  };
  return {
    ...errorItem,
    skew: getClockSkew(errorItem, parent)
  };
}
function getClockSkew(item, parentItem) {
  if (!parentItem) {
    return 0;
  }
  switch (item.docType) {
    // don't calculate skew for spans and errors. Just use parent's skew
    case 'error':
    case 'span':
      return parentItem.skew;
    // transaction is the initial entry in a service. Calculate skew for this, and it will be propagated to all child spans
    case 'transaction':
      {
        const parentStart = parentItem.doc.timestamp.us + parentItem.skew;

        // determine if child starts before the parent
        const offsetStart = parentStart - item.doc.timestamp.us;
        if (offsetStart > 0) {
          const latency = Math.max(parentItem.duration - item.duration, 0) / 2;
          return offsetStart + latency;
        }

        // child transaction starts after parent thus no adjustment is needed
        return 0;
      }
  }
}
function getOrderedWaterfallItems(childrenByParentId, entryWaterfallTransaction) {
  if (!entryWaterfallTransaction) {
    return [];
  }
  const entryTimestamp = entryWaterfallTransaction.doc.timestamp.us;
  const visitedWaterfallItemSet = new Set();
  function getSortedChildren(item, parentItem) {
    if (visitedWaterfallItemSet.has(item)) {
      return [];
    }
    visitedWaterfallItemSet.add(item);
    const children = (0, _lodash.sortBy)(childrenByParentId[item.id] || [], 'doc.timestamp.us');
    item.parent = parentItem;
    // get offset from the beginning of trace
    item.offset = item.doc.timestamp.us - entryTimestamp;
    // move the item to the right if it starts before its parent
    item.skew = getClockSkew(item, parentItem);
    const deepChildren = (0, _lodash.flatten)(children.map(child => getSortedChildren(child, item)));
    return [item, ...deepChildren];
  }
  return getSortedChildren(entryWaterfallTransaction);
}
function getRootWaterfallTransaction(childrenByParentId) {
  const item = (0, _lodash.first)(childrenByParentId.root);
  if (item && item.docType === 'transaction') {
    return item;
  }
}
function generateLegendsAndAssignColorsToWaterfall(waterfallItems) {
  const onlyBaseSpanItems = waterfallItems.filter(item => item.docType === 'span' || item.docType === 'transaction');
  const legends = [WaterfallLegendType.ServiceName, WaterfallLegendType.SpanType].flatMap(legendType => {
    const allLegendValues = (0, _lodash.uniq)(onlyBaseSpanItems.map(item => item.legendValues[legendType]));
    const palette = (0, _eui.euiPaletteColorBlind)({
      rotations: Math.ceil(allLegendValues.length / 10)
    });
    return allLegendValues.map((value, index) => ({
      type: legendType,
      value,
      color: palette[index]
    }));
  });
  const serviceLegends = legends.filter(({
    type
  }) => type === WaterfallLegendType.ServiceName);
  // only color by span type if there are only events for one service
  const colorBy = serviceLegends.length > 1 ? WaterfallLegendType.ServiceName : WaterfallLegendType.SpanType;
  const serviceColorsMap = serviceLegends.reduce((colorMap, legend) => {
    colorMap[legend.value] = legend.color;
    return colorMap;
  }, {});
  const legendsByValue = legends.reduce((acc, curr) => {
    if (curr.type === colorBy) {
      acc[curr.value] = curr;
    }
    return acc;
  }, {});

  // mutate items rather than rebuilding both items and childrenByParentId
  waterfallItems.forEach(item => {
    let color = '';
    if ('legendValues' in item) {
      color = legendsByValue[item.legendValues[colorBy]].color;
    }
    if (!color) {
      // fall back to service color if there's no span.type, e.g. for transactions
      color = serviceColorsMap[item.doc.service.name];
    }
    item.color = color;
  });
  return {
    legends,
    colorBy
  };
}
const getWaterfallDuration = waterfallItems => Math.max(...waterfallItems.map(item => item.offset + item.skew + ('duration' in item ? item.duration : 0)), 0);
const getWaterfallItems = (items, spanLinksCountById) => items.map(item => {
  const docType = item.processor.event;
  switch (docType) {
    case 'span':
      {
        const span = item;
        return getSpanItem(span, spanLinksCountById[span.span.id]);
      }
    case 'transaction':
      const transaction = item;
      return getTransactionItem(transaction, spanLinksCountById[transaction.transaction.id]);
  }
});
function reparentSpans(waterfallItems) {
  // find children that needs to be re-parented and map them to their correct parent id
  const childIdToParentIdMapping = Object.fromEntries((0, _lodash.flatten)(waterfallItems.map(waterfallItem => {
    if (waterfallItem.docType === 'span') {
      var _waterfallItem$doc$ch, _waterfallItem$doc$ch2;
      const childIds = (_waterfallItem$doc$ch = (_waterfallItem$doc$ch2 = waterfallItem.doc.child) === null || _waterfallItem$doc$ch2 === void 0 ? void 0 : _waterfallItem$doc$ch2.id) !== null && _waterfallItem$doc$ch !== void 0 ? _waterfallItem$doc$ch : [];
      return childIds.map(id => [id, waterfallItem.id]);
    }
    return [];
  })));

  // update parent id for children that needs it or return unchanged
  return waterfallItems.map(waterfallItem => {
    const newParentId = childIdToParentIdMapping[waterfallItem.id];
    if (newParentId) {
      return {
        ...waterfallItem,
        parentId: newParentId
      };
    }
    return waterfallItem;
  });
}
const getChildrenGroupedByParentId = waterfallItems => {
  const childrenGroups = (0, _lodash.groupBy)(waterfallItems, item => {
    var _item$parentId;
    return (_item$parentId = item.parentId) !== null && _item$parentId !== void 0 ? _item$parentId : ROOT_ID;
  });
  const childrenGroupedByParentId = Object.entries(childrenGroups).reduce((acc, [parentId, items]) => {
    // we shouldn't include the parent item in the children list
    const filteredItems = items.filter(item => item.id !== parentId);
    return {
      ...acc,
      [parentId]: filteredItems
    };
  }, {});
  return childrenGroupedByParentId;
};
const getEntryWaterfallTransaction = (entryTransactionId, waterfallItems) => waterfallItems.find(item => item.docType === 'transaction' && item.id === entryTransactionId);
function isInEntryTransaction(parentIdLookup, entryTransactionId, currentId) {
  if (currentId === entryTransactionId) {
    return true;
  }
  const parentId = parentIdLookup.get(currentId);
  if (parentId) {
    return isInEntryTransaction(parentIdLookup, entryTransactionId, parentId);
  }
  return false;
}
function getWaterfallErrors(errorDocs, items, entryWaterfallTransaction) {
  const errorItems = errorDocs.map(errorDoc => getErrorItem(errorDoc, items, entryWaterfallTransaction));
  if (!entryWaterfallTransaction) {
    return errorItems;
  }
  const parentIdLookup = [...items, ...errorItems].reduce((map, {
    id,
    parentId
  }) => {
    map.set(id, parentId !== null && parentId !== void 0 ? parentId : ROOT_ID);
    return map;
  }, new Map());
  return errorItems.filter(errorItem => isInEntryTransaction(parentIdLookup, entryWaterfallTransaction === null || entryWaterfallTransaction === void 0 ? void 0 : entryWaterfallTransaction.id, errorItem.id));
}

// map parent.id to the number of errors
/*
  { 'parentId': 2 }
  */
function getErrorCountByParentId(errorDocs) {
  return errorDocs.reduce((acc, doc) => {
    var _doc$parent, _acc$parentId;
    const parentId = (_doc$parent = doc.parent) === null || _doc$parent === void 0 ? void 0 : _doc$parent.id;
    if (!parentId) {
      return acc;
    }
    acc[parentId] = ((_acc$parentId = acc[parentId]) !== null && _acc$parentId !== void 0 ? _acc$parentId : 0) + 1;
    return acc;
  }, {});
}
function getOrphanItemsIds(waterfall, entryWaterfallTransactionId) {
  const waterfallItemsIds = new Set(waterfall.map(item => item.id));
  return waterfall.filter(item =>
  // the root transaction should never be orphan
  entryWaterfallTransactionId !== item.id && item.parentId && !waterfallItemsIds.has(item.parentId)).map(item => item.id);
}
function reparentOrphanItems(orphanItemsIds, waterfallItems, entryWaterfallTransaction) {
  const orphanIdsMap = new Set(orphanItemsIds);
  return waterfallItems.reduce((acc, item) => {
    if (orphanIdsMap.has(item.id)) {
      // we need to filter out the orphan item if it's longer or if it has started before the entry transaction
      // as this means it's a parent of the entry transaction
      const isLongerThanEntryTransaction = entryWaterfallTransaction && item.duration > (entryWaterfallTransaction === null || entryWaterfallTransaction === void 0 ? void 0 : entryWaterfallTransaction.duration);
      const hasStartedBeforeEntryTransaction = entryWaterfallTransaction && item.doc.timestamp.us < entryWaterfallTransaction.doc.timestamp.us;
      if (isLongerThanEntryTransaction || hasStartedBeforeEntryTransaction) {
        return acc;
      }
      acc.push({
        ...item,
        parentId: entryWaterfallTransaction === null || entryWaterfallTransaction === void 0 ? void 0 : entryWaterfallTransaction.id,
        isOrphan: true
      });
    } else {
      acc.push(item);
    }
    return acc;
  }, []);
}
function getWaterfall(apiResponse) {
  const {
    traceItems,
    entryTransaction
  } = apiResponse;
  if ((0, _lodash.isEmpty)(traceItems.traceDocs) || !entryTransaction) {
    return {
      duration: 0,
      items: [],
      legends: [],
      colorBy: WaterfallLegendType.ServiceName,
      errorItems: [],
      childrenByParentId: {},
      getErrorCount: () => 0,
      exceedsMax: false,
      totalErrorsCount: 0,
      traceDocsTotal: 0,
      maxTraceItems: 0,
      orphanTraceItemsCount: 0
    };
  }
  const errorCountByParentId = getErrorCountByParentId(traceItems.errorDocs);
  const waterfallItems = getWaterfallItems(traceItems.traceDocs, traceItems.spanLinksCountById);
  const entryWaterfallTransaction = getEntryWaterfallTransaction(entryTransaction.transaction.id, waterfallItems);
  const orphanItemsIds = getOrphanItemsIds(waterfallItems, entryWaterfallTransaction === null || entryWaterfallTransaction === void 0 ? void 0 : entryWaterfallTransaction.id);
  const childrenByParentId = getChildrenGroupedByParentId(reparentOrphanItems(orphanItemsIds, reparentSpans(waterfallItems), entryWaterfallTransaction));
  const items = getOrderedWaterfallItems(childrenByParentId, entryWaterfallTransaction);
  const errorItems = getWaterfallErrors(traceItems.errorDocs, items, entryWaterfallTransaction);
  const rootWaterfallTransaction = getRootWaterfallTransaction(childrenByParentId);
  const duration = getWaterfallDuration(items);
  const {
    legends,
    colorBy
  } = generateLegendsAndAssignColorsToWaterfall(items);
  return {
    entryWaterfallTransaction,
    rootWaterfallTransaction,
    entryTransaction,
    duration,
    items,
    legends,
    colorBy,
    errorItems,
    childrenByParentId: getChildrenGroupedByParentId(items),
    getErrorCount: parentId => {
      var _errorCountByParentId;
      return (_errorCountByParentId = errorCountByParentId[parentId]) !== null && _errorCountByParentId !== void 0 ? _errorCountByParentId : 0;
    },
    exceedsMax: traceItems.exceedsMax,
    totalErrorsCount: traceItems.errorDocs.length,
    traceDocsTotal: traceItems.traceDocsTotal,
    maxTraceItems: traceItems.maxTraceItems,
    orphanTraceItemsCount: orphanItemsIds.length
  };
}
function getChildren({
  path,
  waterfall,
  waterfallItemId,
  rootId
}) {
  var _waterfall$childrenBy;
  const children = (_waterfall$childrenBy = waterfall.childrenByParentId[waterfallItemId]) !== null && _waterfall$childrenBy !== void 0 ? _waterfall$childrenBy : [];
  return path.showCriticalPath ? children.filter(child => {
    var _path$criticalPathSeg;
    return (_path$criticalPathSeg = path.criticalPathSegmentsById[child.id]) === null || _path$criticalPathSeg === void 0 ? void 0 : _path$criticalPathSeg.length;
  }) : children;
}
function buildTree({
  root,
  waterfall,
  maxLevelOpen,
  path
}) {
  const tree = {
    ...root
  };
  const queue = [tree];
  for (let queueIndex = 0; queueIndex < queue.length; queueIndex++) {
    const node = queue[queueIndex];
    const children = getChildren({
      path,
      waterfall,
      waterfallItemId: node.item.id,
      rootId: root.item.id
    });

    // Set childrenToLoad for all nodes enqueued.
    // this allows lazy loading of child nodes
    node.childrenToLoad = children.length;
    if (maxLevelOpen > node.level) {
      children.forEach((child, index) => {
        var _node$item$doc$span, _node$item$doc$span$d, _node$item$doc$span$d2;
        const level = node.level + 1;
        const currentNode = {
          id: `${level}-${child.id}-${index}`,
          item: child,
          children: [],
          level,
          expanded: level < maxLevelOpen,
          childrenToLoad: 0,
          hasInitializedChildren: false
        };

        // It is missing a destination when a child (currentNode) is a transaction
        // and its parent (node) is a span without destination for Otel agents.

        if (currentNode.item.docType === 'transaction' && node.item.docType === 'span' && !((_node$item$doc$span = node.item.doc.span) !== null && _node$item$doc$span !== void 0 && (_node$item$doc$span$d = _node$item$doc$span.destination) !== null && _node$item$doc$span$d !== void 0 && (_node$item$doc$span$d2 = _node$item$doc$span$d.service) !== null && _node$item$doc$span$d2 !== void 0 && _node$item$doc$span$d2.resource) && (0, _agent_name.isOpenTelemetryAgentName)(node.item.doc.agent.name)) {
          node.item.missingDestination = true;
        }
        node.children.push(currentNode);
        queue.push(currentNode);
      });
      node.hasInitializedChildren = true;
    }
  }
  return tree;
}
function buildTraceTree({
  waterfall,
  maxLevelOpen,
  isOpen,
  path
}) {
  const entry = waterfall.entryWaterfallTransaction;
  if (!entry) {
    return null;
  }
  const root = {
    id: entry.id,
    item: entry,
    children: [],
    level: 0,
    expanded: isOpen,
    childrenToLoad: 0,
    hasInitializedChildren: false
  };
  return buildTree({
    root,
    maxLevelOpen,
    waterfall,
    path
  });
}
const convertTreeToList = root => {
  if (!root) {
    return [];
  }
  const result = [];
  const stack = [root];
  while (stack.length > 0) {
    const node = stack.pop();
    const {
      children,
      ...nodeWithoutChildren
    } = node;
    result.push(nodeWithoutChildren);
    if (node.expanded) {
      for (let i = node.children.length - 1; i >= 0; i--) {
        stack.push(node.children[i]);
      }
    }
  }
  return result;
};
exports.convertTreeToList = convertTreeToList;
const updateTraceTreeNode = ({
  root,
  updatedNode,
  waterfall,
  path
}) => {
  if (!root) {
    return;
  }
  const tree = {
    ...root
  };
  const stack = [{
    parent: null,
    index: 0,
    node: root
  }];
  while (stack.length > 0) {
    const {
      parent,
      index,
      node
    } = stack.pop();
    if (node.id === updatedNode.id) {
      Object.assign(node, updatedNode);
      if (updatedNode.expanded && !updatedNode.hasInitializedChildren) {
        Object.assign(node, buildTree({
          root: node,
          waterfall,
          maxLevelOpen: node.level + 1,
          // Only one level above the current node will be loaded
          path
        }));
      }
      if (parent) {
        parent.children[index] = node;
      } else {
        Object.assign(tree, node);
      }
      return tree;
    }
    for (let i = node.children.length - 1; i >= 0; i--) {
      stack.push({
        parent: node,
        index: i,
        node: node.children[i]
      });
    }
  }
  return tree;
};
exports.updateTraceTreeNode = updateTraceTreeNode;