"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_PUSH_ATTEMPTS = exports.ConnectorUsageReportingTask = exports.CONNECTOR_USAGE_TYPE = exports.CONNECTOR_USAGE_REPORTING_TASK_TYPE = exports.CONNECTOR_USAGE_REPORTING_TASK_TIMEOUT = exports.CONNECTOR_USAGE_REPORTING_TASK_SCHEDULE = exports.CONNECTOR_USAGE_REPORTING_TASK_ID = exports.CONNECTOR_USAGE_REPORTING_SOURCE_ID = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _fs = _interopRequireDefault(require("fs"));
var _axios = _interopRequireDefault(require("axios"));
var _https = _interopRequireDefault(require("https"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CONNECTOR_USAGE_REPORTING_TASK_SCHEDULE = exports.CONNECTOR_USAGE_REPORTING_TASK_SCHEDULE = {
  interval: '1h'
};
const CONNECTOR_USAGE_REPORTING_TASK_ID = exports.CONNECTOR_USAGE_REPORTING_TASK_ID = 'connector_usage_reporting';
const CONNECTOR_USAGE_REPORTING_TASK_TYPE = exports.CONNECTOR_USAGE_REPORTING_TASK_TYPE = `actions:${CONNECTOR_USAGE_REPORTING_TASK_ID}`;
const CONNECTOR_USAGE_REPORTING_TASK_TIMEOUT = exports.CONNECTOR_USAGE_REPORTING_TASK_TIMEOUT = 30000;
const CONNECTOR_USAGE_TYPE = exports.CONNECTOR_USAGE_TYPE = `connector_request_body_bytes`;
const CONNECTOR_USAGE_REPORTING_SOURCE_ID = exports.CONNECTOR_USAGE_REPORTING_SOURCE_ID = `task-connector-usage-report`;
const MAX_PUSH_ATTEMPTS = exports.MAX_PUSH_ATTEMPTS = 5;
class ConnectorUsageReportingTask {
  constructor({
    logger,
    eventLogIndex,
    core: _core,
    taskManager: _taskManager,
    projectId: _projectId,
    config
  }) {
    var _config$enabled, _config$ca;
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "eventLogIndex", void 0);
    (0, _defineProperty2.default)(this, "projectId", void 0);
    (0, _defineProperty2.default)(this, "caCertificate", void 0);
    (0, _defineProperty2.default)(this, "usageApiUrl", void 0);
    (0, _defineProperty2.default)(this, "enabled", void 0);
    (0, _defineProperty2.default)(this, "start", async taskManager => {
      if (!this.projectId) {
        return;
      }
      if (!taskManager) {
        this.logger.error(`Missing required task manager service during start of ${CONNECTOR_USAGE_REPORTING_TASK_TYPE}`);
        return;
      }
      try {
        await taskManager.ensureScheduled({
          id: CONNECTOR_USAGE_REPORTING_TASK_ID,
          taskType: CONNECTOR_USAGE_REPORTING_TASK_TYPE,
          schedule: {
            ...CONNECTOR_USAGE_REPORTING_TASK_SCHEDULE
          },
          state: {},
          params: {}
        });
      } catch (e) {
        this.logger.error(`Error scheduling task ${CONNECTOR_USAGE_REPORTING_TASK_TYPE}, received ${e.message}`);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core) => {
      const {
        state
      } = taskInstance;
      if (!this.enabled) {
        this.logger.warn(`Usage API is disabled, ${CONNECTOR_USAGE_REPORTING_TASK_TYPE} will be skipped`);
        return {
          state
        };
      }
      if (!this.projectId) {
        this.logger.warn(`Missing required project id while running ${CONNECTOR_USAGE_REPORTING_TASK_TYPE}, reporting task will be deleted`);
        return {
          state,
          shouldDeleteTask: true
        };
      }
      if (!this.caCertificate) {
        this.logger.error(`Missing required CA Certificate while running ${CONNECTOR_USAGE_REPORTING_TASK_TYPE}`);
        return {
          state
        };
      }
      const [{
        elasticsearch
      }] = await core.getStartServices();
      const esClient = elasticsearch.client.asInternalUser;
      const now = new Date();
      const oneDayAgo = new Date(new Date().getTime() - 24 * 60 * 60 * 1000);
      const lastReportedUsageDate = state.lastReportedUsageDate ? new Date(state.lastReportedUsageDate) : oneDayAgo;
      let attempts = state.attempts || 0;
      const fromDate = lastReportedUsageDate;
      const toDate = now;
      let totalUsage = 0;
      try {
        totalUsage = await this.getTotalUsage({
          esClient,
          fromDate,
          toDate
        });
      } catch (e) {
        this.logger.error(`Usage data could not be fetched. It will be retried. Error:${e.message}`);
        return {
          state: {
            lastReportedUsageDate,
            attempts
          },
          runAt: now
        };
      }
      const record = this.createUsageRecord({
        totalUsage,
        fromDate,
        toDate,
        projectId: this.projectId
      });
      this.logger.debug(`Record: ${JSON.stringify(record)}`);
      try {
        attempts = attempts + 1;
        await this.pushUsageRecord(record);
        this.logger.info(`Connector usage record has been successfully reported, ${record.creation_timestamp}, usage: ${record.usage.quantity}, period:${record.usage.period_seconds}`);
      } catch (e) {
        if (attempts < MAX_PUSH_ATTEMPTS) {
          this.logger.error(`Usage data could not be pushed to usage-api. It will be retried (${attempts}). Error:${e.message}`);
          return {
            state: {
              lastReportedUsageDate,
              attempts
            },
            runAt: this.getDelayedRetryDate({
              attempts,
              now
            })
          };
        }
        this.logger.error(`Usage data could not be pushed to usage-api. Stopped retrying after ${attempts} attempts. Error:${e.message}`);
        return {
          state: {
            lastReportedUsageDate,
            attempts: 0
          }
        };
      }
      return {
        state: {
          lastReportedUsageDate: toDate,
          attempts: 0
        }
      };
    });
    (0, _defineProperty2.default)(this, "getTotalUsage", async ({
      esClient,
      fromDate,
      toDate
    }) => {
      var _value, _usageResult$aggregat, _usageResult$aggregat2;
      const usageResult = await esClient.search({
        index: this.eventLogIndex,
        sort: '@timestamp',
        size: 0,
        query: {
          bool: {
            filter: {
              bool: {
                must: [{
                  term: {
                    'event.action': 'execute'
                  }
                }, {
                  term: {
                    'event.provider': 'actions'
                  }
                }, {
                  exists: {
                    field: 'kibana.action.execution.usage.request_body_bytes'
                  }
                }, {
                  range: {
                    '@timestamp': {
                      gt: fromDate,
                      lte: toDate
                    }
                  }
                }]
              }
            }
          }
        },
        aggs: {
          total_usage: {
            sum: {
              field: 'kibana.action.execution.usage.request_body_bytes'
            }
          }
        }
      });
      return (_value = (_usageResult$aggregat = usageResult.aggregations) === null || _usageResult$aggregat === void 0 ? void 0 : (_usageResult$aggregat2 = _usageResult$aggregat.total_usage) === null || _usageResult$aggregat2 === void 0 ? void 0 : _usageResult$aggregat2.value) !== null && _value !== void 0 ? _value : 0;
    });
    (0, _defineProperty2.default)(this, "createUsageRecord", ({
      totalUsage,
      fromDate,
      toDate,
      projectId
    }) => {
      const period = Math.round((toDate.getTime() - fromDate.getTime()) / 1000);
      const toStr = toDate.toISOString();
      const timestamp = new Date(toStr);
      timestamp.setMinutes(0);
      timestamp.setSeconds(0);
      timestamp.setMilliseconds(0);
      return {
        id: `connector-request-body-bytes-${projectId}-${timestamp.toISOString()}`,
        usage_timestamp: toStr,
        creation_timestamp: toStr,
        usage: {
          type: CONNECTOR_USAGE_TYPE,
          period_seconds: period,
          quantity: totalUsage
        },
        source: {
          id: CONNECTOR_USAGE_REPORTING_SOURCE_ID,
          instance_group_id: projectId
        }
      };
    });
    (0, _defineProperty2.default)(this, "pushUsageRecord", async record => {
      return _axios.default.post(this.usageApiUrl, [record], {
        headers: {
          'Content-Type': 'application/json'
        },
        timeout: CONNECTOR_USAGE_REPORTING_TASK_TIMEOUT,
        httpsAgent: new _https.default.Agent({
          ca: this.caCertificate
        })
      });
    });
    (0, _defineProperty2.default)(this, "getDelayedRetryDate", ({
      attempts,
      now
    }) => {
      const baseDelay = 60 * 1000;
      const delayByAttempts = baseDelay * attempts;
      const delayedTime = now.getTime() + delayByAttempts;
      return new Date(delayedTime);
    });
    this.logger = logger;
    this.projectId = _projectId;
    this.eventLogIndex = eventLogIndex;
    this.usageApiUrl = config.url;
    this.enabled = (_config$enabled = config.enabled) !== null && _config$enabled !== void 0 ? _config$enabled : true;
    const caCertificatePath = (_config$ca = config.ca) === null || _config$ca === void 0 ? void 0 : _config$ca.path;
    if (caCertificatePath && caCertificatePath.length > 0) {
      try {
        this.caCertificate = _fs.default.readFileSync(caCertificatePath, 'utf8');
      } catch (e) {
        this.caCertificate = undefined;
        this.logger.error(`CA Certificate for the project "${_projectId}" couldn't be loaded, Error: ${e.message}`);
      }
    }
    _taskManager.registerTaskDefinitions({
      [CONNECTOR_USAGE_REPORTING_TASK_TYPE]: {
        title: 'Connector usage reporting task',
        timeout: '1m',
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => this.runTask(taskInstance, _core),
            cancel: async () => {}
          };
        }
      }
    });
  }
}
exports.ConnectorUsageReportingTask = ConnectorUsageReportingTask;