"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createLensEmbeddableFactory = void 0;
var _react = _interopRequireDefault(require("react"));
var _presentationPublishing = require("@kbn/presentation-publishing");
var _constants = require("../../common/constants");
var _data_loader = require("./data_loader");
var _helper = require("./helper");
var _initialize_edit = require("./initializers/initialize_edit");
var _initialize_inspector = require("./initializers/initialize_inspector");
var _initialize_dashboard_services = require("./initializers/initialize_dashboard_services");
var _initialize_internal_api = require("./initializers/initialize_internal_api");
var _initialize_search_context = require("./initializers/initialize_search_context");
var _initialize_actions = require("./initializers/initialize_actions");
var _initialize_integrations = require("./initializers/initialize_integrations");
var _initialize_state_management = require("./initializers/initialize_state_management");
var _lens_embeddable_component = require("./renderer/lens_embeddable_component");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createLensEmbeddableFactory = services => {
  return {
    type: _constants.DOC_TYPE,
    /**
     * This is called before the build and will make sure that the
     * final state will contain the attributes object
     */
    deserializeState: async ({
      rawState,
      references
    }) => (0, _helper.deserializeState)(services.attributeService, rawState, references),
    /**
     * This is called after the deserialize, so some assumptions can be made about its arguments:
     * @param state     the Lens "runtime" state, which means that 'attributes' is always present.
     *                  The difference for a by-value and a by-ref can be determined by the presence of 'savedObjectId' in the state
     * @param buildApi  a utility function to build the Lens API together to instrument the embeddable container on how to detect
     *                  significative changes in the state (i.e. worth a save or not)
     * @param uuid      a unique identifier for the embeddable panel
     * @param parentApi a set of props passed down from the embeddable container. Note: no assumptions can be made about its content
     *                  so the usage of type-guards is recommended before extracting data from it.
     *                  Due to the new embeddable being rendered by a <ReactEmbeddableRenderer /> wrapper, this is the only way
     *                  to pass data/props from a container.
     *                  Typical use cases is the forwarding of the unifiedSearch context to the embeddable, or the passing props
     *                  from the Lens component container to the Lens embeddable.
     * @returns an object with the Lens API and the React component to render in the Embeddable
     */
    buildEmbeddable: async (initialState, buildApi, uuid, parentApi) => {
      const titleManager = (0, _presentationPublishing.initializeTitleManager)(initialState);

      /**
       * Observables and functions declared here are used internally to store mutating state values
       * This is an internal API not exposed outside of the embeddable.
       */
      const internalApi = (0, _initialize_internal_api.initializeInternalApi)(initialState, parentApi, titleManager, services);

      /**
       * Initialize various configurations required to build all the required
       * parts for the Lens embeddable.
       * Each initialize call returns an object with the following properties:
       * - api: a set of methods or observables (also non-serializable) who can be picked up within the component
       * - serialize: a serializable subset of the Lens runtime state
       * - comparators: a set of comparators to help Dashboard determine if the state has changed since its saved state
       * - cleanup: a function to clean up any resources when the component is unmounted
       *
       * Mind: the getState argument is ok to pass as long as it is lazy evaluated (i.e. called within a function).
       * If there's something that should be immediately computed use the "initialState" deserialized variable.
       */
      const stateConfig = (0, _initialize_state_management.initializeStateManagement)(initialState, internalApi);
      const dashboardConfig = (0, _initialize_dashboard_services.initializeDashboardServices)(initialState, getState, internalApi, stateConfig, parentApi, titleManager, services);
      const inspectorConfig = (0, _initialize_inspector.initializeInspector)(services);
      const searchContextConfig = (0, _initialize_search_context.initializeSearchContext)(initialState, internalApi, parentApi, services);
      const editConfig = (0, _initialize_edit.initializeEditApi)(uuid, initialState, getState, internalApi, stateConfig.api, inspectorConfig.api, searchContextConfig.api, _helper.isTextBasedLanguage, services, parentApi);
      const integrationsConfig = (0, _initialize_integrations.initializeIntegrations)(getState, services);
      const actionsConfig = (0, _initialize_actions.initializeActionApi)(uuid, initialState, getState, parentApi, searchContextConfig.api, titleManager.api.title$, internalApi, services);

      /**
       * This is useful to have always the latest version of the state
       * at hand when calling callbacks or performing actions
       */
      function getState() {
        return {
          ...actionsConfig.serialize(),
          ...editConfig.serialize(),
          ...inspectorConfig.serialize(),
          ...dashboardConfig.serialize(),
          ...searchContextConfig.serialize(),
          ...integrationsConfig.serialize(),
          ...stateConfig.serialize()
        };
      }

      /**
       * Lens API is the object that can be passed to the final component/renderer and
       * provide access to the services for and by the outside world
       */
      const api = buildApi(
      // Note: the order matters here, so make sure to have the
      // dashboardConfig who owns the savedObjectId after the
      // stateConfig one who owns the inline editing
      {
        ...editConfig.api,
        ...inspectorConfig.api,
        ...searchContextConfig.api,
        ...actionsConfig.api,
        ...integrationsConfig.api,
        ...stateConfig.api,
        ...dashboardConfig.api
      }, {
        ...stateConfig.comparators,
        ...editConfig.comparators,
        ...inspectorConfig.comparators,
        ...searchContextConfig.comparators,
        ...actionsConfig.comparators,
        ...integrationsConfig.comparators,
        ...dashboardConfig.comparators
      });

      // Compute the expression using the provided parameters
      // Inside a subscription will be updated based on each unifiedSearch change
      // and as side effect update few observables as  expressionParams$, expressionAbortController$ and renderCount$ with the new values upon updates
      const expressionConfig = (0, _data_loader.loadEmbeddableData)(uuid, getState, api, parentApi, internalApi, services);
      const onUnmount = () => {
        editConfig.cleanup();
        inspectorConfig.cleanup();
        searchContextConfig.cleanup();
        expressionConfig.cleanup();
        actionsConfig.cleanup();
        integrationsConfig.cleanup();
        dashboardConfig.cleanup();
      };
      return {
        api,
        Component: () => /*#__PURE__*/_react.default.createElement(_lens_embeddable_component.LensEmbeddableComponent, {
          api: api,
          internalApi: internalApi,
          onUnmount: onUnmount
        })
      };
    }
  };
};
exports.createLensEmbeddableFactory = createLensEmbeddableFactory;