"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeDashboardServices = initializeDashboardServices;
var _lodash = require("lodash");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _presentationContainers = require("@kbn/presentation-containers");
var _helper = require("../helper");
var _type_guards = require("../type_guards");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Convenience type for the serialized props of this initializer

/**
 * Everything about panel and library services
 */
function initializeDashboardServices(initialState, getLatestState, internalApi, stateConfig, parentApi, titleManager, {
  attributeService,
  uiActions
}) {
  // For some legacy reason the title and description default value is picked differently
  // ( based on existing FTR tests ).
  const [defaultTitle$] = (0, _helper.buildObservableVariable)(initialState.title || internalApi.attributes$.getValue().title);
  const [defaultDescription$] = (0, _helper.buildObservableVariable)(initialState.savedObjectId ? internalApi.attributes$.getValue().description || initialState.description : initialState.description);
  // The observable references here are the same to the internalApi,
  // the buildObservableVariable re-uses the same observable when detected but it builds the right comparator
  const [overrides$, overridesComparator] = (0, _helper.buildObservableVariable)(internalApi.overrides$);
  const [disableTriggers$, disabledTriggersComparator] = (0, _helper.buildObservableVariable)(internalApi.disableTriggers$);
  return {
    api: {
      parentApi: (0, _presentationContainers.apiIsPresentationContainer)(parentApi) ? parentApi : undefined,
      defaultTitle$,
      defaultDescription$,
      ...titleManager.api,
      updateOverrides: internalApi.updateOverrides,
      getTriggerCompatibleActions: uiActions.getTriggerCompatibleActions,
      // The functions below fulfill the HasLibraryTransforms interface
      saveToLibrary: async title => {
        const {
          attributes
        } = getLatestState();
        const savedObjectId = await attributeService.saveToLibrary({
          ...attributes,
          title
        }, attributes.references);
        // keep in sync the state
        stateConfig.api.updateSavedObjectId(savedObjectId);
        return savedObjectId;
      },
      checkForDuplicateTitle: async (newTitle, isTitleDuplicateConfirmed, onTitleDuplicate) => {
        await attributeService.checkForDuplicateTitle({
          newTitle,
          isTitleDuplicateConfirmed,
          onTitleDuplicate,
          newCopyOnSave: false,
          newDescription: '',
          displayName: '',
          lastSavedTitle: '',
          copyOnSave: false
        });
      },
      canLinkToLibrary: async () => !getLatestState().savedObjectId && !(0, _helper.isTextBasedLanguage)(getLatestState()),
      canUnlinkFromLibrary: async () => Boolean(getLatestState().savedObjectId),
      getSerializedStateByReference: newId => {
        const currentState = getLatestState();
        currentState.savedObjectId = newId;
        return attributeService.extractReferences(currentState);
      },
      getSerializedStateByValue: () => {
        const {
          savedObjectId,
          ...byValueRuntimeState
        } = getLatestState();
        return attributeService.extractReferences(byValueRuntimeState);
      }
    },
    serialize: () => {
      const {
        style,
        className
      } = (0, _type_guards.apiHasLensComponentProps)(parentApi) ? parentApi : {};
      const settings = (0, _presentationContainers.apiPublishesSettings)(parentApi) ? {
        syncColors: parentApi.settings.syncColors$.getValue(),
        syncCursor: parentApi.settings.syncCursor$.getValue(),
        syncTooltips: parentApi.settings.syncTooltips$.getValue()
      } : {};
      return {
        ...titleManager.serialize(),
        style,
        className,
        ...settings,
        palette: initialState.palette,
        overrides: overrides$.getValue(),
        disableTriggers: disableTriggers$.getValue()
      };
    },
    comparators: {
      ...titleManager.comparators,
      id: (0, _presentationPublishing.getUnchangingComparator)(),
      palette: (0, _presentationPublishing.getUnchangingComparator)(),
      renderMode: (0, _presentationPublishing.getUnchangingComparator)(),
      syncColors: (0, _presentationPublishing.getUnchangingComparator)(),
      syncCursor: (0, _presentationPublishing.getUnchangingComparator)(),
      syncTooltips: (0, _presentationPublishing.getUnchangingComparator)(),
      executionContext: (0, _presentationPublishing.getUnchangingComparator)(),
      noPadding: (0, _presentationPublishing.getUnchangingComparator)(),
      viewMode: (0, _presentationPublishing.getUnchangingComparator)(),
      style: (0, _presentationPublishing.getUnchangingComparator)(),
      className: (0, _presentationPublishing.getUnchangingComparator)(),
      overrides: overridesComparator,
      disableTriggers: disabledTriggersComparator,
      isNewPanel: (0, _presentationPublishing.getUnchangingComparator)(),
      parentApi: (0, _presentationPublishing.getUnchangingComparator)()
    },
    cleanup: _lodash.noop
  };
}