"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bedrockClaudeAdapter = void 0;
var _rxjs = require("rxjs");
var _stream = require("stream");
var _inferenceCommon = require("@kbn/inference-common");
var _serde_utils = require("./serde_utils");
var _serde_eventstream_into_observable = require("./serde_eventstream_into_observable");
var _process_completion_chunks = require("./process_completion_chunks");
var _prompts = require("./prompts");
var _convert_tools = require("./convert_tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bedrockClaudeAdapter = exports.bedrockClaudeAdapter = {
  chatComplete: ({
    executor,
    system,
    messages,
    toolChoice,
    tools,
    temperature = 0,
    modelName,
    abortSignal,
    metadata
  }) => {
    const noToolUsage = toolChoice === _inferenceCommon.ToolChoiceType.none;
    const bedRockTools = (0, _convert_tools.toolsToBedrock)(tools, messages);
    const subActionParams = {
      system: noToolUsage ? (0, _prompts.addNoToolUsageDirective)(system) : system,
      messages: messagesToBedrock(messages),
      tools: bedRockTools !== null && bedRockTools !== void 0 && bedRockTools.length ? bedRockTools : undefined,
      toolChoice: (0, _convert_tools.toolChoiceToBedrock)(toolChoice),
      temperature,
      model: modelName,
      stopSequences: ['\n\nHuman:'],
      signal: abortSignal,
      ...(metadata !== null && metadata !== void 0 && metadata.connectorTelemetry ? {
        telemetryMetadata: metadata.connectorTelemetry
      } : {})
    };
    return (0, _rxjs.from)(executor.invoke({
      subAction: 'invokeStream',
      subActionParams
    })).pipe((0, _rxjs.switchMap)(response => {
      if (response.status === 'error') {
        return (0, _rxjs.throwError)(() => (0, _inferenceCommon.createInferenceInternalError)(`Error calling connector: ${response.serviceMessage}`, {
          rootError: response.serviceMessage
        }));
      }
      if ((0, _stream.isReadable)(response.data)) {
        return (0, _serde_eventstream_into_observable.serdeEventstreamIntoObservable)(response.data);
      }
      return (0, _rxjs.throwError)(() => (0, _inferenceCommon.createInferenceInternalError)('Unexpected error', response.data));
    }), (0, _rxjs.tap)(eventData => {
      if ('modelStreamErrorException' in eventData) {
        throw (0, _inferenceCommon.createInferenceInternalError)(eventData.modelStreamErrorException.originalMessage);
      }
    }), (0, _rxjs.filter)(value => {
      var _value$chunk, _value$chunk$headers, _value$chunk$headers$;
      return 'chunk' in value && ((_value$chunk = value.chunk) === null || _value$chunk === void 0 ? void 0 : (_value$chunk$headers = _value$chunk.headers) === null || _value$chunk$headers === void 0 ? void 0 : (_value$chunk$headers$ = _value$chunk$headers[':event-type']) === null || _value$chunk$headers$ === void 0 ? void 0 : _value$chunk$headers$.value) === 'chunk';
    }), (0, _rxjs.map)(message => {
      return (0, _serde_utils.parseSerdeChunkMessage)(message.chunk);
    }), (0, _process_completion_chunks.processCompletionChunks)());
  }
};
const messagesToBedrock = messages => {
  return messages.map(message => {
    switch (message.role) {
      case _inferenceCommon.MessageRole.User:
        return {
          role: 'user',
          rawContent: (typeof message.content === 'string' ? [message.content] : message.content).map(contentPart => {
            if (typeof contentPart === 'string') {
              return {
                text: contentPart,
                type: 'text'
              };
            } else if (contentPart.type === 'text') {
              return {
                text: contentPart.text,
                type: 'text'
              };
            }
            return {
              type: 'image',
              source: {
                data: contentPart.source.data,
                mediaType: contentPart.source.mimeType,
                type: 'base64'
              }
            };
          })
        };
      case _inferenceCommon.MessageRole.Assistant:
        return {
          role: 'assistant',
          rawContent: [...(message.content ? [{
            type: 'text',
            text: message.content
          }] : []), ...(message.toolCalls ? message.toolCalls.map(toolCall => {
            return {
              type: 'tool_use',
              id: toolCall.toolCallId,
              name: toolCall.function.name,
              input: 'arguments' in toolCall.function ? toolCall.function.arguments : {}
            };
          }) : [])]
        };
      case _inferenceCommon.MessageRole.Tool:
        return {
          role: 'user',
          rawContent: [{
            type: 'tool_result',
            tool_use_id: message.toolCallId,
            content: JSON.stringify(message.response)
          }]
        };
    }
  });
};