"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createOutputApi = createOutputApi;
var _inferenceCommon = require("@kbn/inference-common");
var _rxjs = require("rxjs");
var _ensure_multi_turn = require("../utils/ensure_multi_turn");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createOutputApi(chatCompleteApi) {
  return function callOutputApi({
    id,
    connectorId,
    input,
    schema,
    system,
    previousMessages,
    modelName,
    functionCalling,
    stream,
    abortSignal,
    metadata,
    retry
  }) {
    if (stream && retry !== undefined) {
      throw new Error(`Retry options are not supported in streaming mode`);
    }
    const messages = (0, _ensure_multi_turn.ensureMultiTurn)([...(previousMessages || []), {
      role: _inferenceCommon.MessageRole.User,
      content: input
    }]);
    const response = chatCompleteApi({
      connectorId,
      stream,
      modelName,
      functionCalling,
      abortSignal,
      metadata,
      system,
      messages,
      ...(schema ? {
        tools: {
          structuredOutput: {
            description: `Use the following schema to respond to the user's request in structured data, so it can be parsed and handled.`,
            schema
          }
        },
        toolChoice: {
          function: 'structuredOutput'
        }
      } : {})
    });
    if ((0, _rxjs.isObservable)(response)) {
      return response.pipe((0, _inferenceCommon.withoutTokenCountEvents)(), (0, _rxjs.map)(event => {
        if (event.type === _inferenceCommon.ChatCompletionEventType.ChatCompletionChunk) {
          return {
            type: _inferenceCommon.OutputEventType.OutputUpdate,
            id,
            content: event.content
          };
        }
        return {
          id,
          output: event.toolCalls.length && 'arguments' in event.toolCalls[0].function ? event.toolCalls[0].function.arguments : undefined,
          content: event.content,
          type: _inferenceCommon.OutputEventType.OutputComplete
        };
      }));
    } else {
      return response.then(chatResponse => {
        return {
          id,
          content: chatResponse.content,
          output: chatResponse.toolCalls.length && 'arguments' in chatResponse.toolCalls[0].function ? chatResponse.toolCalls[0].function.arguments : undefined
        };
      }, error => {
        if ((0, _inferenceCommon.isToolValidationError)(error) && retry !== null && retry !== void 0 && retry.onValidationError) {
          var _error$meta$toolCalls, _error$meta$toolCalls2;
          const retriesLeft = typeof retry.onValidationError === 'number' ? retry.onValidationError : 1;
          return callOutputApi({
            id,
            connectorId,
            input,
            schema,
            system,
            abortSignal,
            previousMessages: messages.concat({
              role: _inferenceCommon.MessageRole.Assistant,
              content: '',
              toolCalls: error.meta.toolCalls
            }, ...((_error$meta$toolCalls = (_error$meta$toolCalls2 = error.meta.toolCalls) === null || _error$meta$toolCalls2 === void 0 ? void 0 : _error$meta$toolCalls2.map(toolCall => {
              return {
                name: toolCall.function.name,
                role: _inferenceCommon.MessageRole.Tool,
                toolCallId: toolCall.toolCallId,
                response: {
                  error: error.meta
                }
              };
            })) !== null && _error$meta$toolCalls !== void 0 ? _error$meta$toolCalls : [])),
            functionCalling,
            modelName,
            stream: false,
            retry: {
              onValidationError: retriesLeft - 1
            }
          });
        }
        throw error;
      });
    }
  };
}