"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createIndex = createIndex;
exports.createOrUpdateIndex = createOrUpdateIndex;
exports.updateIndices = updateIndices;
var _lodash = require("lodash");
var _retry_transient_es_errors = require("./retry_transient_es_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateTotalFieldLimitSetting = async ({
  logger,
  esClient,
  indexName,
  totalFieldsLimit
}) => {
  logger.debug(`Updating total field limit setting for ${indexName} data stream.`);
  try {
    const body = {
      'index.mapping.total_fields.limit': totalFieldsLimit
    };
    await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.putSettings({
      index: indexName,
      body
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Failed to PUT index.mapping.total_fields.limit settings for ${indexName}: ${err.message}`);
    throw err;
  }
};

// This will update the mappings but *not* the settings. This
// is due to the fact settings can be classed as dynamic and static, and static
// updates will fail on an index that isn't closed. New settings *will* be applied as part
// of the ILM policy rollovers. More info: https://github.com/elastic/kibana/pull/113389#issuecomment-940152654
const updateMapping = async ({
  logger,
  esClient,
  indexName,
  writeIndexOnly
}) => {
  logger.debug(`Updating mappings for ${indexName} data stream.`);
  let simulatedIndexMapping;
  try {
    simulatedIndexMapping = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.simulateIndexTemplate({
      name: indexName
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Ignored PUT mappings for ${indexName}; error generating simulated mappings: ${err.message}`);
    return;
  }
  const simulatedMapping = (0, _lodash.get)(simulatedIndexMapping, ['template', 'mappings']);
  if (simulatedMapping == null) {
    logger.error(`Ignored PUT mappings for ${indexName}; simulated mappings were empty`);
    return;
  }
  try {
    await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.putMapping({
      index: indexName,
      body: simulatedMapping,
      write_index_only: writeIndexOnly
    }), {
      logger
    });
  } catch (err) {
    logger.error(`Failed to PUT mapping for ${indexName}: ${err.message}`);
    throw err;
  }
};
/**
 * Updates the data stream mapping and total field limit setting
 */
const updateIndexMappings = async ({
  logger,
  esClient,
  totalFieldsLimit,
  indexNames,
  writeIndexOnly
}) => {
  // Update total field limit setting of found indices
  // Other index setting changes are not updated at this time
  await Promise.all(indexNames.map(indexName => updateTotalFieldLimitSetting({
    logger,
    esClient,
    totalFieldsLimit,
    indexName
  })));
  // Update mappings of the found indices.
  await Promise.all(indexNames.map(indexName => updateMapping({
    logger,
    esClient,
    totalFieldsLimit,
    indexName,
    writeIndexOnly
  })));
};
async function createOrUpdateIndex({
  logger,
  esClient,
  name,
  totalFieldsLimit
}) {
  logger.info(`Creating index - ${name}`);

  // check if index exists
  let indexExists = false;
  try {
    indexExists = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.exists({
      index: name,
      expand_wildcards: 'all'
    }), {
      logger
    });
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.error(`Error fetching index for ${name} - ${error.message}`);
      throw error;
    }
  }

  // if a index exists, update the underlying mapping
  if (indexExists) {
    await updateIndexMappings({
      logger,
      esClient,
      indexNames: [name],
      totalFieldsLimit
    });
  } else {
    try {
      await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.create({
        index: name
      }), {
        logger
      });
    } catch (error) {
      var _error$meta, _error$meta$body, _error$meta$body$erro;
      if ((error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : _error$meta$body$erro.type) !== 'resource_already_exists_exception') {
        logger.error(`Error creating index ${name} - ${error.message}`);
        throw error;
      }
    }
  }
}
async function createIndex({
  logger,
  esClient,
  name
}) {
  logger.debug(`Checking existence of index - ${name}`);

  // check if index exists
  let indexExists = false;
  try {
    indexExists = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.exists({
      index: name,
      expand_wildcards: 'all'
    }), {
      logger
    });
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.error(`Error fetching index for ${name} - ${error.message}`);
      throw error;
    }
  }

  // return if index already created
  if (indexExists) {
    return;
  }
  logger.info(`Creating index - ${name}`);
  try {
    await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.create({
      index: name
    }), {
      logger
    });
  } catch (error) {
    var _error$meta2, _error$meta2$body, _error$meta2$body$err;
    if ((error === null || error === void 0 ? void 0 : (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : (_error$meta2$body$err = _error$meta2$body.error) === null || _error$meta2$body$err === void 0 ? void 0 : _error$meta2$body$err.type) !== 'resource_already_exists_exception') {
      logger.error(`Error creating index ${name} - ${error.message}`);
      throw error;
    }
  }
}
async function updateIndices({
  logger,
  esClient,
  name,
  totalFieldsLimit,
  writeIndexOnly
}) {
  logger.info(`Updating indices - ${name}`);

  // check if data stream exists
  let indices = [];
  try {
    const response = await (0, _retry_transient_es_errors.retryTransientEsErrors)(() => esClient.indices.get({
      index: name,
      expand_wildcards: 'all'
    }), {
      logger
    });
    indices = Object.keys(response);
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.error(`Error fetching indices for ${name} - ${error.message}`);
      throw error;
    }
  }
  if (indices.length > 0) {
    await updateIndexMappings({
      logger,
      esClient,
      totalFieldsLimit,
      indexNames: indices,
      writeIndexOnly
    });
  }
}