"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NewAgentPolicySchema = exports.AgentPolicySchema = exports.AgentPolicyNamespaceSchema = exports.AgentPolicyBaseSchema = void 0;
var _configSchema = require("@kbn/config-schema");
var _constants = require("../../../common/constants");
var _services = require("../../../common/services");
var _form_settings = require("../../services/form_settings");
var _package_policy = require("./package_policy");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const AgentPolicyNamespaceSchema = exports.AgentPolicyNamespaceSchema = _configSchema.schema.string({
  minLength: 1,
  validate: value => {
    const namespaceValidation = (0, _services.isValidNamespace)(value || '');
    if (!namespaceValidation.valid && namespaceValidation.error) {
      return namespaceValidation.error;
    }
  }
});
function validateNonEmptyString(val) {
  if (val.trim() === '') {
    return 'Invalid empty string';
  }
}
const TWO_WEEKS_SECONDS = 1209600;
function isInteger(n) {
  if (!Number.isInteger(n)) {
    return `${n} is not a valid integer`;
  }
}
const memoryRegex = /^\d+(Mi|Gi)$/;
function validateMemory(s) {
  if (!memoryRegex.test(s)) {
    return 'Invalid memory format';
  }
}
const cpuRegex = /^(\d+m|\d+(\.\d+)?)$/;
function validateCPU(s) {
  if (!cpuRegex.test(s)) {
    return 'Invalid CPU format';
  }
}
const AgentPolicyBaseSchema = exports.AgentPolicyBaseSchema = {
  id: _configSchema.schema.maybe(_configSchema.schema.string()),
  space_ids: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
  name: _configSchema.schema.string({
    minLength: 1,
    validate: validateNonEmptyString
  }),
  namespace: AgentPolicyNamespaceSchema,
  description: _configSchema.schema.maybe(_configSchema.schema.string()),
  is_managed: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  has_fleet_server: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  is_default: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  is_default_fleet_server: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  unenroll_timeout: _configSchema.schema.maybe(_configSchema.schema.number({
    min: 0,
    validate: isInteger
  })),
  inactivity_timeout: _configSchema.schema.number({
    min: 0,
    defaultValue: TWO_WEEKS_SECONDS,
    validate: isInteger
  }),
  monitoring_enabled: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.dataTypes.Logs), _configSchema.schema.literal(_constants.dataTypes.Metrics), _configSchema.schema.literal(_constants.dataTypes.Traces)]))),
  keep_monitoring_alive: _configSchema.schema.maybe(_configSchema.schema.boolean({
    defaultValue: false
  })),
  data_output_id: _configSchema.schema.maybe(_configSchema.schema.nullable(_configSchema.schema.string())),
  monitoring_output_id: _configSchema.schema.maybe(_configSchema.schema.nullable(_configSchema.schema.string())),
  download_source_id: _configSchema.schema.maybe(_configSchema.schema.nullable(_configSchema.schema.string())),
  fleet_server_host_id: _configSchema.schema.maybe(_configSchema.schema.nullable(_configSchema.schema.string())),
  agent_features: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
    name: _configSchema.schema.string(),
    enabled: _configSchema.schema.boolean()
  }))),
  is_protected: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  overrides: _configSchema.schema.maybe(_configSchema.schema.nullable(_configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any(), {
    validate: val => {
      if (Object.keys(val).some(key => key.match(/^inputs(\.)?/))) {
        return 'inputs overrides is not allowed';
      }
    }
  }))),
  ...(0, _form_settings.getSettingsAPISchema)('AGENT_POLICY_ADVANCED_SETTINGS'),
  supports_agentless: _configSchema.schema.maybe(_configSchema.schema.boolean({
    defaultValue: false
  })),
  global_data_tags: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
    name: _configSchema.schema.string(),
    value: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.number()])
  }), {
    validate: validateGlobalDataTagInput
  })),
  agentless: _configSchema.schema.maybe(_configSchema.schema.object({
    resources: _configSchema.schema.maybe(_configSchema.schema.object({
      requests: _configSchema.schema.maybe(_configSchema.schema.object({
        memory: _configSchema.schema.maybe(_configSchema.schema.string({
          validate: validateMemory
        })),
        cpu: _configSchema.schema.maybe(_configSchema.schema.string({
          validate: validateCPU
        }))
      }))
    }))
  })),
  monitoring_pprof_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
  monitoring_http: _configSchema.schema.maybe(_configSchema.schema.object({
    enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
    host: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: 'localhost'
    })),
    port: _configSchema.schema.maybe(_configSchema.schema.number({
      min: 0,
      max: 65353,
      defaultValue: 6791
    })),
    buffer: _configSchema.schema.maybe(_configSchema.schema.object({
      enabled: _configSchema.schema.boolean({
        defaultValue: false
      })
    }))
  })),
  monitoring_diagnostics: _configSchema.schema.maybe(_configSchema.schema.object({
    limit: _configSchema.schema.maybe(_configSchema.schema.object({
      interval: _configSchema.schema.maybe(_configSchema.schema.string()),
      burst: _configSchema.schema.maybe(_configSchema.schema.number())
    })),
    uploader: _configSchema.schema.maybe(_configSchema.schema.object({
      max_retries: _configSchema.schema.maybe(_configSchema.schema.number()),
      init_dur: _configSchema.schema.maybe(_configSchema.schema.string()),
      max_dur: _configSchema.schema.maybe(_configSchema.schema.string())
    }))
  }))
};
function validateGlobalDataTagInput(tags) {
  const seen = new Set([]);
  const duplicates = [];
  const namesWithSpaces = [];
  const errors = [];
  for (const tag of tags) {
    if (/\s/.test(tag.name)) {
      namesWithSpaces.push(`'${tag.name}'`);
    }
    if (!seen.has(tag.name.trim())) {
      seen.add(tag.name.trim());
    } else {
      duplicates.push(`'${tag.name.trim()}'`);
    }
  }
  if (duplicates.length !== 0) {
    errors.push(`Found duplicate tag names: [${duplicates.join(', ')}], duplicate tag names are not allowed.`);
  }
  if (namesWithSpaces.length !== 0) {
    errors.push(`Found tag names with spaces: [${namesWithSpaces.join(', ')}], tag names with spaces are not allowed.`);
  }
  if (errors.length !== 0) {
    return errors.join(' ');
  }
}
const NewAgentPolicySchema = exports.NewAgentPolicySchema = _configSchema.schema.object({
  ...AgentPolicyBaseSchema,
  force: _configSchema.schema.maybe(_configSchema.schema.boolean())
});
const AgentPolicySchema = exports.AgentPolicySchema = _configSchema.schema.object({
  ...AgentPolicyBaseSchema,
  id: _configSchema.schema.string(),
  is_managed: _configSchema.schema.boolean(),
  status: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.agentPolicyStatuses.Active), _configSchema.schema.literal(_constants.agentPolicyStatuses.Inactive)]),
  package_policies: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.arrayOf(_package_policy.PackagePolicySchema)]),
  updated_at: _configSchema.schema.string(),
  updated_by: _configSchema.schema.string()
});