"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FeatureRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _server = require("@kbn/core/server");
var _common = require("../common");
var _feature_schema = require("./feature_schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Describes parameters used to retrieve all Kibana features (for internal consumers).
 */

/**
 * Describes parameters used to retrieve all Kibana features (for public consumers).
 */

class FeatureRegistry {
  constructor() {
    (0, _defineProperty2.default)(this, "locked", false);
    (0, _defineProperty2.default)(this, "kibanaFeatures", {});
    (0, _defineProperty2.default)(this, "esFeatures", {});
  }
  lockRegistration() {
    this.locked = true;
  }
  registerKibanaFeature(feature) {
    if (this.locked) {
      throw new Error(`Features are locked, can't register new features. Attempt to register ${feature.id} failed.`);
    }
    (0, _feature_schema.validateKibanaFeature)(feature);
    if (feature.id in this.kibanaFeatures || feature.id in this.esFeatures) {
      throw new Error(`Feature with id ${feature.id} is already registered.`);
    }
    const featureCopy = (0, _lodash.cloneDeep)(feature);
    this.kibanaFeatures[feature.id] = applyAutomaticPrivilegeGrants(featureCopy);
  }
  registerElasticsearchFeature(feature) {
    if (this.locked) {
      throw new Error(`Features are locked, can't register new features. Attempt to register ${feature.id} failed.`);
    }
    if (feature.id in this.kibanaFeatures || feature.id in this.esFeatures) {
      throw new Error(`Feature with id ${feature.id} is already registered.`);
    }
    (0, _feature_schema.validateElasticsearchFeature)(feature);
    const featureCopy = (0, _lodash.cloneDeep)(feature);
    this.esFeatures[feature.id] = featureCopy;
  }

  /**
   * Updates definitions for the registered features using configuration overrides, if any.
   */
  applyOverrides(overrides) {
    for (const [featureId, featureOverride] of Object.entries(overrides)) {
      var _featureOverride$subF;
      const feature = this.kibanaFeatures[featureId];
      if (!feature) {
        throw new Error(`Cannot override feature "${featureId}" since feature with such ID is not registered.`);
      }
      if (featureOverride.hidden) {
        feature.hidden = featureOverride.hidden;
      }

      // Note that the name doesn't currently support localizable strings. We'll revisit this approach when i18n support
      // becomes necessary.
      if (featureOverride.name) {
        feature.name = featureOverride.name;
      }
      if (featureOverride.category) {
        feature.category = _server.DEFAULT_APP_CATEGORIES[featureOverride.category];
      }
      if (featureOverride.order != null) {
        feature.order = featureOverride.order;
      }
      if (featureOverride.privileges) {
        for (const [privilegeId, privilegeOverride] of Object.entries(featureOverride.privileges)) {
          var _feature$privileges;
          const typedPrivilegeId = privilegeId;
          const targetPrivilege = (_feature$privileges = feature.privileges) === null || _feature$privileges === void 0 ? void 0 : _feature$privileges[typedPrivilegeId];
          if (!targetPrivilege) {
            throw new Error(`Cannot override privilege "${privilegeId}" of feature "${featureId}" since "${privilegeId}" privilege is not registered.`);
          }
          for (const featureReference of (_privilegeOverride$co = privilegeOverride.composedOf) !== null && _privilegeOverride$co !== void 0 ? _privilegeOverride$co : []) {
            var _privilegeOverride$co, _referencedFeature$pr;
            const referencedFeature = this.kibanaFeatures[featureReference.feature];
            if (!referencedFeature) {
              throw new Error(`Cannot compose privilege "${privilegeId}" of feature "${featureId}" with privileges of feature "${featureReference.feature}" since such feature is not registered.`);
            }

            // Collect all known feature and sub-feature privileges for the referenced feature.
            const knownPrivileges = new Map(Object.entries((_referencedFeature$pr = referencedFeature.privileges) !== null && _referencedFeature$pr !== void 0 ? _referencedFeature$pr : {}).concat(collectSubFeaturesPrivileges(referencedFeature)));
            for (const privilegeReference of featureReference.privileges) {
              const referencedPrivilege = knownPrivileges.get(privilegeReference);
              if (!referencedPrivilege) {
                throw new Error(`Cannot compose privilege "${privilegeId}" of feature "${featureId}" with privilege "${privilegeReference}" of feature "${featureReference.feature}" since such privilege is not registered.`);
              }
            }
          }

          // It's safe to assume that `feature.privileges` is defined here since we've checked it above.
          feature.privileges[typedPrivilegeId] = {
            ...targetPrivilege,
            ...privilegeOverride
          };
        }
      }
      if ((_featureOverride$subF = featureOverride.subFeatures) !== null && _featureOverride$subF !== void 0 && _featureOverride$subF.privileges) {
        // Collect all known sub-feature privileges for the feature.
        const knownPrivileges = new Map(collectSubFeaturesPrivileges(feature));
        if (knownPrivileges.size === 0) {
          throw new Error(`Cannot override sub-feature privileges of feature "${featureId}" since it didn't register any.`);
        }
        for (const [privilegeId, privilegeOverride] of Object.entries(featureOverride.subFeatures.privileges)) {
          const targetPrivilege = knownPrivileges.get(privilegeId);
          if (!targetPrivilege) {
            throw new Error(`Cannot override sub-feature privilege "${privilegeId}" of feature "${featureId}" since "${privilegeId}" sub-feature privilege is not registered. Known sub-feature privileges are: ${Array.from(knownPrivileges.keys())}.`);
          }
          targetPrivilege.disabled = privilegeOverride.disabled;
          if (privilegeOverride.includeIn) {
            targetPrivilege.includeIn = privilegeOverride.includeIn;
          }
        }
      }
    }
  }

  /**
   * Once all features are registered and the registry is locked, this method should validate the integrity of the registered feature set, including any potential cross-feature dependencies.
   */
  validateFeatures() {
    if (!this.locked) {
      throw new Error('Cannot validate features while the registry is not locked and still allows further feature registrations.');
    }
    for (const feature of Object.values(this.kibanaFeatures)) {
      var _feature$deprecated;
      if (!feature.privileges) {
        continue;
      }

      // Iterate over all top-level and sub-feature privileges.
      const isFeatureDeprecated = !!feature.deprecated;
      const replacementFeatureIds = new Set();
      for (const [privilegeId, privilege] of [...Object.entries(feature.privileges), ...collectSubFeaturesPrivileges(feature)]) {
        if (isFeatureDeprecated && !privilege.replacedBy) {
          throw new Error(`Feature "${feature.id}" is deprecated and must define a "replacedBy" property for privilege "${privilegeId}".`);
        }
        if (!isFeatureDeprecated && privilege.replacedBy) {
          throw new Error(`Feature "${feature.id}" is not deprecated and must not define a "replacedBy" property for privilege "${privilegeId}".`);
        }
        const replacedByReferences = privilege.replacedBy ? 'default' in privilege.replacedBy ? [...privilege.replacedBy.default, ...privilege.replacedBy.minimal] : privilege.replacedBy : [];
        for (const featureReference of replacedByReferences) {
          const referencedFeature = this.kibanaFeatures[featureReference.feature];
          if (!referencedFeature) {
            throw new Error(`Cannot replace privilege "${privilegeId}" of deprecated feature "${feature.id}" with privileges of feature "${featureReference.feature}" since such feature is not registered.`);
          }
          if (referencedFeature.deprecated) {
            throw new Error(`Cannot replace privilege "${privilegeId}" of deprecated feature "${feature.id}" with privileges of feature "${featureReference.feature}" since the referenced feature is deprecated.`);
          }

          // Collect all known feature and sub-feature privileges for the referenced feature.
          const knownPrivileges = new Map(collectPrivileges(referencedFeature).concat(collectSubFeaturesPrivileges(referencedFeature)));
          for (const privilegeReference of featureReference.privileges) {
            const referencedPrivilege = knownPrivileges.get(privilegeReference);
            if (!referencedPrivilege) {
              throw new Error(`Cannot replace privilege "${privilegeId}" of deprecated feature "${feature.id}" with privilege "${privilegeReference}" of feature "${featureReference.feature}" since such privilege is not registered.`);
            }
            if (referencedPrivilege.disabled) {
              throw new Error(`Cannot replace privilege "${privilegeId}" of deprecated feature "${feature.id}" with disabled privilege "${privilegeReference}" of feature "${featureReference.feature}".`);
            }
          }
          replacementFeatureIds.add(featureReference.feature);
        }
      }
      const featureReplacedBy = (_feature$deprecated = feature.deprecated) === null || _feature$deprecated === void 0 ? void 0 : _feature$deprecated.replacedBy;
      if (featureReplacedBy) {
        if (featureReplacedBy.length === 0) {
          throw new Error(`Feature “${feature.id}” is deprecated and must have at least one feature ID added to the “replacedBy” property, or the property must be left out completely.`);
        }

        // The feature can be marked as replaced by another feature only if that feature is actually used to replace any
        // of the deprecated feature’s privileges.
        const invalidFeatureIds = featureReplacedBy.filter(featureId => !replacementFeatureIds.has(featureId));
        if (invalidFeatureIds.length > 0) {
          throw new Error(`Cannot replace deprecated feature “${feature.id}” with the following features, as they aren’t used to replace feature privileges: ${invalidFeatureIds.join(', ')}.`);
        }
      }
    }
  }
  getAllKibanaFeatures({
    license,
    ignoreLicense = false,
    omitDeprecated = false
  } = {}) {
    if (!this.locked) {
      throw new Error('Cannot retrieve Kibana features while registration is still open');
    }
    const performLicenseCheck = license && !ignoreLicense;
    const features = [];
    for (const feature of Object.values(this.kibanaFeatures)) {
      if (omitDeprecated && feature.deprecated) {
        continue;
      }
      if (performLicenseCheck) {
        var _feature$subFeatures;
        const isCompatibleLicense = !feature.minimumLicense || license.hasAtLeast(feature.minimumLicense);
        if (!isCompatibleLicense) {
          continue;
        }
        (_feature$subFeatures = feature.subFeatures) === null || _feature$subFeatures === void 0 ? void 0 : _feature$subFeatures.forEach(subFeature => {
          subFeature.privilegeGroups.forEach(group => {
            group.privileges = group.privileges.filter(privilege => !privilege.minimumLicense || license.hasAtLeast(privilege.minimumLicense));
          });
        });
      }
      features.push(new _common.KibanaFeature(feature));
    }
    return features;
  }
  getAllElasticsearchFeatures() {
    if (!this.locked) {
      throw new Error('Cannot retrieve elasticsearch features while registration is still open');
    }
    return Object.values(this.esFeatures).map(featureConfig => new _common.ElasticsearchFeature(featureConfig));
  }
}
exports.FeatureRegistry = FeatureRegistry;
function applyAutomaticPrivilegeGrants(feature) {
  var _feature$privileges2, _feature$privileges3, _feature$reserved$pri, _feature$reserved;
  const allPrivilege = (_feature$privileges2 = feature.privileges) === null || _feature$privileges2 === void 0 ? void 0 : _feature$privileges2.all;
  const readPrivilege = (_feature$privileges3 = feature.privileges) === null || _feature$privileges3 === void 0 ? void 0 : _feature$privileges3.read;
  const reservedPrivileges = ((_feature$reserved$pri = (_feature$reserved = feature.reserved) === null || _feature$reserved === void 0 ? void 0 : _feature$reserved.privileges) !== null && _feature$reserved$pri !== void 0 ? _feature$reserved$pri : []).map(rp => rp.privilege);
  applyAutomaticAllPrivilegeGrants(allPrivilege, ...reservedPrivileges);
  applyAutomaticReadPrivilegeGrants(readPrivilege);
  return feature;
}
function applyAutomaticAllPrivilegeGrants(...allPrivileges) {
  allPrivileges.forEach(allPrivilege => {
    if (allPrivilege) {
      allPrivilege.savedObject.all = (0, _lodash.uniq)([...allPrivilege.savedObject.all, 'telemetry']);
      allPrivilege.savedObject.read = (0, _lodash.uniq)([...allPrivilege.savedObject.read, 'config', 'config-global', 'url', 'tag', 'cloud']);
    }
  });
}
function applyAutomaticReadPrivilegeGrants(...readPrivileges) {
  readPrivileges.forEach(readPrivilege => {
    if (readPrivilege) {
      readPrivilege.savedObject.read = (0, _lodash.uniq)([...readPrivilege.savedObject.read, 'config', 'config-global', 'telemetry', 'url', 'tag', 'cloud']);
    }
  });
}
function collectPrivileges(feature) {
  var _feature$privileges4;
  return Object.entries((_feature$privileges4 = feature.privileges) !== null && _feature$privileges4 !== void 0 ? _feature$privileges4 : {}).flatMap(([id, privilege]) => [[id, privilege], [`minimal_${id}`, privilege]]);
}
function collectSubFeaturesPrivileges(feature) {
  var _feature$subFeatures$, _feature$subFeatures2;
  return (_feature$subFeatures$ = (_feature$subFeatures2 = feature.subFeatures) === null || _feature$subFeatures2 === void 0 ? void 0 : _feature$subFeatures2.flatMap(subFeature => subFeature.privilegeGroups.flatMap(({
    privileges
  }) => privileges.map(privilege => [privilege.id, privilege])))) !== null && _feature$subFeatures$ !== void 0 ? _feature$subFeatures$ : [];
}