"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.strings = exports.imageFunction = void 0;
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/presentation-util-plugin/common");
var _constants = require("../constants");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const strings = exports.strings = {
  help: _i18n.i18n.translate('expressionImage.functions.imageHelpText', {
    defaultMessage: 'Displays an image. Provide an image asset as a {BASE64} data {URL}, or pass in a sub-expression.',
    values: {
      BASE64: _constants.BASE64,
      URL: _constants.URL
    }
  }),
  args: {
    dataurl: _i18n.i18n.translate('expressionImage.functions.image.args.dataurlHelpText', {
      defaultMessage: 'The {https} {URL} or {BASE64} data {URL} of an image.',
      values: {
        BASE64: _constants.BASE64,
        https: 'HTTP(S)',
        URL: _constants.URL
      }
    }),
    mode: _i18n.i18n.translate('expressionImage.functions.image.args.modeHelpText', {
      defaultMessage: '{contain} shows the entire image, scaled to fit. ' + '{cover} fills the container with the image, cropping from the sides or bottom as needed. ' + '{stretch} resizes the height and width of the image to 100% of the container.',
      values: {
        contain: `\`"${_types.ImageMode.CONTAIN}"\``,
        cover: `\`"${_types.ImageMode.COVER}"\``,
        stretch: `\`"${_types.ImageMode.STRETCH}"\``
      }
    })
  }
};
const errors = {
  invalidImageMode: () => _i18n.i18n.translate('expressionImage.functions.image.invalidImageModeErrorMessage', {
    defaultMessage: '"mode" must be "{contain}", "{cover}", or "{stretch}"',
    values: {
      contain: _types.ImageMode.CONTAIN,
      cover: _types.ImageMode.COVER,
      stretch: _types.ImageMode.STRETCH
    }
  })
};
const imageFunction = () => {
  const {
    help,
    args: argHelp
  } = strings;
  return {
    name: 'image',
    aliases: [],
    type: 'image',
    inputTypes: ['null'],
    help,
    args: {
      dataurl: {
        // This was accepting dataurl, but there was no facility in fn for checking type and handling a dataurl type.
        types: ['string', 'null'],
        help: argHelp.dataurl,
        aliases: ['_', 'url'],
        default: null
      },
      mode: {
        types: ['string'],
        help: argHelp.mode,
        default: 'contain',
        options: Object.values(_types.ImageMode)
      }
    },
    fn: async (input, {
      dataurl,
      mode
    }) => {
      if (!mode || !Object.values(_types.ImageMode).includes(mode)) {
        throw new Error(errors.invalidImageMode());
      }
      const modeStyle = mode === 'stretch' ? '100% 100%' : mode;
      const {
        elasticLogo
      } = await (0, _common.getElasticLogo)();
      return {
        type: 'image',
        mode: modeStyle,
        dataurl: (0, _common.resolveWithMissingImage)(dataurl, elasticLogo)
      };
    }
  };
};
exports.imageFunction = imageFunction;