"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggest = suggest;
var _esqlAst = require("@kbn/esql-ast");
var _builtin = require("../../../definitions/builtin");
var _types = require("../../../definitions/types");
var _helpers = require("../../../shared/helpers");
var _factories = require("../../factories");
var _helper = require("../../helper");
var _util = require("./util");
var _complete_items = require("../../complete_items");
var _constants = require("../../../shared/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

async function suggest(innerText, command, getColumnsByType, _columnExists, _getSuggestedVariableName, getExpressionType, _getPreferences, previousCommands) {
  var _previousCommands$map, _last$text;
  const suggestions = [];

  /**
   * The logic for WHERE suggestions is basically the logic for expression suggestions.
   * I assume we will eventually extract much of this to be a shared function among WHERE and EVAL
   * and anywhere else the user can enter a generic expression.
   */
  const expressionRoot = command.args[0];
  const position = (0, _util.getPosition)(innerText, command);
  switch (position) {
    /**
     * After a column name
     */
    case 'after_column':
      const columnType = getExpressionType(expressionRoot);
      if (!(0, _types.isParameterType)(columnType)) {
        break;
      }
      suggestions.push(...(0, _factories.getOperatorSuggestions)({
        command: 'where',
        leftParamType: columnType,
        // no assignments allowed in WHERE
        ignored: ['=']
      }));
      break;

    /**
     * After a complete (non-operator) function call
     */
    case 'after_function':
      const returnType = getExpressionType(expressionRoot);
      if (!(0, _types.isParameterType)(returnType)) {
        break;
      }
      suggestions.push(...(0, _factories.getOperatorSuggestions)({
        command: 'where',
        leftParamType: returnType,
        ignored: ['=']
      }));
      break;

    /**
     * After a NOT keyword
     *
     * the NOT function is a special operator that can be used in different ways,
     * and not all these are mapped within the AST data structure: in particular
     * <COMMAND> <field> NOT <here>
     * is an incomplete statement and it results in a missing AST node, so we need to detect
     * from the query string itself
     *
     * (this comment was copied but seems to still apply)
     */
    case 'after_not':
      if (expressionRoot && (0, _helpers.isFunctionItem)(expressionRoot) && expressionRoot.name === 'not') {
        suggestions.push(...(0, _factories.getFunctionSuggestions)({
          command: 'where',
          returnTypes: ['boolean']
        }), ...(await getColumnsByType('boolean', [], {
          advanceCursor: true,
          openSuggestions: true
        })));
      } else {
        suggestions.push(...(0, _factories.getSuggestionsAfterNot)());
      }
      break;

    /**
     * After an operator (e.g. AND, OR, IS NULL, +, etc.)
     */
    case 'after_operator':
      if (!expressionRoot) {
        break;
      }
      if (!(0, _helpers.isFunctionItem)(expressionRoot) || expressionRoot.subtype === 'variadic-call') {
        // this is already guaranteed in the getPosition function, but TypeScript doesn't know
        break;
      }
      let rightmostOperator = expressionRoot;
      // get rightmost function
      const walker = new _esqlAst.Walker({
        visitFunction: fn => {
          if (fn.location.min > rightmostOperator.location.min && fn.subtype !== 'variadic-call') rightmostOperator = fn;
        }
      });
      walker.walkFunction(expressionRoot);

      // See https://github.com/elastic/kibana/issues/199401 for an explanation of
      // why this check has to be so convoluted
      if (rightmostOperator.text.toLowerCase().trim().endsWith('null')) {
        suggestions.push(..._builtin.logicalOperators.map(_factories.getOperatorSuggestion));
        break;
      }
      suggestions.push(...(await (0, _helper.getSuggestionsToRightOfOperatorExpression)({
        queryText: innerText,
        commandName: 'where',
        rootOperator: rightmostOperator,
        preferredExpressionType: 'boolean',
        getExpressionType,
        getColumnsByType
      })));
      break;
    case 'empty_expression':
      // Don't suggest MATCH or QSTR after unsupported commands
      const priorCommands = (_previousCommands$map = previousCommands === null || previousCommands === void 0 ? void 0 : previousCommands.map(a => a.name)) !== null && _previousCommands$map !== void 0 ? _previousCommands$map : [];
      const ignored = [];
      if (priorCommands.some(c => _constants.UNSUPPORTED_COMMANDS_BEFORE_MATCH.has(c))) {
        ignored.push('match');
      }
      if (priorCommands.some(c => _constants.UNSUPPORTED_COMMANDS_BEFORE_QSTR.has(c))) {
        ignored.push('qstr');
      }
      const last = previousCommands === null || previousCommands === void 0 ? void 0 : previousCommands[previousCommands.length - 1];
      let columnSuggestions = [];
      if (!(last !== null && last !== void 0 && (_last$text = last.text) !== null && _last$text !== void 0 && _last$text.endsWith(`:${_constants.EDITOR_MARKER}`))) {
        columnSuggestions = await getColumnsByType('any', [], {
          advanceCursor: true,
          openSuggestions: true
        });
      }
      suggestions.push(...columnSuggestions, ...(0, _factories.getFunctionSuggestions)({
        command: 'where',
        ignored
      }));
      break;
  }

  // Is this a complete expression of the right type?
  // If so, we can call it done and suggest a pipe
  if (getExpressionType(expressionRoot) === 'boolean') {
    suggestions.push(_complete_items.pipeCompleteItem);
  }

  /**
   * Attach replacement ranges if there's a prefix.
   *
   * Can't rely on Monaco because
   * - it counts "." as a word separator
   * - it doesn't handle multi-word completions (like "is null")
   *
   * TODO - think about how to generalize this.
   */
  const hasNonWhitespacePrefix = !/\s/.test(innerText[innerText.length - 1]);
  if (hasNonWhitespacePrefix) {
    // get index of first char of final word
    const lastWhitespaceIndex = innerText.search(/\S(?=\S*$)/);
    suggestions.forEach(s => {
      if (['IS NULL', 'IS NOT NULL'].includes(s.text)) {
        // this suggestion has spaces in it (e.g. "IS NOT NULL")
        // so we need to see if there's an overlap
        const overlap = (0, _helper.getOverlapRange)(innerText, s.text);
        if (overlap.start < overlap.end) {
          // there's an overlap so use that
          s.rangeToReplace = overlap;
          return;
        }
      }

      // no overlap, so just replace from the last whitespace
      s.rangeToReplace = {
        start: lastWhitespaceIndex + 1,
        end: innerText.length
      };
    });
  }
  return suggestions;
}