"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getESQLQueryVariables = void 0;
exports.getIndexPatternFromESQLQuery = getIndexPatternFromESQLQuery;
exports.getLimitFromESQLQuery = getLimitFromESQLQuery;
exports.getValuesFromQueryField = exports.getTimeFieldFromESQLQuery = exports.getQueryColumnsFromESQLQuery = void 0;
exports.hasTransformationalCommand = hasTransformationalCommand;
exports.prettifyQuery = exports.mapVariableToColumn = exports.isQueryWrappedByPipes = void 0;
exports.removeDropCommandsFromESQLQuery = removeDropCommandsFromESQLQuery;
exports.retrieveMetadataColumns = void 0;
var _esqlAst = require("@kbn/esql-ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEFAULT_ESQL_LIMIT = 1000;

// retrieves the index pattern from the aggregate query for ES|QL using ast parsing
function getIndexPatternFromESQLQuery(esql) {
  var _sourceCommand$args;
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const sourceCommand = ast.find(({
    name
  }) => ['from', 'metrics'].includes(name));
  const args = (_sourceCommand$args = sourceCommand === null || sourceCommand === void 0 ? void 0 : sourceCommand.args) !== null && _sourceCommand$args !== void 0 ? _sourceCommand$args : [];
  const indices = args.filter(arg => arg.sourceType === 'index');
  return indices === null || indices === void 0 ? void 0 : indices.map(index => index.name).join(',');
}

// For ES|QL we consider stats and keep transformational command
// The metrics command too but only if it aggregates
function hasTransformationalCommand(esql) {
  const transformationalCommands = ['stats', 'keep'];
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const hasAtLeastOneTransformationalCommand = transformationalCommands.some(command => ast.find(({
    name
  }) => name === command));
  if (hasAtLeastOneTransformationalCommand) {
    return true;
  }
  const metricsCommand = ast.find(({
    name
  }) => name === 'metrics');
  if (metricsCommand && 'aggregates' in metricsCommand) {
    return true;
  }
  return false;
}
function getLimitFromESQLQuery(esql) {
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const limitCommands = ast.filter(({
    name
  }) => name === 'limit');
  if (!limitCommands || !limitCommands.length) {
    return DEFAULT_ESQL_LIMIT;
  }
  const limits = [];
  (0, _esqlAst.walk)(ast, {
    visitLiteral: node => {
      if (!isNaN(Number(node.value))) {
        limits.push(Number(node.value));
      }
    }
  });
  if (!limits.length) {
    return DEFAULT_ESQL_LIMIT;
  }

  // ES returns always the smallest limit
  return Math.min(...limits);
}
function removeDropCommandsFromESQLQuery(esql) {
  const pipes = (esql || '').split('|');
  return pipes.filter(statement => !/DROP\s/i.test(statement)).join('|');
}

/**
 * When the ?_tstart and ?_tend params are used, we want to retrieve the timefield from the query.
 * @param esql:string
 * @returns string
 */
const getTimeFieldFromESQLQuery = esql => {
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const functions = [];
  (0, _esqlAst.walk)(ast, {
    visitFunction: node => functions.push(node)
  });
  const params = _esqlAst.Walker.params(ast);
  const timeNamedParam = params.find(param => param.value === '_tstart' || param.value === '_tend');
  if (!timeNamedParam || !functions.length) {
    return undefined;
  }
  const allFunctionsWithNamedParams = functions.filter(({
    location
  }) => location.min <= timeNamedParam.location.min && location.max >= timeNamedParam.location.max);
  if (!allFunctionsWithNamedParams.length) {
    return undefined;
  }
  const lowLevelFunction = allFunctionsWithNamedParams[allFunctionsWithNamedParams.length - 1];
  const column = lowLevelFunction.args.find(arg => {
    const argument = arg;
    return argument.type === 'column';
  });
  return column === null || column === void 0 ? void 0 : column.name;
};
exports.getTimeFieldFromESQLQuery = getTimeFieldFromESQLQuery;
const isQueryWrappedByPipes = query => {
  const {
    ast
  } = (0, _esqlAst.parse)(query);
  const numberOfCommands = ast.length;
  const pipesWithNewLine = query.split('\n  |');
  return numberOfCommands === (pipesWithNewLine === null || pipesWithNewLine === void 0 ? void 0 : pipesWithNewLine.length);
};
exports.isQueryWrappedByPipes = isQueryWrappedByPipes;
const prettifyQuery = (query, isWrapped) => {
  const {
    root
  } = (0, _esqlAst.parse)(query);
  return _esqlAst.BasicPrettyPrinter.print(root, {
    multiline: !isWrapped
  });
};
exports.prettifyQuery = prettifyQuery;
const retrieveMetadataColumns = esql => {
  var _metadataOptions$args;
  const {
    ast
  } = (0, _esqlAst.parse)(esql);
  const options = [];
  (0, _esqlAst.walk)(ast, {
    visitCommandOption: node => options.push(node)
  });
  const metadataOptions = options.find(({
    name
  }) => name === 'metadata');
  return (_metadataOptions$args = metadataOptions === null || metadataOptions === void 0 ? void 0 : metadataOptions.args.map(column => column.name)) !== null && _metadataOptions$args !== void 0 ? _metadataOptions$args : [];
};
exports.retrieveMetadataColumns = retrieveMetadataColumns;
const getQueryColumnsFromESQLQuery = esql => {
  const {
    root
  } = (0, _esqlAst.parse)(esql);
  const columns = [];
  (0, _esqlAst.walk)(root, {
    visitColumn: node => columns.push(node)
  });
  return columns.map(column => column.name);
};
exports.getQueryColumnsFromESQLQuery = getQueryColumnsFromESQLQuery;
const getESQLQueryVariables = esql => {
  const {
    root
  } = (0, _esqlAst.parse)(esql);
  const usedVariablesInQuery = _esqlAst.Walker.params(root);
  return usedVariablesInQuery.map(v => v.text.replace('?', ''));
};

/**
 * This function is used to map the variables to the columns in the datatable
 * @param esql:string
 * @param variables:ESQLControlVariable[]
 * @param columns:DatatableColumn[]
 * @returns DatatableColumn[]
 */
exports.getESQLQueryVariables = getESQLQueryVariables;
const mapVariableToColumn = (esql, variables, columns) => {
  if (!variables.length) {
    return columns;
  }
  const usedVariablesInQuery = getESQLQueryVariables(esql);
  const uniqueVariablesInQyery = new Set(usedVariablesInQuery);
  columns.map(column => {
    if (variables.some(variable => variable.value === column.id)) {
      var _variable$key;
      const potentialColumnVariables = variables.filter(variable => variable.value === column.id);
      const variable = potentialColumnVariables.find(v => uniqueVariablesInQyery.has(v.key));
      column.variable = (_variable$key = variable === null || variable === void 0 ? void 0 : variable.key) !== null && _variable$key !== void 0 ? _variable$key : '';
    }
  });
  return columns;
};
exports.mapVariableToColumn = mapVariableToColumn;
const getValuesFromQueryField = queryString => {
  const validQuery = `${queryString} ""`;
  const {
    root
  } = (0, _esqlAst.parse)(validQuery);
  const lastCommand = root.commands[root.commands.length - 1];
  const columns = [];
  (0, _esqlAst.walk)(lastCommand, {
    visitColumn: node => columns.push(node)
  });
  const column = _esqlAst.Walker.match(lastCommand, {
    type: 'column'
  });
  if (column) {
    return `${column.name}`;
  }
};
exports.getValuesFromQueryField = getValuesFromQueryField;