"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.makeSynthNode = exports.createSynthMethod = exports.SynthNode = void 0;
var _builder = require("../builder");
var _walker = require("../walker/walker");
var _pretty_print = require("../pretty_print");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const serialize = node => {
  return node.type === 'command' ? _pretty_print.BasicPrettyPrinter.command(node) : _pretty_print.BasicPrettyPrinter.expression(node);
};

/**
 * This is used as a prototype of AST nodes created by the synth methods.
 * It implements the `toString` method, which is invoked when the node is
 * coerced to a string. So you can easily convert the node to a string by
 * calling `String(node)` or `${node}`:
 *
 * ```js
 * const node = expr`a.b`;  // { type: 'column', name: 'a.b' }
 * String(node)             // 'a.b'
 * ```
 */
class SynthNode {
  toString() {
    return serialize(this);
  }
}
exports.SynthNode = SynthNode;
const makeSynthNode = ast => {
  // Add SynthNode prototype to the AST node.
  Object.setPrototypeOf(ast, new SynthNode());

  // Remove parser generated fields.
  _walker.Walker.walk(ast, {
    visitAny: node => {
      Object.assign(node, _builder.Builder.parserFields({}));
    }
  });
};
exports.makeSynthNode = makeSynthNode;
const createSynthMethod = generator => {
  const templateStringTag = opts => {
    return (template, ...params) => {
      let src = '';
      const length = template.length;
      for (let i = 0; i < length; i++) {
        src += template[i];
        if (i < params.length) {
          const param = params[i];
          if (typeof param === 'string') src += param;else src += serialize(param);
        }
      }
      return generator(src, opts);
    };
  };
  const method = (...args) => {
    const [first] = args;

    /**
     * Usage as function:
     *
     * ```js
     * expr('42');
     * ```
     */
    if (typeof first === 'string') return generator(first, args[1]);

    /**
     * Usage as tagged template:
     *
     * ```js
     * expr`42`;
     * ```
     */
    if (Array.isArray(first)) {
      return templateStringTag()(first, ...args.slice(1));
    }

    /**
     * Usage as tagged template, with ability to specify parsing options:
     *
     * ```js
     * expr({ withFormatting: false })`42`;
     * ```
     */
    return templateStringTag(args[0]);
  };
  return method;
};
exports.createSynthMethod = createSynthMethod;