"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseErrors = exports.parse = exports.getParser = exports.getLexer = exports.createParser = void 0;
var _antlr = require("antlr4");
var _esql_error_listener = require("./esql_error_listener");
var _esql_ast_builder_listener = require("./esql_ast_builder_listener");
var _constants = require("./constants");
var _formatting = require("./formatting");
var _builder = require("../builder");
var _esql_lexer = _interopRequireDefault(require("../antlr/esql_lexer"));
var _esql_parser = _interopRequireDefault(require("../antlr/esql_parser"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getLexer = (inputStream, errorListener) => {
  const lexer = new _esql_lexer.default(inputStream);
  lexer.removeErrorListeners();
  lexer.addErrorListener(errorListener);
  return lexer;
};
exports.getLexer = getLexer;
const getParser = (inputStream, errorListener, parseListener) => {
  const lexer = getLexer(inputStream, errorListener);
  const tokens = new _antlr.CommonTokenStream(lexer);
  const parser = new _esql_parser.default(tokens);

  // lexer.symbolicNames

  parser.removeErrorListeners();
  parser.addErrorListener(errorListener);
  if (parseListener) {
    // @ts-expect-error the addParseListener API does exist and is documented here
    // https://github.com/antlr/antlr4/blob/dev/doc/listeners.md
    parser.addParseListener(parseListener);
  }
  return {
    lexer,
    tokens,
    parser
  };
};
exports.getParser = getParser;
const createParser = src => {
  const errorListener = new _esql_error_listener.ESQLErrorListener();
  const parseListener = new _esql_ast_builder_listener.ESQLAstBuilderListener(src);
  return getParser(_antlr.CharStreams.fromString(src), errorListener, parseListener);
};

// These will need to be manually updated whenever the relevant grammar changes.
exports.createParser = createParser;
const SYNTAX_ERRORS_TO_IGNORE = [`SyntaxError: mismatched input '<EOF>' expecting {'explain', 'from', 'row', 'show'}`];
const parse = (text, options = {}) => {
  try {
    if (text == null) {
      const commands = [];
      return {
        ast: commands,
        root: _builder.Builder.expression.query(commands),
        errors: [],
        tokens: []
      };
    }
    const errorListener = new _esql_error_listener.ESQLErrorListener();
    const parseListener = new _esql_ast_builder_listener.ESQLAstBuilderListener(text);
    const {
      tokens,
      parser
    } = getParser(_antlr.CharStreams.fromString(text), errorListener, parseListener);
    parser[_constants.GRAMMAR_ROOT_RULE]();
    const errors = errorListener.getErrors().filter(error => {
      return !SYNTAX_ERRORS_TO_IGNORE.includes(error.message);
    });
    const {
      ast: commands
    } = parseListener.getAst();
    const root = _builder.Builder.expression.query(commands, {
      location: {
        min: 0,
        max: text.length - 1
      }
    });
    if (options.withFormatting) {
      const decorations = (0, _formatting.collectDecorations)(tokens);
      (0, _formatting.attachDecorations)(root, tokens.tokens, decorations.lines);
    }
    return {
      root,
      ast: commands,
      errors,
      tokens: tokens.tokens
    };
  } catch (error) {
    /**
     * Parsing should never fail, meaning this branch should never execute. But
     * if it does fail, we want to log the error message for easier debugging.
     */
    // eslint-disable-next-line no-console
    console.error(error);
    const root = _builder.Builder.expression.query();
    return {
      root,
      ast: root.commands,
      errors: [{
        startLineNumber: 0,
        endLineNumber: 0,
        startColumn: 0,
        endColumn: 0,
        message: 'Parsing internal error: ' + (!!error && typeof error === 'object' ? String(error.message) : String(error)),
        severity: 'error'
      }],
      tokens: []
    };
  }
};
exports.parse = parse;
const parseErrors = text => {
  const errorListener = new _esql_error_listener.ESQLErrorListener();
  const {
    parser
  } = getParser(_antlr.CharStreams.fromString(text), errorListener);
  parser[_constants.GRAMMAR_ROOT_RULE]();
  return errorListener.getErrors();
};
exports.parseErrors = parseErrors;