"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.readSourceDefinitions = readSourceDefinitions;
exports.storeSourceDefinition = storeSourceDefinition;
var _constants = require("../constants");
var _run_esql_query = require("../run_esql_query");
var _entity_definition_conflict = require("../errors/entity_definition_conflict");
var _type_definition = require("./type_definition");
var _unknown_entity_type = require("../errors/unknown_entity_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function storeSourceDefinition({
  source,
  clusterClient,
  logger,
  replace = false
}) {
  const esClient = clusterClient.asInternalUser;
  try {
    await (0, _type_definition.readTypeDefinitionById)(source.type_id, clusterClient, logger);
  } catch (error) {
    if (error instanceof _unknown_entity_type.UnknownEntityType) {
      throw new _unknown_entity_type.UnknownEntityType(`Type with ID ${source.type_id} not found, cannot attach source with ID ${source.id}`);
    }
    throw error;
  }
  const sources = await (0, _run_esql_query.runESQLQuery)('fetch source definition for conflict check', {
    esClient,
    query: `FROM ${_constants.DEFINITIONS_ALIAS} METADATA _id | WHERE definition_type == "source" AND _id == "${source.type_id}:${source.id}" | KEEP _id`,
    logger
  });
  if (sources.length !== 0 && replace === false) {
    logger.debug(`Entity source definition with ID ${source.id} already exists`);
    throw new _entity_definition_conflict.EntityDefinitionConflict('source', source.id);
  }
  const definition = {
    template_version: _constants.TEMPLATE_VERSION,
    definition_type: 'source',
    source
  };
  logger.debug(`Installing entity source definition ${source.id} for type ${source.type_id}`);
  await esClient.index({
    index: _constants.DEFINITIONS_ALIAS,
    id: `${source.type_id}:${definition.source.id}`,
    document: definition,
    refresh: 'wait_for'
  });
  return definition.source;
}
async function readSourceDefinitions(clusterClient, logger, options) {
  const esClient = clusterClient.asInternalUser;
  const typeFilter = options !== null && options !== void 0 && options.type ? `AND source.type_id == "${options.type}"` : '';
  const sources = await (0, _run_esql_query.runESQLQuery)('fetch all source definitions', {
    esClient,
    query: `FROM ${_constants.DEFINITIONS_ALIAS} METADATA _source | WHERE definition_type == "source" ${typeFilter} | KEEP _source`,
    logger
  });
  return sources.map(storedTypeDefinition => storedTypeDefinition._source.source);
}