"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_NAME_LENGTH = exports.GroupLogic = void 0;
var _kea = require("kea");
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _kibana = require("../../../shared/kibana");
var _routes = require("../../routes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_NAME_LENGTH = exports.MAX_NAME_LENGTH = 40;
const GroupLogic = exports.GroupLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'workplace_search', 'group'],
  actions: {
    onInitializeGroup: group => group,
    onGroupNameChanged: group => group,
    onGroupPrioritiesChanged: group => group,
    onGroupNameInputChange: groupName => groupName,
    addGroupSource: sourceId => sourceId,
    removeGroupSource: sourceId => sourceId,
    onGroupSourcesSaved: group => group,
    setGroupModalErrors: errors => errors,
    hideOrgSourcesModal: group => group,
    selectAllSources: contentSources => contentSources,
    updatePriority: (id, boost) => ({
      id,
      boost
    }),
    resetGroup: () => true,
    showConfirmDeleteModal: () => true,
    hideConfirmDeleteModal: () => true,
    showOrgSourcesModal: () => true,
    resetFlashMessages: () => true,
    initializeGroup: groupId => ({
      groupId
    }),
    deleteGroup: () => true,
    updateGroupName: () => true,
    saveGroupSources: () => true,
    saveGroupSourcePrioritization: () => true
  },
  reducers: {
    group: [{}, {
      // @ts-expect-error upgrade typescript v5.1.6
      onInitializeGroup: (_, group) => group,
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupNameChanged: (_, group) => group,
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupSourcesSaved: (_, group) => group,
      resetGroup: () => ({})
    }],
    dataLoading: [true, {
      onInitializeGroup: () => false,
      onGroupPrioritiesChanged: () => false,
      resetGroup: () => true
    }],
    managerModalFormErrors: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setGroupModalErrors: (_, errors) => errors
    }],
    orgSourcesModalVisible: [false, {
      showOrgSourcesModal: () => true,
      hideOrgSourcesModal: () => false,
      onGroupSourcesSaved: () => false
    }],
    confirmDeleteModalVisible: [false, {
      showConfirmDeleteModal: () => true,
      hideConfirmDeleteModal: () => false,
      deleteGroup: () => false
    }],
    groupNameInputValue: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      onInitializeGroup: (_, {
        name
      }) => name,
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupNameChanged: (_, {
        name
      }) => name,
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupNameInputChange: (_, name) => name
    }],
    selectedGroupSources: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      onInitializeGroup: (_, {
        contentSources
      }) => contentSources.map(({
        id
      }) => id),
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupSourcesSaved: (_, {
        contentSources
      }) => contentSources.map(({
        id
      }) => id),
      // @ts-expect-error upgrade typescript v5.1.6
      selectAllSources: (_, contentSources) => contentSources.map(({
        id
      }) => id),
      // @ts-expect-error upgrade typescript v5.1.6
      hideOrgSourcesModal: (_, {
        contentSources
      }) => contentSources.map(({
        id
      }) => id),
      // @ts-expect-error upgrade typescript v5.1.6
      addGroupSource: (state, sourceId) => [...state, sourceId].sort(),
      // @ts-expect-error upgrade typescript v5.1.6
      removeGroupSource: (state, sourceId) => state.filter(id => id !== sourceId)
    }],
    cachedSourcePriorities: [{}, {
      // @ts-expect-error upgrade typescript v5.1.6
      onInitializeGroup: (_, {
        contentSources
      }) => mapPriorities(contentSources),
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupPrioritiesChanged: (_, {
        contentSources
      }) => mapPriorities(contentSources),
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupSourcesSaved: (_, {
        contentSources
      }) => mapPriorities(contentSources)
    }],
    activeSourcePriorities: [{}, {
      // @ts-expect-error upgrade typescript v5.1.6
      onInitializeGroup: (_, {
        contentSources
      }) => mapPriorities(contentSources),
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupPrioritiesChanged: (_, {
        contentSources
      }) => mapPriorities(contentSources),
      // @ts-expect-error upgrade typescript v5.1.6
      onGroupSourcesSaved: (_, {
        contentSources
      }) => mapPriorities(contentSources),
      // @ts-expect-error upgrade typescript v5.1.6
      updatePriority: (state, {
        id,
        boost
      }) => {
        const updated = {
          ...state
        };
        updated[id] = boost;
        return updated;
      }
    }]
  },
  selectors: ({
    selectors
  }) => ({
    groupPrioritiesUnchanged: [() => [selectors.cachedSourcePriorities, selectors.activeSourcePriorities], (cached, active) => (0, _lodash.isEqual)(cached, active)]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    initializeGroup: async ({
      groupId
    }) => {
      try {
        const response = await _http.HttpLogic.values.http.get(`/internal/workplace_search/groups/${groupId}`);
        actions.onInitializeGroup(response);
      } catch (e) {
        var _e$response;
        const NOT_FOUND_MESSAGE = _i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.groups.groupNotFound', {
          defaultMessage: 'Unable to find group with ID: "{groupId}".',
          values: {
            groupId
          }
        });
        const error = ((_e$response = e.response) === null || _e$response === void 0 ? void 0 : _e$response.status) === 404 ? NOT_FOUND_MESSAGE : e;
        (0, _flash_messages.setQueuedErrorMessage)(error);
        _kibana.KibanaLogic.values.navigateToUrl(_routes.GROUPS_PATH);
      }
    },
    deleteGroup: async () => {
      const {
        group: {
          id,
          name
        }
      } = values;
      try {
        await _http.HttpLogic.values.http.delete(`/internal/workplace_search/groups/${id}`);
        const GROUP_DELETED_MESSAGE = _i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.groups.groupDeleted', {
          defaultMessage: 'Group "{groupName}" was successfully deleted.',
          values: {
            groupName: name
          }
        });
        (0, _flash_messages.flashSuccessToast)(GROUP_DELETED_MESSAGE);
        _kibana.KibanaLogic.values.navigateToUrl(_routes.GROUPS_PATH);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    updateGroupName: async () => {
      const {
        group: {
          id
        },
        groupNameInputValue
      } = values;
      try {
        const response = await _http.HttpLogic.values.http.put(`/internal/workplace_search/groups/${id}`, {
          body: JSON.stringify({
            group: {
              name: groupNameInputValue
            }
          })
        });
        actions.onGroupNameChanged(response);
        const GROUP_RENAMED_MESSAGE = _i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.groups.groupRenamed', {
          defaultMessage: 'Successfully renamed this group to "{groupName}".',
          values: {
            groupName: response.name
          }
        });
        (0, _flash_messages.flashSuccessToast)(GROUP_RENAMED_MESSAGE);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    saveGroupSources: async () => {
      const {
        group: {
          id
        },
        selectedGroupSources
      } = values;
      try {
        const response = await _http.HttpLogic.values.http.post(`/internal/workplace_search/groups/${id}/share`, {
          body: JSON.stringify({
            content_source_ids: selectedGroupSources
          })
        });
        actions.onGroupSourcesSaved(response);
        const GROUP_SOURCES_UPDATED_MESSAGE = _i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.groups.groupSourcesUpdated', {
          defaultMessage: 'Successfully updated organizational content sources.'
        });
        (0, _flash_messages.flashSuccessToast)(GROUP_SOURCES_UPDATED_MESSAGE);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    saveGroupSourcePrioritization: async () => {
      const {
        group: {
          id
        },
        activeSourcePriorities
      } = values;

      // server expects an array of id, value for each boost.
      // example: [['123abc', 7], ['122abv', 1]]
      const boosts = [];
      Object.keys(activeSourcePriorities).forEach(k => boosts.push([k, Number(activeSourcePriorities[k])]));
      try {
        const response = await _http.HttpLogic.values.http.put(`/internal/workplace_search/groups/${id}/boosts`, {
          body: JSON.stringify({
            content_source_boosts: boosts
          })
        });
        const GROUP_PRIORITIZATION_UPDATED_MESSAGE = _i18n.i18n.translate('xpack.enterpriseSearch.workplaceSearch.groups.groupPrioritizationUpdated', {
          defaultMessage: 'Successfully updated organizational source prioritization.'
        });
        (0, _flash_messages.flashSuccessToast)(GROUP_PRIORITIZATION_UPDATED_MESSAGE);
        actions.onGroupPrioritiesChanged(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    showConfirmDeleteModal: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    showOrgSourcesModal: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    resetFlashMessages: () => {
      (0, _flash_messages.clearFlashMessages)();
    }
  })
});
const mapPriorities = contentSources => {
  const prioritiesMap = {};
  contentSources.forEach(({
    id,
    boost
  }) => {
    prioritiesMap[id] = boost;
  });
  return prioritiesMap;
};