"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mountHttpLogic = exports.HttpLogic = void 0;
var _kea = require("kea");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const HttpLogic = exports.HttpLogic = (0, _kea.kea)({
  actions: {
    initializeHttpInterceptors: () => null,
    onConnectionError: errorConnectingMessage => ({
      errorConnectingMessage
    }),
    setHttpInterceptors: httpInterceptors => ({
      httpInterceptors
    }),
    setReadOnlyMode: readOnlyMode => ({
      readOnlyMode
    })
  },
  events: ({
    values,
    actions
  }) => ({
    afterMount: () => {
      actions.initializeHttpInterceptors();
    },
    beforeUnmount: () => {
      values.httpInterceptors.forEach(removeInterceptorFn => {
        if (removeInterceptorFn) removeInterceptorFn();
      });
    }
  }),
  listeners: ({
    values,
    actions
  }) => ({
    initializeHttpInterceptors: () => {
      const httpInterceptors = [];
      const errorConnectingInterceptor = values.http.intercept({
        responseError: async httpResponse => {
          if (isEnterpriseSearchApi(httpResponse)) {
            const hasErrorConnecting = httpResponse.response.headers.get(_constants.ERROR_CONNECTING_HEADER);
            if (hasErrorConnecting === 'true') {
              const {
                status,
                statusText
              } = httpResponse.response;
              actions.onConnectionError(`${status} ${statusText}`);
            }
          }

          // Re-throw error so that downstream catches work as expected
          return Promise.reject(httpResponse);
        }
      });
      httpInterceptors.push(errorConnectingInterceptor);
      const readOnlyModeInterceptor = values.http.intercept({
        response: async httpResponse => {
          if (isEnterpriseSearchApi(httpResponse)) {
            const readOnlyMode = httpResponse.response.headers.get(_constants.READ_ONLY_MODE_HEADER);
            if (readOnlyMode === 'true') {
              actions.setReadOnlyMode(true);
            } else {
              actions.setReadOnlyMode(false);
            }
          }
          return Promise.resolve(httpResponse);
        }
      });
      httpInterceptors.push(readOnlyModeInterceptor);
      actions.setHttpInterceptors(httpInterceptors);
    }
  }),
  path: ['enterprise_search', 'http_logic'],
  reducers: ({
    props
  }) => ({
    errorConnectingMessage: [props.errorConnectingMessage || '', {
      // @ts-expect-error upgrade typescript v5.1.6
      onConnectionError: (_, {
        errorConnectingMessage
      }) => errorConnectingMessage
    }],
    http: [props.http, {}],
    httpInterceptors: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      setHttpInterceptors: (_, {
        httpInterceptors
      }) => httpInterceptors
    }],
    readOnlyMode: [props.readOnlyMode || false, {
      // @ts-expect-error upgrade typescript v5.1.6
      setReadOnlyMode: (_, {
        readOnlyMode
      }) => readOnlyMode
    }]
  })
});

/**
 * Mount/props helper
 */

const mountHttpLogic = props => {
  HttpLogic(props);
  return HttpLogic.mount();
};

/**
 * Small helper that checks whether or not an http call is for an Enterprise Search API
 */
exports.mountHttpLogic = mountHttpLogic;
const isEnterpriseSearchApi = httpResponse => {
  if (!httpResponse.response) return false; // Typically this means Kibana has stopped working, in which case we short-circuit early to prevent errors

  const {
    url
  } = httpResponse.response;
  return url.includes('/internal/app_search/') || url.includes('/internal/workplace_search/');
};