"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DEFAULT_VALUES = exports.AutomaticCrawlSchedulerLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../../../shared/flash_messages");
var _http = require("../../../../../shared/http");
var _update_connector_scheduling_api_logic = require("../../../../api/connector/update_connector_scheduling_api_logic");
var _types = require("../../../../api/crawler/types");
var _utils = require("../../../../api/crawler/utils");
var _index_name_logic = require("../../index_name_logic");
var _index_view_logic = require("../../index_view_logic");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_VALUES = exports.DEFAULT_VALUES = {
  crawlFrequency: 24,
  crawlUnit: _types.CrawlUnits.hours
};
const AutomaticCrawlSchedulerLogic = exports.AutomaticCrawlSchedulerLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'crawler', 'automatic_crawl_scheduler_logic'],
  connect: {
    actions: [_update_connector_scheduling_api_logic.UpdateConnectorSchedulingApiLogic, ['makeRequest as makeUpdateConnectorSchedulingRequest', 'apiError as updateConnectorSchedulingApiError']],
    values: [_index_view_logic.IndexViewLogic, ['index']]
  },
  actions: () => ({
    clearCrawlSchedule: true,
    deleteCrawlSchedule: true,
    disableCrawlAutomatically: true,
    onDoneSubmitting: true,
    enableCrawlAutomatically: true,
    fetchCrawlSchedule: true,
    saveChanges: true,
    setCrawlSchedule: crawlSchedule => ({
      crawlSchedule
    }),
    submitConnectorSchedule: scheduling => ({
      scheduling
    }),
    submitCrawlSchedule: true,
    setCrawlAutomatically: crawlAutomatically => ({
      crawlAutomatically
    }),
    setCrawlFrequency: crawlFrequency => ({
      crawlFrequency
    }),
    setCrawlUnit: crawlUnit => ({
      crawlUnit
    }),
    setUseConnectorSchedule: useConnectorSchedule => ({
      useConnectorSchedule
    })
  }),
  reducers: () => ({
    crawlAutomatically: [false, {
      clearCrawlSchedule: () => false,
      // @ts-expect-error upgrade typescript v5.1.6
      setCrawlAutomatically: (_, {
        crawlAutomatically
      }) => crawlAutomatically,
      setCrawlSchedule: () => true
    }],
    crawlFrequency: [DEFAULT_VALUES.crawlFrequency, {
      clearCrawlSchedule: () => DEFAULT_VALUES.crawlFrequency,
      // @ts-expect-error upgrade typescript v5.1.6
      setCrawlSchedule: (_, {
        crawlSchedule: {
          frequency
        }
      }) => frequency,
      // @ts-expect-error upgrade typescript v5.1.6
      setCrawlFrequency: (_, {
        crawlFrequency
      }) => crawlFrequency,
      // @ts-expect-error upgrade typescript v5.1.6
      setUseConnectorSchedule: crawlFrequency => crawlFrequency || DEFAULT_VALUES.crawlFrequency
    }],
    crawlUnit: [DEFAULT_VALUES.crawlUnit, {
      clearCrawlSchedule: () => DEFAULT_VALUES.crawlUnit,
      // @ts-expect-error upgrade typescript v5.1.6
      setCrawlSchedule: (_, {
        crawlSchedule: {
          unit
        }
      }) => unit,
      // @ts-expect-error upgrade typescript v5.1.6
      setCrawlUnit: (_, {
        crawlUnit
      }) => crawlUnit,
      // @ts-expect-error upgrade typescript v5.1.6
      setUseConnectorSchedule: crawlUnit => crawlUnit || DEFAULT_VALUES.crawlUnit
    }],
    isSubmitting: [false, {
      deleteCrawlSchedule: () => true,
      onDoneSubmitting: () => false,
      submitCrawlSchedule: () => true
    }],
    useConnectorSchedule: [false, {
      // @ts-expect-error upgrade typescript v5.1.6
      setCrawlAutomatically: (useConnectorSchedule, {
        crawlAutomatically
      }) => crawlAutomatically || useConnectorSchedule,
      // @ts-expect-error upgrade typescript v5.1.6
      setCrawlSchedule: (_, {
        crawlSchedule: {
          useConnectorSchedule = false
        }
      }) => useConnectorSchedule,
      // @ts-expect-error upgrade typescript v5.1.6
      setUseConnectorSchedule: (_, {
        useConnectorSchedule
      }) => useConnectorSchedule
    }]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    deleteCrawlSchedule: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      try {
        await http.delete(`/internal/enterprise_search/indices/${indexName}/crawler/crawl_schedule`);
      } catch (e) {
        var _e$response;
        // A 404 is expected and means the user has no crawl schedule to delete
        if (((_e$response = e.response) === null || _e$response === void 0 ? void 0 : _e$response.status) !== 404) {
          (0, _flash_messages.flashAPIErrors)(e);
        }
      } finally {
        actions.onDoneSubmitting();
      }
    },
    fetchCrawlSchedule: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      try {
        const crawlSchedule = await http.get(`/internal/enterprise_search/indices/${indexName}/crawler/crawl_schedule`);
        actions.setCrawlSchedule((0, _utils.crawlScheduleServerToClient)(crawlSchedule));
      } catch (e) {
        var _e$response2;
        // A 404 is expected and means the user does not have crawl schedule
        // for this index. We continue to use the defaults.
        if (((_e$response2 = e.response) === null || _e$response2 === void 0 ? void 0 : _e$response2.status) !== 404) {
          (0, _flash_messages.flashAPIErrors)(e);
        }
      }
    },
    saveChanges: () => {
      if (values.crawlAutomatically) {
        actions.submitCrawlSchedule();
      } else {
        actions.deleteCrawlSchedule();
      }
      actions.submitConnectorSchedule({
        ...values.index.connector.scheduling.full,
        enabled: values.crawlAutomatically && values.useConnectorSchedule
      });
    },
    setCrawlAutomatically: actions.saveChanges,
    setCrawlFrequency: actions.saveChanges,
    setCrawlUnit: actions.saveChanges,
    setUseConnectorSchedule: actions.saveChanges,
    submitConnectorSchedule: ({
      scheduling
    }) => {
      actions.makeUpdateConnectorSchedulingRequest({
        connectorId: values.index.connector.id,
        scheduling: {
          ...values.index.connector.scheduling,
          full: scheduling
        }
      });
    },
    submitCrawlSchedule: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        indexName
      } = _index_name_logic.IndexNameLogic.values;
      if (!values.crawlUnit || !values.crawlFrequency) {
        return;
      }
      try {
        const crawlSchedule = await http.put(`/internal/enterprise_search/indices/${indexName}/crawler/crawl_schedule`, {
          body: JSON.stringify({
            frequency: values.crawlFrequency,
            unit: values.crawlUnit,
            use_connector_schedule: values.useConnectorSchedule
          })
        });
        actions.setCrawlSchedule((0, _utils.crawlScheduleServerToClient)(crawlSchedule));
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      } finally {
        actions.onDoneSubmitting();
      }
    },
    updateConnectorSchedulingApiError: e => (0, _flash_messages.flashAPIErrors)(e)
  }),
  events: ({
    actions
  }) => ({
    afterMount: () => {
      actions.fetchCrawlSchedule();
    }
  })
});