"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CurationsLogic = void 0;
var _kea = require("kea");
var _constants = require("../../../shared/constants");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _kibana = require("../../../shared/kibana");
var _table_pagination = require("../../../shared/table_pagination");
var _routes = require("../../routes");
var _engine = require("../engine");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CurationsLogic = exports.CurationsLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'curations_logic'],
  actions: () => ({
    onCurationsLoad: ({
      results,
      meta
    }) => ({
      results,
      meta
    }),
    onPaginate: newPageIndex => ({
      newPageIndex
    }),
    loadCurations: true,
    deleteCuration: id => id,
    createCuration: queries => queries,
    onSelectPageTab: pageTab => ({
      pageTab
    })
  }),
  reducers: () => ({
    selectedPageTab: ['overview', {
      // @ts-expect-error upgrade typescript v5.1.6
      onSelectPageTab: (_, {
        pageTab
      }) => pageTab
    }],
    dataLoading: [true, {
      onCurationsLoad: () => false
    }],
    curations: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      onCurationsLoad: (_, {
        results
      }) => results
    }],
    meta: [_constants.DEFAULT_META, {
      // @ts-expect-error upgrade typescript v5.1.6
      onCurationsLoad: (_, {
        meta
      }) => meta,
      // @ts-expect-error upgrade typescript v5.1.6
      onPaginate: (state, {
        newPageIndex
      }) => (0, _table_pagination.updateMetaPageIndex)(state, newPageIndex)
    }]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    loadCurations: async () => {
      const {
        meta
      } = values;
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.get(`/internal/app_search/engines/${engineName}/curations`, {
          query: {
            'page[current]': meta.page.current,
            'page[size]': meta.page.size
          }
        });
        actions.onCurationsLoad(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    deleteCuration: async id => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      (0, _flash_messages.clearFlashMessages)();
      if (window.confirm(_constants2.DELETE_CONFIRMATION_MESSAGE)) {
        try {
          await http.delete(`/internal/app_search/engines/${engineName}/curations/${id}`);
          actions.loadCurations();
          (0, _flash_messages.flashSuccessToast)(_constants2.DELETE_SUCCESS_MESSAGE);
        } catch (e) {
          (0, _flash_messages.flashAPIErrors)(e);
        }
      }
    },
    createCuration: async queries => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      const {
        navigateToUrl
      } = _kibana.KibanaLogic.values;
      (0, _flash_messages.clearFlashMessages)();
      try {
        const response = await http.post(`/internal/app_search/engines/${engineName}/curations`, {
          body: JSON.stringify({
            queries
          })
        });
        navigateToUrl((0, _engine.generateEnginePath)(_routes.ENGINE_CURATION_PATH, {
          curationId: response.id
        }));
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    onSelectPageTab: () => {
      (0, _flash_messages.clearFlashMessages)();
    }
  })
});