"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CurationLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../../shared/flash_messages");
var _http = require("../../../../shared/http");
var _kibana = require("../../../../shared/kibana");
var _routes = require("../../../routes");
var _engine = require("../../engine");
var _constants = require("../constants");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CurationLogic = exports.CurationLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'curation_logic'],
  actions: () => ({
    convertToManual: true,
    deleteCuration: true,
    loadCuration: true,
    onCurationLoad: curation => ({
      curation
    }),
    updateCuration: true,
    onCurationError: true,
    updateQueries: queries => ({
      queries
    }),
    setActiveQuery: query => ({
      query
    }),
    setPromotedIds: promotedIds => ({
      promotedIds
    }),
    addPromotedId: id => ({
      id
    }),
    removePromotedId: id => ({
      id
    }),
    clearPromotedIds: true,
    addHiddenId: id => ({
      id
    }),
    removeHiddenId: id => ({
      id
    }),
    clearHiddenIds: true,
    onSelectPageTab: pageTab => ({
      pageTab
    })
  }),
  reducers: () => ({
    dataLoading: [true, {
      loadCuration: () => true,
      onCurationLoad: () => false,
      onCurationError: () => false
    }],
    curation: [{
      id: '',
      last_updated: '',
      queries: [],
      promoted: [],
      organic: [],
      hidden: []
    }, {
      // @ts-expect-error upgrade typescript v5.1.6
      onCurationLoad: (_, {
        curation
      }) => curation
    }],
    queries: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      onCurationLoad: (_, {
        curation
      }) => curation.queries,
      // @ts-expect-error upgrade typescript v5.1.6
      updateQueries: (_, {
        queries
      }) => queries
    }],
    queriesLoading: [false, {
      updateQueries: () => true,
      onCurationLoad: () => false,
      onCurationError: () => false
    }],
    activeQuery: ['', {
      // @ts-expect-error upgrade typescript v5.1.6
      setActiveQuery: (_, {
        query
      }) => query,
      // @ts-expect-error upgrade typescript v5.1.6
      onCurationLoad: (activeQuery, {
        curation
      }) => activeQuery || curation.queries[0]
    }],
    organicDocumentsLoading: [false, {
      setActiveQuery: () => true,
      onCurationLoad: () => false,
      onCurationError: () => false
    }],
    promotedIds: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      onCurationLoad: (_, {
        curation
      }) => curation.promoted.map(document => document.id),
      // @ts-expect-error upgrade typescript v5.1.6
      setPromotedIds: (_, {
        promotedIds
      }) => promotedIds,
      // @ts-expect-error upgrade typescript v5.1.6
      addPromotedId: (promotedIds, {
        id
      }) => (0, _utils.addDocument)(promotedIds, id),
      // @ts-expect-error upgrade typescript v5.1.6
      removePromotedId: (promotedIds, {
        id
      }) => (0, _utils.removeDocument)(promotedIds, id),
      clearPromotedIds: () => []
    }],
    promotedDocumentsLoading: [false, {
      setPromotedIds: () => true,
      addPromotedId: () => true,
      removePromotedId: () => true,
      clearPromotedIds: () => true,
      onCurationLoad: () => false,
      onCurationError: () => false
    }],
    hiddenIds: [[], {
      // @ts-expect-error upgrade typescript v5.1.6
      onCurationLoad: (_, {
        curation
      }) => curation.hidden.map(document => document.id),
      // @ts-expect-error upgrade typescript v5.1.6
      addHiddenId: (hiddenIds, {
        id
      }) => (0, _utils.addDocument)(hiddenIds, id),
      // @ts-expect-error upgrade typescript v5.1.6
      removeHiddenId: (hiddenIds, {
        id
      }) => (0, _utils.removeDocument)(hiddenIds, id),
      clearHiddenIds: () => []
    }],
    hiddenDocumentsLoading: [false, {
      addHiddenId: () => true,
      removeHiddenId: () => true,
      clearHiddenIds: () => true,
      onCurationLoad: () => false,
      onCurationError: () => false
    }],
    selectedPageTab: ['promoted', {
      // @ts-expect-error upgrade typescript v5.1.6
      onSelectPageTab: (_, {
        pageTab
      }) => pageTab
    }]
  }),
  selectors: ({
    selectors
  }) => ({
    isAutomated: [() => [selectors.curation], curation => {
      var _curation$suggestion;
      return ((_curation$suggestion = curation.suggestion) === null || _curation$suggestion === void 0 ? void 0 : _curation$suggestion.status) === 'automated';
    }]
  }),
  listeners: ({
    actions,
    values,
    props
  }) => ({
    convertToManual: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        await http.put(`/internal/app_search/engines/${engineName}/adaptive_relevance/suggestions`, {
          body: JSON.stringify([{
            query: values.activeQuery,
            type: 'curation',
            status: 'applied'
          }])
        });
        actions.loadCuration();
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    deleteCuration: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      const {
        navigateToUrl
      } = _kibana.KibanaLogic.values;
      try {
        await http.delete(`/internal/app_search/engines/${engineName}/curations/${props.curationId}`);
        navigateToUrl((0, _engine.generateEnginePath)(_routes.ENGINE_CURATIONS_PATH));
        (0, _flash_messages.flashSuccessToast)(_constants.DELETE_SUCCESS_MESSAGE);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    loadCuration: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.get(`/internal/app_search/engines/${engineName}/curations/${props.curationId}`, {
          query: {
            skip_record_analytics: 'true'
          }
        });
        actions.onCurationLoad(response);
      } catch (e) {
        const {
          navigateToUrl
        } = _kibana.KibanaLogic.values;
        (0, _flash_messages.flashAPIErrors)(e, {
          isQueued: true
        });
        navigateToUrl((0, _engine.generateEnginePath)(_routes.ENGINE_CURATIONS_PATH));
      }
    },
    updateCuration: async (_, breakpoint) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      await breakpoint(100);
      (0, _flash_messages.clearFlashMessages)();
      try {
        const response = await http.put(`/internal/app_search/engines/${engineName}/curations/${props.curationId}`, {
          query: {
            skip_record_analytics: 'true'
          },
          body: JSON.stringify({
            queries: values.queries,
            query: values.activeQuery,
            promoted: values.promotedIds,
            hidden: values.hiddenIds
          })
        });
        actions.onCurationLoad(response);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
        actions.onCurationError();
      }
    },
    updateQueries: ({
      queries
    }) => {
      const activeQueryDeleted = !queries.includes(values.activeQuery);
      if (activeQueryDeleted) actions.setActiveQuery(queries[0]);
      actions.updateCuration();
    },
    onSelectPageTab: () => {
      (0, _flash_messages.clearFlashMessages)();
    },
    setActiveQuery: () => actions.updateCuration(),
    setPromotedIds: () => actions.updateCuration(),
    addPromotedId: () => actions.updateCuration(),
    removePromotedId: () => actions.updateCuration(),
    clearPromotedIds: () => actions.updateCuration(),
    addHiddenId: () => actions.updateCuration(),
    removeHiddenId: () => actions.updateCuration(),
    clearHiddenIds: () => actions.updateCuration()
  })
});