"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AddDomainLogic = void 0;
var _kea = require("kea");
var _i18n = require("@kbn/i18n");
var _flash_messages = require("../../../../../shared/flash_messages");
var _handle_api_errors = require("../../../../../shared/flash_messages/handle_api_errors");
var _http = require("../../../../../shared/http");
var _kibana = require("../../../../../shared/kibana");
var _routes = require("../../../../routes");
var _engine = require("../../../engine");
var _crawler_logic = require("../../crawler_logic");
var _utils = require("../../utils");
var _utils2 = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_SELECTOR_VALUES = {
  addDomainFormInputValue: 'https://',
  entryPointValue: '/',
  domainValidationResult: {
    steps: {
      contentVerification: {
        state: ''
      },
      indexingRestrictions: {
        state: ''
      },
      initialValidation: {
        state: ''
      },
      networkConnectivity: {
        state: ''
      }
    }
  },
  allowSubmit: false,
  ignoreValidationFailure: false,
  isValidationLoading: false
};
const AddDomainLogic = exports.AddDomainLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'crawler', 'add_domain'],
  actions: () => ({
    clearDomainFormInputValue: true,
    initialValidation: true,
    performDomainValidationStep: (stepName, checks) => ({
      stepName,
      checks
    }),
    onSubmitNewDomainSuccess: domain => ({
      domain
    }),
    onSubmitNewDomainError: errors => ({
      errors
    }),
    setAddDomainFormInputValue: newValue => newValue,
    setDomainValidationResult: change => ({
      change
    }),
    setIgnoreValidationFailure: newValue => newValue,
    startDomainValidation: true,
    submitNewDomain: true,
    validateDomainInitialVerification: (newValue, newEntryPointValue) => ({
      newValue,
      newEntryPointValue
    }),
    validateDomainContentVerification: true,
    validateDomainIndexingRestrictions: true,
    validateDomainNetworkConnectivity: true
  }),
  // @ts-expect-error upgrade typescript v5.1.6
  reducers: () => ({
    addDomainFormInputValue: [DEFAULT_SELECTOR_VALUES.addDomainFormInputValue, {
      clearDomainFormInputValue: () => DEFAULT_SELECTOR_VALUES.addDomainFormInputValue,
      // @ts-expect-error upgrade typescript v5.1.6
      setAddDomainFormInputValue: (_, newValue) => newValue,
      // @ts-expect-error upgrade typescript v5.1.6
      validateDomainInitialVerification: (_, {
        newValue
      }) => newValue
    }],
    domainValidationResult: [DEFAULT_SELECTOR_VALUES.domainValidationResult, {
      clearDomainFormInputValue: () => DEFAULT_SELECTOR_VALUES.domainValidationResult,
      setAddDomainFormInputValue: () => DEFAULT_SELECTOR_VALUES.domainValidationResult,
      // @ts-expect-error upgrade typescript v5.1.6
      setDomainValidationResult: ({
        steps
      }, {
        change
      }) => ({
        steps: {
          ...steps,
          ...change
        }
      }),
      startDomainValidation: () => ({
        steps: {
          contentVerification: {
            state: 'loading'
          },
          indexingRestrictions: {
            state: 'loading'
          },
          initialValidation: {
            state: 'loading'
          },
          networkConnectivity: {
            state: 'loading'
          }
        }
      })
    }],
    entryPointValue: [DEFAULT_SELECTOR_VALUES.entryPointValue, {
      clearDomainFormInputValue: () => DEFAULT_SELECTOR_VALUES.entryPointValue,
      setAddDomainFormInputValue: () => DEFAULT_SELECTOR_VALUES.entryPointValue,
      // @ts-expect-error upgrade typescript v5.1.6
      validateDomainInitialVerification: (_, {
        newEntryPointValue
      }) => newEntryPointValue
    }],
    errors: [[], {
      clearDomainFormInputValue: () => [],
      setAddDomainFormInputValue: () => [],
      validateDomainInitialVerification: () => [],
      submitNewDomain: () => [],
      // @ts-expect-error upgrade typescript v5.1.6
      onSubmitNewDomainError: (_, {
        errors
      }) => errors
    }],
    ignoreValidationFailure: [DEFAULT_SELECTOR_VALUES.ignoreValidationFailure, {
      clearDomainFormInputValue: () => DEFAULT_SELECTOR_VALUES.ignoreValidationFailure,
      setAddDomainFormInputValue: () => DEFAULT_SELECTOR_VALUES.ignoreValidationFailure,
      // @ts-expect-error upgrade typescript v5.1.6
      setIgnoreValidationFailure: (_, newValue) => newValue
    }]
  }),
  selectors: ({
    selectors
  }) => ({
    isValidationLoading: [() => [selectors.domainValidationResult], domainValidationResult => !!Object.values(domainValidationResult.steps).find(step => step.state === 'loading')],
    hasValidationCompleted: [() => [selectors.domainValidationResult], domainValidationResult => !Object.values(domainValidationResult.steps).find(step => step.state === 'loading' || step.state === '')],
    hasBlockingFailure: [() => [selectors.domainValidationResult], domainValidationResult => !!Object.values(domainValidationResult.steps).find(step => step.blockingFailure)],
    canIgnoreValidationFailure: [() => [selectors.hasValidationCompleted, selectors.domainValidationResult], (hasValidationCompleted, domainValidationResult) => {
      if (!hasValidationCompleted) {
        return false;
      }
      return domainValidationResult.steps.indexingRestrictions.blockingFailure || domainValidationResult.steps.contentVerification.blockingFailure;
    }],
    allowSubmit: [() => [selectors.ignoreValidationFailure, selectors.hasValidationCompleted, selectors.hasBlockingFailure], (ignoreValidationFailure, hasValidationCompleted, hasBlockingFailure) => {
      if (ignoreValidationFailure) {
        return true;
      }
      return hasValidationCompleted && !hasBlockingFailure;
    }],
    displayValidation: [() => [selectors.isValidationLoading, selectors.hasValidationCompleted], (isValidationLoading, hasValidationCompleted) => isValidationLoading || hasValidationCompleted]
  }),
  listeners: ({
    actions,
    values
  }) => ({
    onSubmitNewDomainSuccess: ({
      domain
    }) => {
      (0, _flash_messages.flashSuccessToast)(_i18n.i18n.translate('xpack.enterpriseSearch.appSearch.crawler.domainsTable.action.add.successMessage', {
        defaultMessage: "Successfully added domain ''{domainUrl}''",
        values: {
          domainUrl: domain.url
        }
      }));
      _kibana.KibanaLogic.values.navigateToUrl((0, _engine.generateEnginePath)(_routes.ENGINE_CRAWLER_DOMAIN_PATH, {
        domainId: domain.id
      }));
    },
    performDomainValidationStep: async ({
      stepName,
      checks
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const failureResultChange = (0, _utils2.domainValidationFailureResultChange)(stepName);
      const route = '/internal/app_search/crawler/validate_url';
      try {
        const data = await http.post(route, {
          body: JSON.stringify({
            url: values.addDomainFormInputValue.trim(),
            checks
          })
        });
        const result = (0, _utils.crawlDomainValidationToResult)(data);
        if (result.blockingFailure) {
          actions.setDomainValidationResult({
            [stepName]: result,
            ...failureResultChange
          });
        } else {
          actions.setDomainValidationResult({
            [stepName]: result
          });

          // Trigger next step
          switch (stepName) {
            case 'initialValidation':
              actions.validateDomainNetworkConnectivity();
              break;
            case 'networkConnectivity':
              actions.validateDomainIndexingRestrictions();
              break;
            case 'indexingRestrictions':
              actions.validateDomainContentVerification();
              break;
          }
        }
      } catch (e) {
        actions.setDomainValidationResult({
          [stepName]: {
            state: 'invalid',
            blockingFailure: true,
            message: _i18n.i18n.translate('xpack.enterpriseSearch.appSearch.crawler.addDomainForm.unexpectedValidationErrorMessage', {
              defaultMessage: 'Unexpected error'
            })
          },
          ...failureResultChange
        });
      }
    },
    submitNewDomain: async () => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      const requestBody = JSON.stringify({
        name: values.addDomainFormInputValue.trim(),
        entry_points: [{
          value: values.entryPointValue
        }]
      });
      try {
        const response = await http.post(`/internal/app_search/engines/${engineName}/crawler/domains`, {
          query: {
            respond_with: 'crawler_details'
          },
          body: requestBody
        });
        const crawlerData = (0, _utils.crawlerDataServerToClient)(response);
        _crawler_logic.CrawlerLogic.actions.onReceiveCrawlerData(crawlerData);
        const newDomain = crawlerData.domains[crawlerData.domains.length - 1];
        if (newDomain) {
          actions.onSubmitNewDomainSuccess(newDomain);
        }
        // If there is not a new domain, that means the server responded with a 200 but
        // didn't actually persist the new domain to our BE, and we take no action
      } catch (e) {
        // we surface errors inside the form instead of in flash messages
        const errorMessages = (0, _handle_api_errors.getErrorsFromHttpResponse)(e);
        actions.onSubmitNewDomainError(errorMessages);
      }
    },
    startDomainValidation: async () => {
      const {
        domain,
        entryPoint
      } = (0, _utils2.extractDomainAndEntryPointFromUrl)(values.addDomainFormInputValue.trim());
      const domainWithProtocol = await (0, _utils2.getDomainWithProtocol)(domain);
      actions.validateDomainInitialVerification(domainWithProtocol, entryPoint);
    },
    validateDomainInitialVerification: () => {
      actions.performDomainValidationStep('initialValidation', ['url']);
    },
    validateDomainContentVerification: () => {
      actions.performDomainValidationStep('contentVerification', ['url_request', 'url_content']);
    },
    validateDomainIndexingRestrictions: () => {
      actions.performDomainValidationStep('indexingRestrictions', ['robots_txt']);
    },
    validateDomainNetworkConnectivity: () => {
      actions.performDomainValidationStep('networkConnectivity', ['dns', 'tcp']);
    }
  })
});