"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getKnowledgeBaseIndicesRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _lodash = require("lodash");
var _build_response = require("../../lib/build_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get the indices that have fields of `semantic_text` type
 *
 * @param router IRouter for registering routes
 */
const getKnowledgeBaseIndicesRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _elasticAssistantCommon.ELASTIC_AI_ASSISTANT_KNOWLEDGE_BASE_INDICES_URL,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: false
  }, async (context, _, response) => {
    const resp = (0, _build_response.buildResponse)(response);
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const logger = ctx.elasticAssistant.logger;
    const esClient = ctx.core.elasticsearch.client.asCurrentUser;
    try {
      const res = await esClient.fieldCaps({
        index: '*',
        fields: '*',
        types: ['sparse_vector'],
        include_unmapped: true,
        filter_path: 'fields.*.sparse_vector.indices'
      });
      const indicesWithSemanticTextField = (0, _lodash.reduce)(res.fields, (acc, value, key) => {
        if (key.endsWith('.inference.chunks.embeddings')) {
          var _value$sparse_vector$, _value$sparse_vector;
          (0, _lodash.forEach)((_value$sparse_vector$ = value === null || value === void 0 ? void 0 : (_value$sparse_vector = value.sparse_vector) === null || _value$sparse_vector === void 0 ? void 0 : _value$sparse_vector.indices) !== null && _value$sparse_vector$ !== void 0 ? _value$sparse_vector$ : [], index => acc.add(index));
        }
        return acc;
      }, new Set());
      const mappings = await esClient.indices.getMapping({
        index: Array.from(indicesWithSemanticTextField),
        filter_path: '*.mappings.properties'
      });
      const findSemanticTextPaths = (obj, currentPath = []) => {
        let paths = [];
        for (const key in obj) {
          if (Object.prototype.hasOwnProperty.call(obj, key)) {
            // @ts-expect-error
            const value = obj[key];
            const newPath = currentPath.concat(key);
            if (value && typeof value === 'object') {
              if (value.type === 'semantic_text') {
                paths.push(newPath.filter(pathPart => pathPart !== 'properties' && pathPart !== 'mappings'));
              } else {
                paths = paths.concat(findSemanticTextPaths(value, newPath));
              }
            }
          }
        }
        return paths;
      };
      const result = {};
      for (const index in mappings) {
        if (Object.prototype.hasOwnProperty.call(mappings, index)) {
          const paths = findSemanticTextPaths(mappings[index]);
          if (paths.length) {
            result[index] = paths.map(path => path.join('.'));
          }
        }
      }
      return response.ok({
        body: result
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.getKnowledgeBaseIndicesRoute = getKnowledgeBaseIndicesRoute;