"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDataStateContainer = getDataStateContainer;
var _rxjs = require("rxjs");
var _common = require("@kbn/inspector-plugin/common");
var _esQuery = require("@kbn/es-query");
var _ebtTools = require("@kbn/ebt-tools");
var _discoverUtils = require("@kbn/discover-utils");
var _get_esql_data_view = require("./utils/get_esql_data_view");
var _types = require("../../types");
var _validate_time_range = require("./utils/validate_time_range");
var _fetch_all = require("../data_fetching/fetch_all");
var _use_saved_search_messages = require("../hooks/use_saved_search_messages");
var _get_fetch_observable = require("../data_fetching/get_fetch_observable");
var _get_default_profile_state = require("./utils/get_default_profile_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Container responsible for fetching of data in Discover Main
 * Either by triggering requests to Elasticsearch directly, or by
 * orchestrating unified plugins / components like the histogram
 */
function getDataStateContainer({
  services,
  searchSessionManager,
  appStateContainer,
  internalStateContainer,
  getSavedSearch,
  setDataView
}) {
  const {
    data,
    uiSettings,
    toastNotifications,
    profilesManager
  } = services;
  const {
    timefilter
  } = data.query.timefilter;
  const inspectorAdapters = {
    requests: new _common.RequestAdapter()
  };
  const fetchChart$ = new _rxjs.Subject();
  const disableNextFetchOnStateChange$ = new _rxjs.BehaviorSubject(false);

  /**
   * The observable to trigger data fetching in UI
   * By refetch$.next('reset') rows and fieldcounts are reset to allow e.g. editing of runtime fields
   * to be processed correctly
   */
  const refetch$ = new _rxjs.Subject();
  const getInitialFetchStatus = () => {
    const shouldSearchOnPageLoad = uiSettings.get(_discoverUtils.SEARCH_ON_PAGE_LOAD_SETTING) || getSavedSearch().id !== undefined || !timefilter.getRefreshInterval().pause || searchSessionManager.hasSearchSessionIdInURL();
    return shouldSearchOnPageLoad ? _types.FetchStatus.LOADING : _types.FetchStatus.UNINITIALIZED;
  };

  /**
   * The observables the UI (aka React component) subscribes to get notified about
   * the changes in the data fetching process (high level: fetching started, data was received)
   */
  const initialState = {
    fetchStatus: getInitialFetchStatus()
  };
  const dataSubjects = {
    main$: new _rxjs.BehaviorSubject(initialState),
    documents$: new _rxjs.BehaviorSubject(initialState),
    totalHits$: new _rxjs.BehaviorSubject(initialState)
  };
  // This is debugging code, helping you to understand which messages are sent to the data observables
  // Adding a debugger in the functions can be helpful to understand what triggers a message
  // dataSubjects.main$.subscribe((msg) => addLog('dataSubjects.main$', msg));
  // dataSubjects.documents$.subscribe((msg) => addLog('dataSubjects.documents$', msg));
  // dataSubjects.totalHits$.subscribe((msg) => addLog('dataSubjects.totalHits$', msg););
  // Add window.ELASTIC_DISCOVER_LOGGER = 'debug' to see messages in console

  let autoRefreshDone = null;
  /**
   * handler emitted by `timefilter.getAutoRefreshFetch$()`
   * to notify when data completed loading and to start a new autorefresh loop
   */
  const setAutoRefreshDone = fn => {
    autoRefreshDone = fn;
  };
  const fetch$ = (0, _get_fetch_observable.getFetch$)({
    setAutoRefreshDone,
    data,
    main$: dataSubjects.main$,
    refetch$,
    searchSource: getSavedSearch().searchSource,
    searchSessionManager
  }).pipe((0, _rxjs.filter)(() => (0, _validate_time_range.validateTimeRange)(timefilter.getTime(), toastNotifications)), (0, _rxjs.tap)(() => inspectorAdapters.requests.reset()), (0, _rxjs.map)(val => ({
    options: {
      reset: val === 'reset',
      fetchMore: val === 'fetch_more'
    },
    searchSessionId: val === 'fetch_more' && searchSessionManager.getCurrentSearchSessionId() || searchSessionManager.getNextSearchSessionId()
  })), (0, _rxjs.share)());
  let abortController;
  let abortControllerFetchMore;
  function subscribe() {
    const subscription = fetch$.pipe((0, _rxjs.mergeMap)(async ({
      options,
      searchSessionId
    }) => {
      var _abortController, _abortControllerFetch;
      const commonFetchDeps = {
        initialFetchStatus: getInitialFetchStatus(),
        inspectorAdapters,
        searchSessionId,
        services,
        getAppState: appStateContainer.getState,
        getInternalState: internalStateContainer.getState,
        savedSearch: getSavedSearch(),
        useNewFieldsApi: !uiSettings.get(_discoverUtils.SEARCH_FIELDS_FROM_SOURCE)
      };
      (_abortController = abortController) === null || _abortController === void 0 ? void 0 : _abortController.abort();
      (_abortControllerFetch = abortControllerFetchMore) === null || _abortControllerFetch === void 0 ? void 0 : _abortControllerFetch.abort();
      if (options.fetchMore) {
        abortControllerFetchMore = new AbortController();
        const fetchMoreStartTime = window.performance.now();
        await (0, _fetch_all.fetchMoreDocuments)(dataSubjects, {
          abortController: abortControllerFetchMore,
          ...commonFetchDeps
        });
        const fetchMoreDuration = window.performance.now() - fetchMoreStartTime;
        (0, _ebtTools.reportPerformanceMetricEvent)(services.analytics, {
          eventName: 'discoverFetchMore',
          duration: fetchMoreDuration
        });
        return;
      }
      internalStateContainer.transitions.setDataRequestParams({
        timeRangeAbsolute: timefilter.getAbsoluteTime(),
        timeRangeRelative: timefilter.getTime()
      });
      await profilesManager.resolveDataSourceProfile({
        dataSource: appStateContainer.getState().dataSource,
        dataView: getSavedSearch().searchSource.getField('index'),
        query: appStateContainer.getState().query
      });
      const {
        resetDefaultProfileState,
        dataView
      } = internalStateContainer.getState();
      const defaultProfileState = dataView ? (0, _get_default_profile_state.getDefaultProfileState)({
        profilesManager,
        resetDefaultProfileState,
        dataView
      }) : undefined;
      const preFetchStateUpdate = defaultProfileState === null || defaultProfileState === void 0 ? void 0 : defaultProfileState.getPreFetchState();
      if (preFetchStateUpdate) {
        disableNextFetchOnStateChange$.next(true);
        await appStateContainer.replaceUrlState(preFetchStateUpdate);
        disableNextFetchOnStateChange$.next(false);
      }

      // Trigger chart fetching after the pre fetch state has been updated
      // to ensure state values that would affect data fetching are set
      fetchChart$.next();
      abortController = new AbortController();
      const prevAutoRefreshDone = autoRefreshDone;
      const fetchAllStartTime = window.performance.now();
      await (0, _fetch_all.fetchAll)(dataSubjects, options.reset, {
        abortController,
        ...commonFetchDeps
      }, async () => {
        const {
          resetDefaultProfileState: currentResetDefaultProfileState
        } = internalStateContainer.getState();
        if (currentResetDefaultProfileState.resetId !== resetDefaultProfileState.resetId) {
          return;
        }
        const {
          esqlQueryColumns
        } = dataSubjects.documents$.getValue();
        const defaultColumns = uiSettings.get(_discoverUtils.DEFAULT_COLUMNS_SETTING, []);
        const postFetchStateUpdate = defaultProfileState === null || defaultProfileState === void 0 ? void 0 : defaultProfileState.getPostFetchState({
          defaultColumns,
          esqlQueryColumns
        });
        if (postFetchStateUpdate) {
          await appStateContainer.replaceUrlState(postFetchStateUpdate);
        }

        // Clear the default profile state flags after the data fetching
        // is done so refetches don't reset the state again
        internalStateContainer.transitions.setResetDefaultProfileState({
          columns: false,
          rowHeight: false,
          breakdownField: false
        });
      });
      const fetchAllDuration = window.performance.now() - fetchAllStartTime;
      (0, _ebtTools.reportPerformanceMetricEvent)(services.analytics, {
        eventName: 'discoverFetchAll',
        duration: fetchAllDuration
      });

      // If the autoRefreshCallback is still the same as when we started i.e. there was no newer call
      // replacing this current one, call it to make sure we tell that the auto refresh is done
      // and a new one can be scheduled. null is checked to always start initial looping.
      if (autoRefreshDone === prevAutoRefreshDone || prevAutoRefreshDone === null) {
        var _autoRefreshDone;
        // if this function was set and is executed, another refresh fetch can be triggered
        (_autoRefreshDone = autoRefreshDone) === null || _autoRefreshDone === void 0 ? void 0 : _autoRefreshDone();
        autoRefreshDone = undefined;
      }
    })).subscribe();
    return () => {
      var _abortController2, _abortControllerFetch2;
      (_abortController2 = abortController) === null || _abortController2 === void 0 ? void 0 : _abortController2.abort();
      (_abortControllerFetch2 = abortControllerFetchMore) === null || _abortControllerFetch2 === void 0 ? void 0 : _abortControllerFetch2.abort();
      subscription.unsubscribe();
    };
  }
  const fetchQuery = async resetQuery => {
    const query = appStateContainer.getState().query;
    const currentDataView = getSavedSearch().searchSource.getField('index');
    if ((0, _esQuery.isOfAggregateQueryType)(query)) {
      const nextDataView = await (0, _get_esql_data_view.getEsqlDataView)(query, currentDataView, services);
      if (nextDataView !== currentDataView) {
        setDataView(nextDataView);
      }
    }
    if (resetQuery) {
      refetch$.next('reset');
    } else {
      refetch$.next(undefined);
    }
    return refetch$;
  };
  const fetchMore = () => {
    refetch$.next('fetch_more');
    return refetch$;
  };
  const reset = () => {
    (0, _use_saved_search_messages.sendResetMsg)(dataSubjects, getInitialFetchStatus());
  };
  const cancel = () => {
    var _abortController3, _abortControllerFetch3;
    (_abortController3 = abortController) === null || _abortController3 === void 0 ? void 0 : _abortController3.abort();
    (_abortControllerFetch3 = abortControllerFetchMore) === null || _abortControllerFetch3 === void 0 ? void 0 : _abortControllerFetch3.abort();
  };
  const getAbortController = () => {
    return abortController;
  };
  return {
    fetch: fetchQuery,
    fetchMore,
    data$: dataSubjects,
    refetch$,
    fetchChart$,
    disableNextFetchOnStateChange$,
    subscribe,
    reset,
    inspectorAdapters,
    getInitialFetchStatus,
    cancel,
    getAbortController
  };
}