"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addMappingsToIndices = addMappingsToIndices;
exports.addNamespace = addNamespace;
exports.getAllIndices = getAllIndices;
exports.getIndexBasicStats = getIndexBasicStats;
exports.getIndexFieldStats = getIndexFieldStats;
exports.getIndexStats = getIndexStats;
exports.groupStatsByPatternName = groupStatsByPatternName;
exports.indexStatsToTelemetryEvents = indexStatsToTelemetryEvents;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieves all indices and data streams for each stream of logs.
 */
function getAllIndices({
  esClient,
  logsIndexPatterns,
  excludeStreamsStartingWith,
  breatheDelay
}) {
  const uniqueIndices = new Set();
  const indicesInfo = [];
  return (0, _rxjs.from)(logsIndexPatterns).pipe((0, _rxjs.concatMap)(pattern => (0, _rxjs.of)(pattern).pipe((0, _rxjs.delay)(breatheDelay), (0, _rxjs.concatMap)(() => {
    return (0, _rxjs.forkJoin)([(0, _rxjs.from)(getDataStreamsInfoForPattern({
      esClient,
      pattern
    })), (0, _rxjs.from)(getIndicesInfoForPattern({
      esClient,
      pattern
    }))]);
  }), (0, _rxjs.map)(([patternDataStreamsInfo, patternIndicesInfo]) => {
    return [...patternDataStreamsInfo, ...patternIndicesInfo];
  }), (0, _rxjs.map)(indicesAndDataStreams => {
    // Exclude indices that have already been dealt with
    return indicesAndDataStreams.filter(dataStream => {
      return !uniqueIndices.has(dataStream.name);
    });
  }), (0, _rxjs.map)(indicesAndDataStreams => {
    // Exclude internal or backing indices
    return indicesAndDataStreams.filter(dataStream => !dataStream.name.startsWith('.'));
  }), (0, _rxjs.map)(indicesAndDataStreams => {
    return indicesAndDataStreams.filter(
    // Exclude streams starting with non log known signals
    dataStream => !excludeStreamsStartingWith.some(excludeStream => dataStream.name.startsWith(excludeStream)));
  }), (0, _rxjs.map)(indicesAndDataStreams => {
    indicesAndDataStreams.forEach(dataStream => {
      uniqueIndices.add(dataStream.name);
    });
    return indicesAndDataStreams;
  }), (0, _rxjs.map)(dataStreamsInfoRecords => {
    indicesInfo.push(...dataStreamsInfoRecords);
    return dataStreamsInfoRecords;
  }))), (0, _rxjs.toArray)(), (0, _rxjs.map)(() => indicesInfo));
}

/**
 * Retrieves the mappings at once and adds to the indices info.
 */
function addMappingsToIndices({
  esClient,
  dataStreamsInfo,
  logsIndexPatterns
}) {
  return (0, _rxjs.from)(esClient.indices.getMapping({
    index: logsIndexPatterns.map(pattern => pattern.pattern)
  })).pipe((0, _rxjs.map)(mappings => {
    return dataStreamsInfo.map(info => {
      // Add mapping for each index
      info.indices.forEach(index => {
        if (mappings[index]) {
          var _info$mapping;
          info.mapping = {
            ...((_info$mapping = info.mapping) !== null && _info$mapping !== void 0 ? _info$mapping : {}),
            [index]: mappings[index]
          };
        }
      });
      return info;
    });
  }));
}

/**
 * Adds the namespace of the index from index mapping if available.
 */
function addNamespace({
  dataStreamsInfo
}) {
  return (0, _rxjs.from)(dataStreamsInfo).pipe((0, _rxjs.concatMap)(indexInfo => (0, _rxjs.of)(indexInfo).pipe((0, _rxjs.map)(dataStream => getIndexNamespace(dataStream)), (0, _rxjs.map)(namespace => {
    indexInfo.namespace = namespace;
    return indexInfo;
  }))), (0, _rxjs.toArray)());
}
function groupStatsByPatternName(dataStreamsStats) {
  const uniqueNamespaces = new Set();
  const uniqueFields = new Set();
  const statsByStream = dataStreamsStats.reduce((acc, stats) => {
    var _stats$meta, _stats$meta2, _stats$meta2$package, _stats$meta3;
    if (!stats.patternName) {
      return acc;
    }
    if (!acc.get(stats.patternName)) {
      acc.set(stats.patternName, {
        streamName: stats.patternName,
        shipper: stats.shipper,
        totalNamespaces: 0,
        totalDocuments: 0,
        failureStoreDocuments: 0,
        failureStoreIndices: 0,
        totalSize: 0,
        totalIndices: 0,
        totalFields: 0,
        structureLevel: {},
        fieldsCount: {},
        managedBy: [],
        packageName: [],
        beat: []
      });
    }
    const streamStats = acc.get(stats.patternName);

    // Track unique namespaces
    if (stats.namespace) {
      uniqueNamespaces.add(stats.namespace);
    }
    streamStats.totalNamespaces = uniqueNamespaces.size;

    // Track unique fields
    stats.uniqueFields.forEach(field => uniqueFields.add(field));
    streamStats.totalFields = uniqueFields.size;

    // Aggregate structure levels
    for (const [level, count] of Object.entries(stats.structureLevel)) {
      var _streamStats$structur;
      streamStats.structureLevel[Number(level)] = ((_streamStats$structur = streamStats.structureLevel[Number(level)]) !== null && _streamStats$structur !== void 0 ? _streamStats$structur : 0) + count;
    }
    streamStats.totalDocuments += stats.totalDocuments;
    streamStats.totalIndices += stats.totalIndices;
    streamStats.failureStoreDocuments += stats.failureStoreDocuments;
    streamStats.failureStoreIndices += stats.failureStoreIndices;
    streamStats.totalSize += stats.totalSize;
    for (const [field, count] of Object.entries(stats.fieldsCount)) {
      var _streamStats$fieldsCo;
      streamStats.fieldsCount[field] = ((_streamStats$fieldsCo = streamStats.fieldsCount[field]) !== null && _streamStats$fieldsCo !== void 0 ? _streamStats$fieldsCo : 0) + count;
    }
    if ((_stats$meta = stats.meta) !== null && _stats$meta !== void 0 && _stats$meta.managed_by) {
      streamStats.managedBy.push(stats.meta.managed_by);
    }
    if ((_stats$meta2 = stats.meta) !== null && _stats$meta2 !== void 0 && (_stats$meta2$package = _stats$meta2.package) !== null && _stats$meta2$package !== void 0 && _stats$meta2$package.name) {
      streamStats.packageName.push(stats.meta.package.name);
    }
    if ((_stats$meta3 = stats.meta) !== null && _stats$meta3 !== void 0 && _stats$meta3.beat) {
      streamStats.beat.push(stats.meta.beat);
    }
    return acc;
  }, new Map());
  return Array.from(statsByStream.values());
}
function getIndexBasicStats({
  esClient,
  indices,
  breatheDelay
}) {
  const indexNames = indices.map(info => info.name);
  return (0, _rxjs.from)(esClient.indices.stats({
    index: indexNames
  })).pipe((0, _rxjs.delay)(breatheDelay), (0, _rxjs.concatMap)(allIndexStats => {
    return (0, _rxjs.from)(getFailureStoreStats({
      esClient,
      indexName: indexNames.join(',')
    })).pipe((0, _rxjs.map)(allFailureStoreStats => {
      return indices.map(info => getIndexStats(allIndexStats.indices, allFailureStoreStats, info));
    }));
  }));
}
function getIndexFieldStats({
  basicStats
}) {
  return (0, _rxjs.from)(basicStats).pipe((0, _rxjs.map)(stats => getFieldStatsAndStructureLevels(stats, _constants.DATA_TELEMETRY_FIELDS)), (0, _rxjs.toArray)());
}
function indexStatsToTelemetryEvents(stats) {
  return stats.map(stat => ({
    pattern_name: stat.streamName,
    shipper: stat.shipper,
    doc_count: stat.totalDocuments,
    structure_level: stat.structureLevel,
    index_count: stat.totalIndices,
    failure_store_doc_count: stat.failureStoreDocuments,
    failure_store_index_count: stat.failureStoreIndices,
    namespace_count: stat.totalNamespaces,
    field_count: stat.totalFields,
    field_existence: stat.fieldsCount,
    size_in_bytes: stat.totalSize,
    managed_by: Array.from(new Set(stat.managedBy)),
    package_name: Array.from(new Set(stat.packageName)),
    beat: Array.from(new Set(stat.beat))
  }));
}

/**
 * Retrieves information about data streams matching a given pattern.
 */
async function getDataStreamsInfoForPattern({
  esClient,
  pattern
}) {
  const resp = await esClient.indices.getDataStream({
    name: pattern.pattern,
    expand_wildcards: 'all'
  });
  return resp.data_streams.map(dataStream => ({
    patternName: pattern.patternName,
    shipper: pattern.shipper,
    isDataStream: true,
    name: dataStream.name,
    indices: dataStream.indices.map(index => index.index_name),
    mapping: undefined,
    meta: dataStream._meta
  }));
}
async function getIndicesInfoForPattern({
  esClient,
  pattern
}) {
  const resp = await esClient.indices.get({
    index: pattern.pattern
  });
  return Object.entries(resp).map(([index, indexInfo]) => {
    var _indexInfo$mappings;
    // This is needed to keep the format same for data streams and indices
    const indexMapping = indexInfo.mappings ? {
      [index]: {
        mappings: indexInfo.mappings
      }
    } : undefined;
    return {
      patternName: pattern.patternName,
      shipper: pattern.shipper,
      isDataStream: false,
      name: index,
      indices: [index],
      mapping: indexMapping,
      meta: (_indexInfo$mappings = indexInfo.mappings) === null || _indexInfo$mappings === void 0 ? void 0 : _indexInfo$mappings._meta
    };
  });
}

/**
 * Retrieves the namespace of index by checking the mappings of backing indices.
 *
 * @param {Object} indexInfo - The information about the index.
 * @returns {string | undefined} - The namespace of the data stream found in the mapping.
 */
function getIndexNamespace(indexInfo) {
  for (let i = 0; i < indexInfo.indices.length; i++) {
    var _indexInfo$mapping, _indexInfo$mapping$in, _indexMapping$propert, _dataStreamMapping$pr, _dataStreamMapping$pr2;
    const index = indexInfo.indices[i];
    const indexMapping = (_indexInfo$mapping = indexInfo.mapping) === null || _indexInfo$mapping === void 0 ? void 0 : (_indexInfo$mapping$in = _indexInfo$mapping[index]) === null || _indexInfo$mapping$in === void 0 ? void 0 : _indexInfo$mapping$in.mappings;
    const dataStreamMapping = indexMapping === null || indexMapping === void 0 ? void 0 : (_indexMapping$propert = indexMapping.properties) === null || _indexMapping$propert === void 0 ? void 0 : _indexMapping$propert.data_stream;
    if (!dataStreamMapping) {
      continue;
    }
    const namespace = dataStreamMapping === null || dataStreamMapping === void 0 ? void 0 : (_dataStreamMapping$pr = dataStreamMapping.properties) === null || _dataStreamMapping$pr === void 0 ? void 0 : (_dataStreamMapping$pr2 = _dataStreamMapping$pr.namespace) === null || _dataStreamMapping$pr2 === void 0 ? void 0 : _dataStreamMapping$pr2.value;
    if (namespace) {
      return namespace;
    }
  }
  return undefined;
}
async function getFailureStoreStats({
  esClient,
  indexName
}) {
  try {
    // TODO: Use the failure store API when it is available
    const resp = await esClient.transport.request({
      method: 'GET',
      path: `/${indexName}/_stats`,
      querystring: {
        failure_store: 'only'
      }
    });
    return (await resp).indices;
  } catch (e) {
    // Failure store API may not be available
    return {};
  }
}
function getIndexStats(allIndexStats, allFailureStoreStats, info) {
  let totalDocs = 0;
  let totalSize = 0;
  let totalIndices = 0;
  const indexStats = {};
  let failureStoreDocs = 0;
  let failureStoreIndices = 0;
  const failureStoreStats = {};
  Object.entries(allIndexStats !== null && allIndexStats !== void 0 ? allIndexStats : {}).forEach(([indexName, stats]) => {
    if (indexName.includes(info.name)) {
      var _stats$primaries$docs, _stats$primaries, _stats$primaries$docs2, _stats$primaries$stor, _stats$primaries2, _stats$primaries2$sto;
      totalDocs += (_stats$primaries$docs = (_stats$primaries = stats.primaries) === null || _stats$primaries === void 0 ? void 0 : (_stats$primaries$docs2 = _stats$primaries.docs) === null || _stats$primaries$docs2 === void 0 ? void 0 : _stats$primaries$docs2.count) !== null && _stats$primaries$docs !== void 0 ? _stats$primaries$docs : 0;
      totalSize += (_stats$primaries$stor = (_stats$primaries2 = stats.primaries) === null || _stats$primaries2 === void 0 ? void 0 : (_stats$primaries2$sto = _stats$primaries2.store) === null || _stats$primaries2$sto === void 0 ? void 0 : _stats$primaries2$sto.size_in_bytes) !== null && _stats$primaries$stor !== void 0 ? _stats$primaries$stor : 0;
      totalIndices++;
      indexStats[indexName] = stats;
    }
  });
  Object.entries(allFailureStoreStats !== null && allFailureStoreStats !== void 0 ? allFailureStoreStats : {}).forEach(([indexName, stats]) => {
    if (indexName.includes(info.name)) {
      var _stats$primaries$docs3, _stats$primaries3, _stats$primaries3$doc;
      failureStoreDocs += (_stats$primaries$docs3 = (_stats$primaries3 = stats.primaries) === null || _stats$primaries3 === void 0 ? void 0 : (_stats$primaries3$doc = _stats$primaries3.docs) === null || _stats$primaries3$doc === void 0 ? void 0 : _stats$primaries3$doc.count) !== null && _stats$primaries$docs3 !== void 0 ? _stats$primaries$docs3 : 0;
      failureStoreIndices++;
      failureStoreStats[indexName] = stats;
    }
  });
  return {
    patternName: info.patternName,
    shipper: info.shipper,
    namespace: info.namespace,
    totalDocuments: totalDocs,
    totalSize,
    totalIndices,
    failureStoreDocuments: failureStoreDocs,
    failureStoreIndices,
    meta: info.meta,
    mapping: info.mapping,
    indexStats,
    failureStoreStats
  };
}
function getFieldStatsAndStructureLevels(stats, fieldsToCheck) {
  var _stats$indexStats;
  const uniqueFields = new Set();
  const structureLevel = {};

  // Loop through each index and get the number of fields and gather how many documents have that field
  const resourceFieldCounts = {};
  const indexNames = Object.keys((_stats$indexStats = stats.indexStats) !== null && _stats$indexStats !== void 0 ? _stats$indexStats : {});
  for (const backingIndex of indexNames) {
    var _stats$indexStats2, _stats$mapping, _stats$mapping$backin, _indexStats$primaries, _indexStats$primaries2, _indexStats$primaries3, _structureLevel$index;
    const indexStats = (_stats$indexStats2 = stats.indexStats) === null || _stats$indexStats2 === void 0 ? void 0 : _stats$indexStats2[backingIndex];
    const indexMapping = (_stats$mapping = stats.mapping) === null || _stats$mapping === void 0 ? void 0 : (_stats$mapping$backin = _stats$mapping[backingIndex]) === null || _stats$mapping$backin === void 0 ? void 0 : _stats$mapping$backin.mappings;
    if (!indexMapping) {
      continue;
    }

    // Get all fields from the mapping
    const indexFieldsMap = getFieldPathsMapFromMapping(indexMapping);
    const indexFieldsList = Object.keys(indexFieldsMap);
    indexFieldsList.forEach(field => uniqueFields.add(field));
    const indexDocCount = (_indexStats$primaries = indexStats === null || indexStats === void 0 ? void 0 : (_indexStats$primaries2 = indexStats.primaries) === null || _indexStats$primaries2 === void 0 ? void 0 : (_indexStats$primaries3 = _indexStats$primaries2.docs) === null || _indexStats$primaries3 === void 0 ? void 0 : _indexStats$primaries3.count) !== null && _indexStats$primaries !== void 0 ? _indexStats$primaries : 0;
    if (!indexDocCount) {
      continue;
    }
    const indexStructureLevel = getStructureLevelForFieldsList(stats, indexFieldsMap);
    structureLevel[indexStructureLevel] = ((_structureLevel$index = structureLevel[indexStructureLevel]) !== null && _structureLevel$index !== void 0 ? _structureLevel$index : 0) + indexDocCount;
    for (const field of fieldsToCheck) {
      if (indexFieldsMap[field]) {
        var _resourceFieldCounts$;
        resourceFieldCounts[field] = ((_resourceFieldCounts$ = resourceFieldCounts[field]) !== null && _resourceFieldCounts$ !== void 0 ? _resourceFieldCounts$ : 0) + indexDocCount;
      }
    }
  }
  return {
    ...stats,
    uniqueFields: Array.from(uniqueFields),
    structureLevel,
    fieldsCount: resourceFieldCounts
  };
}

/**
 * Determines the structure level of log documents based on the fields present in the list.
 *
 * Structure Levels:
 * - Level 0: Unstructured data. No `@timestamp` or `timestamp` field.
 * - Level 1: Contains `@timestamp` or `timestamp` field.
 * - Level 2: Contains any of resource fields (`host.name`, `service.name`, `host`, `hostname`, `host_name`).
 * - Level 3: Contains `@timestamp`, resource fields, and `message` field.
 * - Level 4: Index name complies with a pattern of known shipper e.g. `logstash-*`, `heartbeat-*`.
 * - Level 5a: Data stream naming scheme exists (`data_stream.dataset`, `data_stream.type`, `data_stream.namespace`).
 * - Level 5b: Contains at least 3 ECS fields or `ecs.version` field.
 * - Level 6: Part of an integration, managed by a known entity.
 *
 * @param stats - Container pattern, shipper and meta info
 * @param fieldsMap - Dictionary/Map of fields present in the index with full path as key.
 * @returns {number} - The structure level of the index.
 */
function getStructureLevelForFieldsList(stats, fieldsMap) {
  var _stats$meta4, _stats$meta5;
  // Check level 1, if @timestamp or timestamp exists
  if (!fieldsMap['@timestamp'] && !fieldsMap.timestamp) {
    return 0;
  }

  // Check level 2, if resource fields exist
  if (!_constants.LEVEL_2_RESOURCE_FIELDS.some(field => fieldsMap[field])) {
    return 1;
  }

  // Check level 3, if basic structure of log message exist
  if (!fieldsMap['@timestamp'] || !fieldsMap.message || !fieldsMap['host.name'] && !fieldsMap['service.name']) {
    return 2;
  }

  // Check level 4 (Shipper is known)
  if (!stats.patternName || stats.patternName === 'generic-logs') {
    return 3;
  }

  // Check level 5a (Data stream scheme exists)
  if (!fieldsMap['data_stream.dataset'] || !fieldsMap['data_stream.type'] || !fieldsMap['data_stream.namespace']) {
    // Check level 5b (ECS fields exist)
    const fieldsList = Object.keys(fieldsMap);
    if (!fieldsMap['ecs.version'] && (0, _lodash.intersection)(_constants.PROMINENT_LOG_ECS_FIELDS, fieldsList).length < 3) {
      return 4;
    }
  }

  // Check level 6 (Index is managed)
  if (!((_stats$meta4 = stats.meta) !== null && _stats$meta4 !== void 0 && _stats$meta4.managed_by) && !((_stats$meta5 = stats.meta) !== null && _stats$meta5 !== void 0 && _stats$meta5.managed)) {
    return 5;
  }

  // All levels are fulfilled
  return 6;
}

/**
 * Recursively traverses a mapping and returns a dictionary of field paths.
 * Each key in the dictionary represents a full field path in dot notation.
 *
 * @param {MappingPropertyBase} mapping - The mapping to traverse.
 * @returns {Record<string, boolean>} - A dictionary of field paths.
 */
function getFieldPathsMapFromMapping(mapping) {
  const fieldPathsMap = {};
  function traverseMapping(nestedMapping, parentField = '') {
    for (const [fieldName, field] of Object.entries((_nestedMapping$proper = nestedMapping.properties) !== null && _nestedMapping$proper !== void 0 ? _nestedMapping$proper : {})) {
      var _nestedMapping$proper;
      const fullFieldName = parentField ? `${parentField}.${fieldName}` : fieldName;
      if (field.properties) {
        traverseMapping(field, fullFieldName);
      } else {
        fieldPathsMap[fullFieldName] = true;
      }
    }
  }
  traverseMapping(mapping);
  return fieldPathsMap;
}