"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FileWrapper = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _constants = require("../../../common/constants");
var _tika_utils = require("../../../common/utils/tika_utils");
var _utils = require("../../application/common/components/utils");
var _tika_analyzer = require("../../application/file_data_visualizer/components/file_data_visualizer_view/tika_analyzer");
var _file_manager = require("./file_manager");
var _file_size_check = require("../../application/file_data_visualizer/components/file_data_visualizer_view/file_size_check");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class FileWrapper {
  constructor(file, fileUpload) {
    (0, _defineProperty2.default)(this, "analyzedFile$", new _rxjs.BehaviorSubject({
      analysisStatus: _file_manager.STATUS.NOT_STARTED,
      fileContents: '',
      fileSize: '',
      results: null,
      explanation: undefined,
      serverSettings: null,
      loaded: false,
      importStatus: _file_manager.STATUS.NOT_STARTED,
      fileName: '',
      data: null,
      fileTooLarge: false,
      fileCouldNotBeRead: false,
      fileCouldNotBeReadPermissionError: false,
      serverError: false,
      importProgress: 0,
      docCount: 0,
      supportedFormat: true
    }));
    (0, _defineProperty2.default)(this, "fileStatus$", this.analyzedFile$.asObservable());
    (0, _defineProperty2.default)(this, "fileSizeChecker", void 0);
    this.file = file;
    this.fileUpload = fileUpload;
    this.fileSizeChecker = new _file_size_check.FileSizeChecker(fileUpload, file);
    this.analyzedFile$.next({
      ...this.analyzedFile$.getValue(),
      fileName: this.file.name,
      loaded: false,
      fileTooLarge: !this.fileSizeChecker.check(),
      fileSize: this.fileSizeChecker.fileSizeFormatted()
    });
  }
  destroy() {
    this.analyzedFile$.complete();
  }
  async analyzeFile() {
    this.setStatus({
      analysisStatus: _file_manager.STATUS.STARTED
    });
    (0, _utils.readFile)(this.file).then(async ({
      data,
      fileContents
    }) => {
      var _analysisResults$resu, _analysisResults$resu2;
      // return after file has been read
      // analysis will be done in the background

      let analysisResults;
      if ((0, _tika_utils.isTikaType)(this.file.type)) {
        analysisResults = await this.analyzeTika(data);
      } else {
        analysisResults = await this.analyzeStandardFile(fileContents, {});
      }
      const supportedFormat = (0, _constants.isSupportedFormat)((_analysisResults$resu = (_analysisResults$resu2 = analysisResults.results) === null || _analysisResults$resu2 === void 0 ? void 0 : _analysisResults$resu2.format) !== null && _analysisResults$resu !== void 0 ? _analysisResults$resu : '');
      this.setStatus({
        ...analysisResults,
        loaded: true,
        fileName: this.file.name,
        fileContents,
        data,
        supportedFormat
      });
    });
  }
  async analyzeTika(data, isRetry = false) {
    const {
      tikaResults,
      standardResults
    } = await (0, _tika_analyzer.analyzeTikaFile)(data, this.fileUpload);
    const serverSettings = (0, _utils.processResults)(standardResults);
    return {
      fileContents: tikaResults.content,
      results: standardResults.results,
      explanation: standardResults.results.explanation,
      serverSettings,
      analysisStatus: _file_manager.STATUS.COMPLETED
    };
  }
  async analyzeStandardFile(fileContents, overrides, isRetry = false) {
    try {
      const resp = await this.fileUpload.analyzeFile(fileContents, overrides);
      const serverSettings = (0, _utils.processResults)(resp);
      return {
        fileContents,
        results: resp.results,
        explanation: resp.results.explanation,
        serverSettings,
        analysisStatus: _file_manager.STATUS.COMPLETED
      };
    } catch (e) {
      return {
        fileContents,
        results: null,
        explanation: undefined,
        serverSettings: null,
        analysisError: e,
        analysisStatus: _file_manager.STATUS.FAILED
      };
    }
  }
  setStatus(status) {
    this.analyzedFile$.next({
      ...this.getStatus(),
      ...status
    });
  }
  getStatus() {
    return this.analyzedFile$.getValue();
  }
  getFileName() {
    return this.analyzedFile$.getValue().fileName;
  }
  getMappings() {
    var _this$analyzedFile$$g;
    return (_this$analyzedFile$$g = this.analyzedFile$.getValue().results) === null || _this$analyzedFile$$g === void 0 ? void 0 : _this$analyzedFile$$g.mappings;
  }
  getPipeline() {
    var _this$analyzedFile$$g2, _this$analyzedFile$$g3;
    return (_this$analyzedFile$$g2 = (_this$analyzedFile$$g3 = this.analyzedFile$.getValue().results) === null || _this$analyzedFile$$g3 === void 0 ? void 0 : _this$analyzedFile$$g3.ingest_pipeline) !== null && _this$analyzedFile$$g2 !== void 0 ? _this$analyzedFile$$g2 : {
      description: '',
      processors: []
    };
  }
  getFormat() {
    var _this$analyzedFile$$g4;
    return (_this$analyzedFile$$g4 = this.analyzedFile$.getValue().results) === null || _this$analyzedFile$$g4 === void 0 ? void 0 : _this$analyzedFile$$g4.format;
  }
  getData() {
    return this.analyzedFile$.getValue().data;
  }
  async import(id, index, mappings, pipelineId) {
    this.setStatus({
      importStatus: _file_manager.STATUS.STARTED
    });
    const format = this.analyzedFile$.getValue().results.format;
    const importer = await this.fileUpload.importerFactory(format, {
      excludeLinesPattern: this.analyzedFile$.getValue().results.exclude_lines_pattern,
      multilineStartPattern: this.analyzedFile$.getValue().results.multiline_start_pattern
    });
    importer.initializeWithoutCreate(index, mappings, this.getPipeline());
    const data = this.getData();
    if (data === null) {
      this.setStatus({
        importStatus: _file_manager.STATUS.FAILED
      });
      return;
    }
    importer.read(data);
    try {
      const resp = await importer.import(id, index, pipelineId, p => {
        this.setStatus({
          importProgress: p
        });
      });
      this.setStatus({
        docCount: resp.docCount,
        importStatus: _file_manager.STATUS.COMPLETED
      });
      return resp;
    } catch (error) {
      this.setStatus({
        importStatus: _file_manager.STATUS.FAILED
      });
      return;
    }
  }
}
exports.FileWrapper = FileWrapper;