"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.STATUS = exports.FileManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _file_wrapper = require("./file_wrapper");
var _import = require("../../application/file_data_visualizer/components/import_view/import");
var _auto_deploy = require("../../application/file_data_visualizer/components/import_view/auto_deploy");
var _merge_tools = require("./merge_tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
let STATUS = exports.STATUS = /*#__PURE__*/function (STATUS) {
  STATUS[STATUS["NA"] = 0] = "NA";
  STATUS[STATUS["NOT_STARTED"] = 1] = "NOT_STARTED";
  STATUS[STATUS["STARTED"] = 2] = "STARTED";
  STATUS[STATUS["COMPLETED"] = 3] = "COMPLETED";
  STATUS[STATUS["FAILED"] = 4] = "FAILED";
  return STATUS;
}({});
class FileManager {
  constructor(fileUpload, http, dataViewsContract, autoAddInferenceEndpointName = null, removePipelinesAfterImport = true, indexSettingsOverride = undefined) {
    (0, _defineProperty2.default)(this, "files$", new _rxjs.BehaviorSubject([]));
    (0, _defineProperty2.default)(this, "analysisValid$", new _rxjs.BehaviorSubject(false));
    (0, _defineProperty2.default)(this, "fileAnalysisStatus$", this.files$.pipe((0, _rxjs.switchMap)(files => files.length > 0 ? (0, _rxjs.combineLatest)(files.map(file => file.fileStatus$)) : (0, _rxjs.of)([]))));
    (0, _defineProperty2.default)(this, "mappingsCheckSubscription", void 0);
    (0, _defineProperty2.default)(this, "settings", void 0);
    (0, _defineProperty2.default)(this, "mappings", null);
    (0, _defineProperty2.default)(this, "pipelines", null);
    (0, _defineProperty2.default)(this, "inferenceId", null);
    (0, _defineProperty2.default)(this, "importer", null);
    (0, _defineProperty2.default)(this, "timeFieldName", null);
    (0, _defineProperty2.default)(this, "commonFileFormat", null);
    (0, _defineProperty2.default)(this, "uploadStatus$", new _rxjs.BehaviorSubject({
      analysisOk: false,
      overallImportStatus: STATUS.NOT_STARTED,
      indexCreated: STATUS.NOT_STARTED,
      pipelineCreated: STATUS.NOT_STARTED,
      modelDeployed: STATUS.NA,
      dataViewCreated: STATUS.NA,
      pipelinesDeleted: STATUS.NOT_STARTED,
      fileImport: STATUS.NOT_STARTED,
      filesStatus: [],
      fileClashes: [],
      formatMix: false,
      errors: []
    }));
    (0, _defineProperty2.default)(this, "autoAddSemanticTextField", false);
    this.fileUpload = fileUpload;
    this.http = http;
    this.dataViewsContract = dataViewsContract;
    this.autoAddInferenceEndpointName = autoAddInferenceEndpointName;
    this.removePipelinesAfterImport = removePipelinesAfterImport;
    this.autoAddSemanticTextField = this.autoAddInferenceEndpointName !== null;
    this.settings = indexSettingsOverride !== null && indexSettingsOverride !== void 0 ? indexSettingsOverride : {};
    this.mappingsCheckSubscription = this.fileAnalysisStatus$.subscribe(statuses => {
      const allFilesAnalyzed = statuses.every(status => status.loaded);
      if (allFilesAnalyzed) {
        this.analysisValid$.next(true);
        const uploadStatus = this.uploadStatus$.getValue();
        if (uploadStatus.fileImport === STATUS.STARTED) {
          return;
        }
        if (this.getFiles().length === 0) {
          this.setStatus({
            fileClashes: []
          });
          return;
        }
        const {
          formatsOk,
          fileClashes
        } = this.getFormatClashes();
        const {
          mappingClashes,
          mergedMappings
        } = this.createMergedMappings();
        const mappingsOk = mappingClashes.length === 0;
        if (formatsOk === false) {
          this.setStatus({
            fileClashes
          });
        } else if (mappingsOk) {
          this.mappings = mergedMappings;
          this.pipelines = this.getPipelines();
          this.addSemanticTextField();
          this.setStatus({
            fileClashes: []
          });
        } else {
          this.setStatus({
            fileClashes: (0, _merge_tools.getMappingClashInfo)(mappingClashes, statuses)
          });
        }
        this.setStatus({
          analysisOk: mappingsOk && formatsOk
        });
      }
    });
  }
  destroy() {
    this.files$.complete();
    this.mappingsCheckSubscription.unsubscribe();
  }
  setStatus(status) {
    this.uploadStatus$.next({
      ...this.uploadStatus$.getValue(),
      ...status
    });
  }
  async addFiles(fileList) {
    this.setStatus({
      analysisOk: false
    });
    const promises = Array.from(fileList).map(file => this.addFile(file));
    await Promise.all(promises);
  }
  async addFile(file) {
    const f = new _file_wrapper.FileWrapper(file, this.fileUpload);
    const files = this.getFiles();
    files.push(f);
    this.files$.next(files);
    await f.analyzeFile();
  }
  async removeFile(index) {
    const files = this.getFiles();
    const f = files[index];
    files.splice(index, 1);
    this.files$.next(files);
    if (f) {
      f.destroy();
    }
  }
  async removeClashingFiles() {
    const fileClashes = this.uploadStatus$.getValue().fileClashes;
    const filesToDestroy = [];
    const files = this.getFiles();
    const newFiles = files.filter((file, i) => {
      if (fileClashes[i].clash) {
        filesToDestroy.push(files[i]);
        return false;
      }
      return true;
    });
    this.files$.next(newFiles);
    filesToDestroy.forEach(file => {
      file.destroy();
    });
  }
  getFiles() {
    return this.files$.getValue();
  }
  getFormatClashes() {
    const files = this.getFiles();
    const fileClashes = (0, _merge_tools.getFormatClashes)(files);
    const formatsOk = fileClashes.every(file => file.clash === false);
    if (formatsOk) {
      this.commonFileFormat = formatsOk ? files[0].getStatus().results.format : null;
    }
    return {
      formatsOk,
      fileClashes
    };
  }
  createMergedMappings() {
    const files = this.getFiles();
    return (0, _merge_tools.createMergedMappings)(files);
  }
  getPipelines() {
    const files = this.getFiles();
    return files.map(file => file.getPipeline());
  }
  async import(indexName, createDataView = true) {
    var _this$inferenceId;
    if (this.mappings === null || this.pipelines === null || this.commonFileFormat === null) {
      this.setStatus({
        overallImportStatus: STATUS.FAILED
      });
      return null;
    }
    this.setStatus({
      overallImportStatus: STATUS.STARTED
    });
    this.importer = await this.fileUpload.importerFactory(this.commonFileFormat, {});
    this.inferenceId = (0, _import.getInferenceId)(this.mappings);
    if (this.inferenceId !== null) {
      this.setStatus({
        modelDeployed: STATUS.NOT_STARTED
      });
      this.setStatus({
        modelDeployed: STATUS.STARTED
      });
      await this.autoDeploy();
      this.setStatus({
        modelDeployed: STATUS.COMPLETED
      });
    }
    this.setStatus({
      indexCreated: STATUS.STARTED,
      pipelineCreated: STATUS.STARTED
    });
    let indexCreated = false;
    let pipelineCreated = false;
    let initializeImportResp;
    try {
      initializeImportResp = await this.importer.initializeImport(indexName, this.settings, this.mappings, this.pipelines[0], this.pipelines);
      this.timeFieldName = this.importer.getTimeField();
      indexCreated = initializeImportResp.index !== undefined;
      pipelineCreated = initializeImportResp.pipelineId !== undefined;
      this.setStatus({
        indexCreated: indexCreated ? STATUS.COMPLETED : STATUS.FAILED,
        pipelineCreated: pipelineCreated ? STATUS.COMPLETED : STATUS.FAILED
      });
      if (initializeImportResp.error) {
        throw initializeImportResp.error;
      }
    } catch (e) {
      this.setStatus({
        overallImportStatus: STATUS.FAILED,
        errors: [{
          title: _i18n.i18n.translate('xpack.dataVisualizer.file.fileManager.errorInitializing', {
            defaultMessage: 'Error initializing index and ingest pipeline'
          }),
          error: e
        }]
      });
      return null;
    }
    if (!indexCreated || !pipelineCreated || !initializeImportResp) {
      return null;
    }
    this.setStatus({
      fileImport: STATUS.STARTED
    });

    // import data
    const files = this.getFiles();
    try {
      await Promise.all(files.map(async (file, i) => {
        await file.import(initializeImportResp.id, indexName, this.mappings, `${indexName}-${i}-pipeline`);
      }));
    } catch (error) {
      this.setStatus({
        overallImportStatus: STATUS.FAILED,
        errors: [{
          title: _i18n.i18n.translate('xpack.dataVisualizer.file.fileManager.errorImportingData', {
            defaultMessage: 'Error importing data'
          }),
          error
        }]
      });
      return null;
    }
    this.setStatus({
      fileImport: STATUS.COMPLETED
    });
    if (this.removePipelinesAfterImport) {
      try {
        this.setStatus({
          pipelinesDeleted: STATUS.STARTED
        });
        await this.importer.deletePipelines(this.pipelines.map((p, i) => `${indexName}-${i}-pipeline`));
        this.setStatus({
          pipelinesDeleted: STATUS.COMPLETED
        });
      } catch (error) {
        this.setStatus({
          pipelinesDeleted: STATUS.FAILED,
          errors: [{
            title: _i18n.i18n.translate('xpack.dataVisualizer.file.fileManager.errorDeletingPipelines', {
              defaultMessage: 'Error deleting pipelines'
            }),
            error
          }]
        });
      }
    }
    const dataView = '';
    let dataViewResp;
    if (createDataView) {
      var _this$timeFieldName;
      this.setStatus({
        dataViewCreated: STATUS.STARTED
      });
      const dataViewName = dataView === '' ? indexName : dataView;
      dataViewResp = await (0, _import.createKibanaDataView)(dataViewName, this.dataViewsContract, (_this$timeFieldName = this.timeFieldName) !== null && _this$timeFieldName !== void 0 ? _this$timeFieldName : undefined);
      if (dataViewResp.success === false) {
        this.setStatus({
          overallImportStatus: STATUS.FAILED,
          errors: [{
            title: _i18n.i18n.translate('xpack.dataVisualizer.file.fileManager.errorCreatingDataView', {
              defaultMessage: 'Error creating data view'
            }),
            error: dataViewResp.error
          }]
        });
        return null;
      } else {
        this.setStatus({
          dataViewCreated: STATUS.COMPLETED
        });
      }
    }
    this.setStatus({
      overallImportStatus: STATUS.COMPLETED
    });
    return {
      index: indexName,
      dataView: dataViewResp ? {
        id: dataViewResp.id,
        title: dataView
      } : undefined,
      inferenceId: (_this$inferenceId = this.inferenceId) !== null && _this$inferenceId !== void 0 ? _this$inferenceId : undefined,
      files: this.getFiles().map(file => {
        const status = file.getStatus();
        return {
          fileName: status.fileName,
          docCount: status.docCount,
          fileFormat: status.results.format,
          documentType: status.results.document_type
        };
      })
    };
  }
  async autoDeploy() {
    if (this.inferenceId === null) {
      return;
    }
    try {
      const autoDeploy = new _auto_deploy.AutoDeploy(this.http, this.inferenceId);
      await autoDeploy.deploy();
    } catch (error) {
      this.setStatus({
        modelDeployed: STATUS.FAILED,
        errors: [{
          title: _i18n.i18n.translate('xpack.dataVisualizer.file.fileManager.errorDeployingModel', {
            defaultMessage: 'Error deploying model'
          }),
          error
        }]
      });
    }
  }
  isTikaFormat() {
    return this.commonFileFormat === 'tika';
  }
  addSemanticTextField() {
    if (this.isTikaFormat() && this.autoAddSemanticTextField && this.autoAddInferenceEndpointName !== null && this.pipelines !== null && this.mappings !== null) {
      this.mappings.properties.content = {
        type: 'semantic_text',
        inference_id: this.autoAddInferenceEndpointName
      };
      this.pipelines.forEach(pipeline => {
        pipeline.processors.push({
          set: {
            field: 'content',
            copy_from: 'attachment.content'
          }
        });
      });
    }
  }
}
exports.FileManager = FileManager;