"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeUnsavedChangesManager = initializeUnsavedChangesManager;
var _presentationContainers = require("@kbn/presentation-containers");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _dashboard_backup_service = require("../services/dashboard_backup_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function initializeUnsavedChangesManager({
  creationOptions,
  controlGroupApi$,
  lastSavedState,
  panelsManager,
  savedObjectId$,
  settingsManager,
  viewModeManager,
  unifiedSearchManager,
  referencesComparator
}) {
  const hasUnsavedChanges$ = new _rxjs.BehaviorSubject(false);
  const lastSavedState$ = new _rxjs.BehaviorSubject(lastSavedState);
  const saveNotification$ = new _rxjs.Subject();
  const dashboardUnsavedChanges = (0, _presentationContainers.initializeUnsavedChanges)(lastSavedState, {
    saveNotification$
  }, {
    ...panelsManager.comparators,
    ...settingsManager.comparators,
    ...viewModeManager.comparators,
    ...unifiedSearchManager.comparators,
    ...referencesComparator
  });
  const unsavedChangesSubscription = (0, _rxjs.combineLatest)([dashboardUnsavedChanges.api.unsavedChanges$, (0, _presentationContainers.childrenUnsavedChanges$)(panelsManager.api.children$), controlGroupApi$.pipe((0, _rxjs.skipWhile)(controlGroupApi => !controlGroupApi), (0, _rxjs.switchMap)(controlGroupApi => {
    return controlGroupApi.unsavedChanges$;
  }))]).pipe((0, _rxjs.debounceTime)(0)).subscribe(([dashboardChanges, unsavedPanelState, controlGroupChanges]) => {
    /**
     * viewMode needs to be stored in session state because its used to exclude 'view' dashboards on the listing page
     * However, viewMode differences should not trigger unsaved changes notification otherwise, opening a dashboard in
     * edit mode will always show unsaved changes. Similarly, differences in references are derived from panels, so
     * we don't consider them unsaved changes
     */
    const hasDashboardChanges = Object.keys((0, _lodash.omit)(dashboardChanges !== null && dashboardChanges !== void 0 ? dashboardChanges : {}, ['viewMode', 'references'])).length > 0;
    const hasUnsavedChanges = hasDashboardChanges || unsavedPanelState !== undefined || controlGroupChanges !== undefined;
    if (hasUnsavedChanges !== hasUnsavedChanges$.value) {
      hasUnsavedChanges$.next(hasUnsavedChanges);
    }

    // backup unsaved changes if configured to do so
    if (creationOptions !== null && creationOptions !== void 0 && creationOptions.useSessionStorageIntegration) {
      // Current behaviour expects time range not to be backed up. Revisit this?
      const dashboardStateToBackup = (0, _lodash.omit)(dashboardChanges !== null && dashboardChanges !== void 0 ? dashboardChanges : {}, ['timeRange', 'refreshInterval']);
      const reactEmbeddableChanges = unsavedPanelState ? {
        ...unsavedPanelState
      } : {};
      if (controlGroupChanges) {
        reactEmbeddableChanges[_dashboard_backup_service.PANELS_CONTROL_GROUP_KEY] = controlGroupChanges;
      }
      (0, _dashboard_backup_service.getDashboardBackupService)().setState(savedObjectId$.value, dashboardStateToBackup, reactEmbeddableChanges);
    }
  });
  return {
    api: {
      asyncResetToLastSavedState: async () => {
        var _controlGroupApi$$val;
        panelsManager.internalApi.reset(lastSavedState$.value);
        settingsManager.internalApi.reset(lastSavedState$.value);
        unifiedSearchManager.internalApi.reset(lastSavedState$.value);
        await ((_controlGroupApi$$val = controlGroupApi$.value) === null || _controlGroupApi$$val === void 0 ? void 0 : _controlGroupApi$$val.asyncResetUnsavedChanges());
      },
      hasUnsavedChanges$,
      saveNotification$
    },
    cleanup: () => {
      dashboardUnsavedChanges.cleanup();
      unsavedChangesSubscription.unsubscribe();
    },
    internalApi: {
      getLastSavedState: () => lastSavedState$.value,
      onSave: savedState => {
        lastSavedState$.next(savedState);
        // sync panels manager with latest saved state
        panelsManager.internalApi.setPanels(savedState.panels);
        saveNotification$.next();
      }
    }
  };
}