"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConfigFromFiles = void 0;
var _fs = require("fs");
var _jsYaml = require("js-yaml");
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _lodash = require("lodash");
var _std = require("@kbn/std");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const readYaml = path => (0, _jsYaml.safeLoad)((0, _fs.readFileSync)(path, 'utf8'));
const listEntries = root => {
  const entries = [];
  const recursiveListEntries = (currentLevel, currentPath) => {
    if ((0, _lodash.isPlainObject)(currentLevel) && Object.keys(currentLevel).length > 0) {
      for (const [subKey, subVal] of Object.entries(currentLevel)) {
        const subKeySplits = (0, _utils.splitKey)(subKey);
        recursiveListEntries(subVal, [...currentPath, ...subKeySplits]);
      }
    } else if (currentPath.length) {
      entries.push({
        path: currentPath,
        value: processEntryValue(currentLevel)
      });
    }
  };
  recursiveListEntries(root, []);
  return entries;
};
const mergeEntries = entries => {
  const root = {};
  entries.forEach(entry => {
    (0, _saferLodashSet.set)(root, entry.path, entry.value);
  });
  return root;
};
function processEntryValue(value) {
  if ((0, _lodash.isPlainObject)(value) || Array.isArray(value)) {
    for (const [subKey, subVal] of Object.entries(value)) {
      const unsplitKey = (0, _utils.getUnsplittableKey)(subKey);
      if (unsplitKey) {
        delete value[subKey];
        (0, _saferLodashSet.set)(value, [unsplitKey], processEntryValue(subVal));
      } else {
        const subKeySplits = (0, _utils.splitKey)(subKey);
        if (subKeySplits.length > 1) delete value[subKey];
        (0, _saferLodashSet.set)(value, subKeySplits, processEntryValue(subVal));
      }
    }
  } else if (typeof value === 'string') {
    return (0, _utils.replaceEnvVarRefs)(value);
  }
  return value;
}
const getConfigFromFiles = configFiles => {
  const yamlEntries = [];
  for (const configFile of configFiles) {
    const yaml = readYaml(configFile);
    if (yaml !== null) {
      yamlEntries.push(...listEntries(yaml));
    }
  }
  for (const entry of yamlEntries) {
    (0, _std.ensureValidObjectPath)(entry.path);
  }
  return mergeEntries(yamlEntries);
};
exports.getConfigFromFiles = getConfigFromFiles;