"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AdHocTaskRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _elasticApmNode = _interopRequireDefault(require("elastic-apm-node"));
var _uuid = require("uuid");
var _server = require("@kbn/core/server");
var _server2 = require("@kbn/task-manager-plugin/server");
var _common = require("@kbn/event-log-plugin/common");
var _task = require("@kbn/task-manager-plugin/server/task");
var _constants = require("../../common/constants");
var _get_executor_services = require("./get_executor_services");
var _lib = require("../lib");
var _types = require("../types");
var _task_runner_timer = require("./task_runner_timer");
var _saved_objects = require("../saved_objects");
var _rule_monitoring_service = require("../monitoring/rule_monitoring_service");
var _ad_hoc_task_running_handler = require("./ad_hoc_task_running_handler");
var _rule_loader = require("./rule_loader");
var _rule_result_service = require("../monitoring/rule_result_service");
var _rule_type_runner = require("./rule_type_runner");
var _alerts_client = require("../alerts_client");
var _lib2 = require("./lib");
var _alerting_event_logger = require("../lib/alerting_event_logger/alerting_event_logger");
var _rule_run_metrics_store = require("../lib/rule_run_metrics_store");
var _errors = require("../lib/errors");
var _result_type = require("../lib/result_type");
var _update_gaps = require("../lib/rule_gaps/update/update_gaps");
var _action_scheduler = require("./action_scheduler");
var _transform_ad_hoc_run_to_backfill_result = require("../application/backfill/transforms/transform_ad_hoc_run_to_backfill_result");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AdHocTaskRunner {
  constructor({
    context,
    internalSavedObjectsRepository,
    taskInstance
  }) {
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "executionId", void 0);
    (0, _defineProperty2.default)(this, "internalSavedObjectsRepository", void 0);
    (0, _defineProperty2.default)(this, "ruleTypeRegistry", void 0);
    (0, _defineProperty2.default)(this, "taskInstance", void 0);
    (0, _defineProperty2.default)(this, "adHocRunSchedule", []);
    (0, _defineProperty2.default)(this, "adHocRange", null);
    (0, _defineProperty2.default)(this, "alertingEventLogger", void 0);
    (0, _defineProperty2.default)(this, "cancelled", false);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "ruleId", '');
    (0, _defineProperty2.default)(this, "ruleMonitoring", void 0);
    (0, _defineProperty2.default)(this, "ruleResult", void 0);
    (0, _defineProperty2.default)(this, "ruleTypeId", '');
    (0, _defineProperty2.default)(this, "ruleTypeRunner", void 0);
    (0, _defineProperty2.default)(this, "runDate", new Date());
    (0, _defineProperty2.default)(this, "scheduleToRunIndex", -1);
    (0, _defineProperty2.default)(this, "searchAbortController", void 0);
    (0, _defineProperty2.default)(this, "shouldDeleteTask", false);
    (0, _defineProperty2.default)(this, "stackTraceLog", null);
    (0, _defineProperty2.default)(this, "taskRunning", void 0);
    (0, _defineProperty2.default)(this, "timer", void 0);
    (0, _defineProperty2.default)(this, "apiKeyToUse", null);
    this.context = context;
    this.executionId = (0, _uuid.v4)();
    this.internalSavedObjectsRepository = internalSavedObjectsRepository;
    this.ruleTypeRegistry = context.ruleTypeRegistry;
    this.taskInstance = taskInstance;
    this.alertingEventLogger = new _alerting_event_logger.AlertingEventLogger(this.context.eventLogger);
    this.logger = context.logger.get(`ad_hoc_run`);
    this.ruleMonitoring = new _rule_monitoring_service.RuleMonitoringService();
    this.ruleResult = new _rule_result_service.RuleResultService();
    this.timer = new _task_runner_timer.TaskRunnerTimer({
      logger: this.logger
    });
    this.ruleTypeRunner = new _rule_type_runner.RuleTypeRunner({
      context: this.context,
      task: this.taskInstance,
      timer: this.timer
    });
    this.searchAbortController = new AbortController();
    this.taskRunning = new _ad_hoc_task_running_handler.AdHocTaskRunningHandler(this.internalSavedObjectsRepository, this.logger);
  }
  async updateAdHocRunSavedObjectPostRun(adHocRunParamsId, namespace, {
    status,
    schedule
  }) {
    try {
      // Checking to see if the update performed at the beginning
      // of the run is complete. Swallowing the error because we still
      // want to move forward with the update post-run
      await this.taskRunning.waitFor();
      // eslint-disable-next-line no-empty
    } catch {}
    try {
      await (0, _lib2.partiallyUpdateAdHocRun)(this.internalSavedObjectsRepository, adHocRunParamsId, {
        ...(status ? {
          status
        } : {}),
        ...(schedule ? {
          schedule
        } : {})
      }, {
        ignore404: true,
        namespace,
        refresh: false
      });
    } catch (err) {
      this.logger.error(`error updating ad hoc run ${adHocRunParamsId} ${err.message}`);
    }
  }
  async runRule({
    adHocRunData,
    fakeRequest,
    scheduleToRun,
    validatedParams: params
  }) {
    const ruleRunMetricsStore = new _rule_run_metrics_store.RuleRunMetricsStore();
    if (scheduleToRun == null) {
      return ruleRunMetricsStore.getMetrics();
    }
    const {
      rule,
      apiKeyToUse,
      apiKeyId
    } = adHocRunData;
    const ruleType = this.ruleTypeRegistry.get(rule.alertTypeId);
    const ruleLabel = `${ruleType.id}:${rule.id}: '${rule.name}'`;
    const ruleTypeRunnerContext = {
      alertingEventLogger: this.alertingEventLogger,
      namespace: this.context.spaceIdToNamespace(adHocRunData.spaceId),
      logger: this.logger,
      request: fakeRequest,
      ruleId: rule.id,
      ruleLogPrefix: ruleLabel,
      ruleRunMetricsStore,
      spaceId: adHocRunData.spaceId,
      isServerless: this.context.isServerless
    };
    const alertsClient = await (0, _alerts_client.initializeAlertsClient)({
      alertsService: this.context.alertsService,
      context: ruleTypeRunnerContext,
      executionId: this.executionId,
      logger: this.logger,
      maxAlerts: this.context.maxAlerts,
      rule: {
        id: rule.id,
        name: rule.name,
        tags: rule.tags,
        consumer: rule.consumer,
        revision: rule.revision,
        params: rule.params
      },
      ruleType,
      runTimestamp: this.runDate,
      startedAt: new Date(scheduleToRun.runAt),
      taskInstance: this.taskInstance
    });
    const executorServices = (0, _get_executor_services.getExecutorServices)({
      context: this.context,
      fakeRequest,
      abortController: this.searchAbortController,
      logger: this.logger,
      ruleMonitoringService: this.ruleMonitoring,
      ruleResultService: this.ruleResult,
      ruleData: {
        name: rule.name,
        alertTypeId: rule.alertTypeId,
        id: rule.id,
        spaceId: adHocRunData.spaceId
      },
      ruleTaskTimeout: ruleType.ruleTaskTimeout
    });
    const {
      error,
      stackTrace
    } = await this.ruleTypeRunner.run({
      context: ruleTypeRunnerContext,
      alertsClient,
      executionId: this.executionId,
      executorServices,
      rule: {
        ...rule,
        actions: [],
        muteAll: false,
        createdAt: new Date(rule.createdAt),
        updatedAt: new Date(rule.updatedAt)
      },
      ruleType,
      startedAt: new Date(scheduleToRun.runAt),
      state: this.taskInstance.state,
      validatedParams: params
    });

    // if there was an error, save the stack trace and throw
    if (error) {
      this.stackTraceLog = stackTrace !== null && stackTrace !== void 0 ? stackTrace : null;
      throw error;
    }
    const actionScheduler = new _action_scheduler.ActionScheduler({
      rule: {
        ...rule,
        muteAll: false,
        mutedInstanceIds: [],
        createdAt: new Date(rule.createdAt),
        updatedAt: new Date(rule.updatedAt)
      },
      ruleType,
      logger: this.logger,
      taskRunnerContext: this.context,
      taskInstance: this.taskInstance,
      ruleRunMetricsStore,
      apiKey: apiKeyToUse,
      apiKeyId,
      ruleConsumer: rule.consumer,
      executionId: this.executionId,
      ruleLabel,
      previousStartedAt: null,
      alertingEventLogger: this.alertingEventLogger,
      actionsClient: await this.context.actionsPlugin.getActionsClientWithRequest(fakeRequest),
      alertsClient,
      priority: _task.TaskPriority.Low
    });
    await actionScheduler.run({
      activeCurrentAlerts: alertsClient.getProcessedAlerts('activeCurrent'),
      recoveredCurrentAlerts: alertsClient.getProcessedAlerts('recoveredCurrent')
    });
    return ruleRunMetricsStore.getMetrics();
  }

  /**
   * Before we actually kick off the ad hoc run:
   * - read decrypted ad hoc run SO
   * - start the RunningHandler
   * - initialize the event logger
   * - set the current APM transaction info
   * - validate that rule type is enabled and params are valid
   */
  async prepareToRun() {
    this.runDate = new Date();
    return await this.timer.runWithTimer(_task_runner_timer.TaskRunnerTimerSpan.PrepareRule, async () => {
      var _this$adHocRunSchedul;
      const {
        params: {
          adHocRunParamsId,
          spaceId
        },
        startedAt
      } = this.taskInstance;
      const namespace = this.context.spaceIdToNamespace(spaceId);
      this.alertingEventLogger.initialize({
        context: {
          savedObjectId: adHocRunParamsId,
          savedObjectType: _saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE,
          spaceId,
          executionId: this.executionId,
          taskScheduledAt: this.taskInstance.scheduledAt,
          ...(namespace ? {
            namespace
          } : {})
        },
        runDate: this.runDate,
        // in the future we might want different types of ad hoc runs (like preview)
        type: _alerting_event_logger.executionType.BACKFILL
      });
      let adHocRunData;
      try {
        const adHocRunSO = await this.context.encryptedSavedObjectsClient.getDecryptedAsInternalUser(_saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE, adHocRunParamsId, {
          namespace
        });
        adHocRunData = (0, _transform_ad_hoc_run_to_backfill_result.transformAdHocRunToAdHocRunData)({
          adHocRunSO,
          isSystemAction: connectorId => this.context.actionsPlugin.isSystemActionConnector(connectorId),
          omitGeneratedActionValues: false
        });
      } catch (err) {
        const errorSource = _server.SavedObjectsErrorHelpers.isNotFoundError(err) ? _server2.TaskErrorSource.USER : _server2.TaskErrorSource.FRAMEWORK;
        throw (0, _server2.createTaskRunError)(new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.Decrypt, err), errorSource);
      }
      const {
        rule,
        apiKeyToUse,
        schedule,
        start,
        end
      } = adHocRunData;
      this.apiKeyToUse = apiKeyToUse;
      let ruleType;
      try {
        ruleType = this.ruleTypeRegistry.get(rule.alertTypeId);
      } catch (err) {
        throw (0, _server2.createTaskRunError)(new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.Read, err), _server2.TaskErrorSource.FRAMEWORK);
      }
      this.ruleTypeId = ruleType.id;
      this.ruleId = rule.id;
      this.alertingEventLogger.addOrUpdateRuleData({
        id: rule.id,
        type: ruleType,
        name: rule.name,
        consumer: rule.consumer,
        revision: rule.revision
      });
      try {
        this.ruleTypeRegistry.ensureRuleTypeEnabled(rule.alertTypeId);
      } catch (err) {
        throw (0, _server2.createTaskRunError)(new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.License, err), _server2.TaskErrorSource.USER);
      }
      let validatedParams;
      try {
        validatedParams = (0, _lib.validateRuleTypeParams)(rule.params, ruleType.validate.params);
      } catch (err) {
        throw (0, _server2.createTaskRunError)(new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.Validate, err), _server2.TaskErrorSource.USER);
      }
      if (_elasticApmNode.default.currentTransaction) {
        _elasticApmNode.default.currentTransaction.name = `Execute Backfill for Alerting Rule`;
        _elasticApmNode.default.currentTransaction.addLabels({
          alerting_rule_space_id: spaceId,
          alerting_rule_id: rule.id,
          alerting_rule_consumer: rule.consumer,
          alerting_rule_name: rule.name,
          alerting_rule_tags: rule.tags.join(', '),
          alerting_rule_type_id: rule.alertTypeId,
          alerting_rule_params: JSON.stringify(rule.params)
        });
      }
      if (startedAt) {
        // Capture how long it took for the task to start running after being claimed
        this.timer.setDuration(_task_runner_timer.TaskRunnerTimerSpan.StartTaskRun, startedAt);
      }

      // Determine which schedule entry we're going to run
      // Find the first index where the status is pending
      this.adHocRunSchedule = schedule;
      this.adHocRange = {
        start,
        end
      };
      this.scheduleToRunIndex = ((_this$adHocRunSchedul = this.adHocRunSchedule) !== null && _this$adHocRunSchedul !== void 0 ? _this$adHocRunSchedul : []).findIndex(s => s.status === _constants.adHocRunStatus.PENDING);
      if (this.scheduleToRunIndex > -1) {
        this.logger.debug(`Executing ad hoc run for rule ${ruleType.id}:${rule.id} for runAt ${this.adHocRunSchedule[this.scheduleToRunIndex].runAt}`);
        this.adHocRunSchedule[this.scheduleToRunIndex].status = _constants.adHocRunStatus.RUNNING;
        this.taskRunning.start(adHocRunParamsId, schedule, this.context.spaceIdToNamespace(spaceId));
      }

      // Generate fake request with API key
      const fakeRequest = (0, _rule_loader.getFakeKibanaRequest)(this.context, spaceId, apiKeyToUse);
      return {
        adHocRunData,
        fakeRequest,
        scheduleToRun: this.scheduleToRunIndex > -1 ? this.adHocRunSchedule[this.scheduleToRunIndex] : null,
        validatedParams
      };
    });
  }
  async processAdHocRunResults(ruleRunMetrics) {
    const {
      params: {
        adHocRunParamsId,
        spaceId
      },
      startedAt
    } = this.taskInstance;
    const namespace = this.context.spaceIdToNamespace(spaceId);
    const {
      executionStatus: execStatus,
      executionMetrics: execMetrics
    } = await this.timer.runWithTimer(_task_runner_timer.TaskRunnerTimerSpan.ProcessRuleRun, async () => {
      var _executionStatus$erro, _executionStatus$erro2, _executionStatus$erro3, _executionStatus$erro4;
      const {
        executionStatus,
        executionMetrics,
        outcome
      } = (0, _lib2.processRunResults)({
        result: this.ruleResult,
        runDate: this.runDate,
        runResultWithMetrics: ruleRunMetrics
      });
      if (!(0, _result_type.isOk)(ruleRunMetrics)) {
        const error = this.stackTraceLog ? this.stackTraceLog.message : ruleRunMetrics.error;
        const stack = this.stackTraceLog ? this.stackTraceLog.stackTrace : ruleRunMetrics.error.stack;
        const message = `Executing ad hoc run with id "${adHocRunParamsId}" has resulted in Error: ${(0, _errors.getEsErrorMessage)(error)} - ${stack !== null && stack !== void 0 ? stack : ''}`;
        const tags = [adHocRunParamsId, 'rule-ad-hoc-run-failed'];
        if (this.ruleTypeId.length > 0) {
          tags.push(this.ruleTypeId);
        }
        if (this.ruleId.length > 0) {
          tags.push(this.ruleId);
        }
        this.logger.error(message, {
          tags,
          error: {
            stack_trace: stack
          }
        });
      }
      if (_elasticApmNode.default.currentTransaction) {
        _elasticApmNode.default.currentTransaction.setOutcome(outcome);
      }

      // set start and duration based on event log
      const {
        start,
        duration
      } = this.alertingEventLogger.getStartAndDuration();
      if (null != start) {
        executionStatus.lastExecutionDate = start;
      }
      if (null != duration) {
        executionStatus.lastDuration = (0, _common.nanosToMillis)(duration);
      }
      if (this.scheduleToRunIndex > -1) {
        let updatedStatus = _constants.adHocRunStatus.COMPLETE;
        if (this.cancelled) {
          updatedStatus = _constants.adHocRunStatus.TIMEOUT;
        } else if (outcome === 'failure') {
          updatedStatus = _constants.adHocRunStatus.ERROR;
        }
        this.adHocRunSchedule[this.scheduleToRunIndex].status = updatedStatus;
      }

      // If execution failed due to decrypt error, we should stop running the task
      // If the user wants to rerun it, they can reschedule
      // In the future, we can consider saving the task in an error state when we
      // have one or both of the following abilities
      // - ability to rerun a failed ad hoc run
      // - ability to clean up failed ad hoc runs (either manually or automatically)
      this.shouldDeleteTask = executionStatus.status === 'error' && ((executionStatus === null || executionStatus === void 0 ? void 0 : (_executionStatus$erro = executionStatus.error) === null || _executionStatus$erro === void 0 ? void 0 : _executionStatus$erro.reason) === _types.RuleExecutionStatusErrorReasons.Decrypt || (executionStatus === null || executionStatus === void 0 ? void 0 : (_executionStatus$erro2 = executionStatus.error) === null || _executionStatus$erro2 === void 0 ? void 0 : _executionStatus$erro2.reason) === _types.RuleExecutionStatusErrorReasons.Read || (executionStatus === null || executionStatus === void 0 ? void 0 : (_executionStatus$erro3 = executionStatus.error) === null || _executionStatus$erro3 === void 0 ? void 0 : _executionStatus$erro3.reason) === _types.RuleExecutionStatusErrorReasons.License || (executionStatus === null || executionStatus === void 0 ? void 0 : (_executionStatus$erro4 = executionStatus.error) === null || _executionStatus$erro4 === void 0 ? void 0 : _executionStatus$erro4.reason) === _types.RuleExecutionStatusErrorReasons.Validate);
      await this.updateAdHocRunSavedObjectPostRun(adHocRunParamsId, namespace, {
        ...(this.shouldDeleteTask ? {
          status: _constants.adHocRunStatus.ERROR
        } : {}),
        ...(this.scheduleToRunIndex > -1 ? {
          schedule: this.adHocRunSchedule
        } : {})
      });
      if (startedAt) {
        // Capture how long it took for the rule to run after being claimed
        this.timer.setDuration(_task_runner_timer.TaskRunnerTimerSpan.TotalRunDuration, startedAt);
      }
      return {
        executionStatus,
        executionMetrics
      };
    });
    this.alertingEventLogger.done({
      status: execStatus,
      metrics: execMetrics,
      // in the future if we have other types of ad hoc runs (like preview)
      // we can differentiate and pass in different info
      backfill: {
        id: adHocRunParamsId,
        start: this.scheduleToRunIndex > -1 ? this.adHocRunSchedule[this.scheduleToRunIndex].runAt : undefined,
        interval: this.scheduleToRunIndex > -1 ? this.adHocRunSchedule[this.scheduleToRunIndex].interval : undefined
      },
      timings: this.timer.toJson()
    });
  }
  hasAnyPendingRuns() {
    let hasPendingRuns = false;
    const anyPendingRuns = this.adHocRunSchedule.findIndex(s => s.status === _constants.adHocRunStatus.PENDING);
    if (anyPendingRuns > -1) {
      hasPendingRuns = true;
    }
    return hasPendingRuns;
  }
  async run() {
    let runMetrics;
    try {
      const runParams = await this.prepareToRun();
      runMetrics = (0, _result_type.asOk)({
        metrics: await this.runRule(runParams)
      });
    } catch (err) {
      runMetrics = (0, _result_type.asErr)(err);
    }
    await this.processAdHocRunResults(runMetrics);
    this.shouldDeleteTask = this.shouldDeleteTask || !this.hasAnyPendingRuns();
    return {
      state: {},
      ...(this.shouldDeleteTask ? {} : {
        runAt: new Date()
      })
    };
  }
  async cancel() {
    if (this.cancelled) {
      return;
    }
    this.cancelled = true;
    this.searchAbortController.abort();
    this.ruleTypeRunner.cancelRun();

    // Write event log entry
    const {
      params: {
        adHocRunParamsId
      },
      timeoutOverride
    } = this.taskInstance;
    this.logger.debug(`Cancelling execution for ad hoc run with id ${adHocRunParamsId} for rule type ${this.ruleTypeId} with id ${this.ruleId} - execution exceeded rule type timeout of ${timeoutOverride}`);
    this.logger.debug(`Aborting any in-progress ES searches for rule type ${this.ruleTypeId} with id ${this.ruleId}`);
    this.alertingEventLogger.logTimeout({
      backfill: {
        id: adHocRunParamsId,
        start: this.scheduleToRunIndex > -1 ? this.adHocRunSchedule[this.scheduleToRunIndex].runAt : undefined,
        interval: this.scheduleToRunIndex > -1 ? this.adHocRunSchedule[this.scheduleToRunIndex].interval : undefined
      }
    });
    this.shouldDeleteTask = !this.hasAnyPendingRuns();

    // cleanup function is not called for timed out tasks
    await this.cleanup();
  }
  async cleanup() {
    if (!this.shouldDeleteTask) return;
    try {
      await this.internalSavedObjectsRepository.delete(_saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE, this.taskInstance.params.adHocRunParamsId, {
        refresh: true,
        namespace: this.context.spaceIdToNamespace(this.taskInstance.params.spaceId)
      });
      await this.updateGapsAfterBackfillComplete();
    } catch (e) {
      // Log error only, we shouldn't fail the task because of an error here (if ever there's retry logic)
      this.logger.error(`Failed to cleanup ${_saved_objects.AD_HOC_RUN_SAVED_OBJECT_TYPE} object [id="${this.taskInstance.params.adHocRunParamsId}"]: ${e.message}`);
    }
  }
  async updateGapsAfterBackfillComplete() {
    if (!this.shouldDeleteTask) return;
    if (this.scheduleToRunIndex < 0 || !this.adHocRange) return null;
    const fakeRequest = (0, _rule_loader.getFakeKibanaRequest)(this.context, this.taskInstance.params.spaceId, this.apiKeyToUse);
    const eventLogClient = await this.context.getEventLogClient(fakeRequest);
    const actionsClient = await this.context.actionsPlugin.getActionsClientWithRequest(fakeRequest);
    return (0, _update_gaps.updateGaps)({
      ruleId: this.ruleId,
      start: new Date(this.adHocRange.start),
      end: this.adHocRange.end ? new Date(this.adHocRange.end) : new Date(),
      eventLogger: this.context.eventLogger,
      eventLogClient,
      logger: this.logger,
      backfillSchedule: this.adHocRunSchedule,
      savedObjectsRepository: this.internalSavedObjectsRepository,
      backfillClient: this.context.backfillClient,
      actionsClient
    });
  }
}
exports.AdHocTaskRunner = AdHocTaskRunner;