"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.shouldTrackGenAiToken = exports.getGenAiTokenTracking = void 0;
var _stream = require("stream");
var _get_token_count_from_bedrock_converse = require("./get_token_count_from_bedrock_converse");
var _get_token_count_from_invoke_async_iterator = require("./get_token_count_from_invoke_async_iterator");
var _get_token_count_from_bedrock_invoke = require("./get_token_count_from_bedrock_invoke");
var _get_token_count_from_openai_stream = require("./get_token_count_from_openai_stream");
var _get_token_count_from_invoke_stream = require("./get_token_count_from_invoke_stream");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
 * Calculates the total, prompt, and completion token counts from different types of responses.
 * It handles both streamed and non-streamed responses from OpenAI and Bedrock.
 * It returns null if it cannot calculate the token counts.
 * @param actionTypeId the action type id
 * @param logger the logger
 * @param result the result from the action executor
 * @param validatedParams the validated params from the action executor
 */
const getGenAiTokenTracking = async ({
  actionTypeId,
  logger,
  result,
  validatedParams
}) => {
  // this is an async iterator from the OpenAI sdk
  let telemetryMetadata;
  if (hasTelemetryMetadata(validatedParams.subActionParams)) {
    telemetryMetadata = validatedParams.subActionParams.telemetryMetadata;
  }
  if (validatedParams.subAction === 'invokeAsyncIterator' && actionTypeId === '.gen-ai' || actionTypeId === '.inference' && validatedParams.subAction === 'unified_completion_async_iterator') {
    try {
      const data = result.data;
      // the async interator is teed in the subaction response, double check that it has two streams
      if (data.tokenCountStream) {
        const body = actionTypeId === `.inference` ? validatedParams.subActionParams.body : validatedParams.subActionParams;
        const {
          total,
          prompt,
          completion
        } = await (0, _get_token_count_from_invoke_async_iterator.getTokenCountFromInvokeAsyncIterator)({
          streamIterable: data.tokenCountStream,
          body,
          logger
        });
        return {
          total_tokens: total,
          prompt_tokens: prompt,
          completion_tokens: completion,
          telemetry_metadata: telemetryMetadata
        };
      }
      logger.error('Failed to calculate tokens from Invoke Async Iterator subaction streaming response - unexpected response from actions client');
      return {
        total_tokens: 0,
        prompt_tokens: 0,
        completion_tokens: 0,
        telemetry_metadata: telemetryMetadata
      };
    } catch (e) {
      logger.error('Failed to calculate tokens from Invoke Async Iterator subaction streaming response');
      logger.error(e);
      // silently fail and null is returned at bottom of fuction
    }
  }

  // this is a streamed Gemini response, using the subAction invokeStream to stream the response as a simple string
  if (validatedParams.subAction === 'invokeStream' && result.data instanceof _stream.Readable && actionTypeId === '.gemini') {
    try {
      const {
        totalTokenCount,
        promptTokenCount,
        candidatesTokenCount
      } = await (0, _get_token_count_from_invoke_stream.parseGeminiStreamForUsageMetadata)({
        responseStream: result.data.pipe(new _stream.PassThrough()),
        logger
      });
      return {
        total_tokens: totalTokenCount,
        prompt_tokens: promptTokenCount,
        completion_tokens: candidatesTokenCount,
        telemetry_metadata: telemetryMetadata
      };
    } catch (e) {
      logger.error('Failed to calculate tokens from Invoke Stream subaction streaming response');
      logger.error(e);
      // silently fail and null is returned at bottom of fuction
    }
  }

  // this is a streamed OpenAI or Bedrock response, using the subAction invokeStream to stream the response as a simple string
  if (validatedParams.subAction === 'invokeStream' && result.data instanceof _stream.Readable && actionTypeId !== '.gemini') {
    try {
      const {
        total,
        prompt,
        completion
      } = await (0, _get_token_count_from_invoke_stream.getTokenCountFromInvokeStream)({
        responseStream: result.data.pipe(new _stream.PassThrough()),
        actionTypeId,
        body: validatedParams.subActionParams,
        logger
      });
      return {
        total_tokens: total,
        prompt_tokens: prompt,
        completion_tokens: completion,
        telemetry_metadata: telemetryMetadata
      };
    } catch (e) {
      logger.error('Failed to calculate tokens from Invoke Stream subaction streaming response');
      logger.error(e);
      // silently fail and null is returned at bottom of fuction
    }
  }

  // this is a streamed OpenAI response, which did not use the subAction invokeStream
  if (actionTypeId === '.gen-ai' && result.data instanceof _stream.Readable) {
    try {
      const {
        total,
        prompt,
        completion
      } = await (0, _get_token_count_from_openai_stream.getTokenCountFromOpenAIStream)({
        responseStream: result.data.pipe(new _stream.PassThrough()),
        body: validatedParams.subActionParams.body,
        logger
      });
      return {
        total_tokens: total,
        prompt_tokens: prompt,
        completion_tokens: completion,
        telemetry_metadata: telemetryMetadata
      };
    } catch (e) {
      logger.error('Failed to calculate tokens from streaming response');
      logger.error(e);
      // silently fail and null is returned at bottom of fuction
    }
  }

  // this is a non-streamed OpenAI response, which comes with the usage object
  if (actionTypeId === '.gen-ai') {
    var _data$usage$total_tok, _data$usage, _data$usage$prompt_to, _data$usage2, _data$usage$completio, _data$usage3;
    const data = result.data;
    if (data.usage == null) {
      logger.error('Response did not contain usage object');
      return null;
    }
    return {
      total_tokens: (_data$usage$total_tok = (_data$usage = data.usage) === null || _data$usage === void 0 ? void 0 : _data$usage.total_tokens) !== null && _data$usage$total_tok !== void 0 ? _data$usage$total_tok : 0,
      prompt_tokens: (_data$usage$prompt_to = (_data$usage2 = data.usage) === null || _data$usage2 === void 0 ? void 0 : _data$usage2.prompt_tokens) !== null && _data$usage$prompt_to !== void 0 ? _data$usage$prompt_to : 0,
      completion_tokens: (_data$usage$completio = (_data$usage3 = data.usage) === null || _data$usage3 === void 0 ? void 0 : _data$usage3.completion_tokens) !== null && _data$usage$completio !== void 0 ? _data$usage$completio : 0,
      telemetry_metadata: telemetryMetadata
    };
  }

  // this is a non-streamed Bedrock response
  if (actionTypeId === '.bedrock' && (validatedParams.subAction === 'run' || validatedParams.subAction === 'test')) {
    try {
      const rData = result.data;
      if (typeof rData.completion === 'string') {
        const {
          total,
          prompt,
          completion
        } = await (0, _get_token_count_from_bedrock_invoke.getTokenCountFromBedrockInvoke)({
          response: rData.completion,
          body: validatedParams.subActionParams.body,
          usage: rData.usage
        });
        return {
          total_tokens: total,
          prompt_tokens: prompt,
          completion_tokens: completion,
          telemetry_metadata: telemetryMetadata
        };
      } else {
        logger.error('Response from Bedrock run response did not contain completion string');
        return {
          total_tokens: 0,
          prompt_tokens: 0,
          completion_tokens: 0,
          telemetry_metadata: telemetryMetadata
        };
      }
    } catch (e) {
      logger.error('Failed to calculate tokens from Bedrock run response');
      logger.error(e);
    }
  }

  // Process non-streamed Gemini response from `usageMetadata` object
  if (actionTypeId === '.gemini') {
    var _data$usageMetadata$t, _data$usageMetadata, _data$usageMetadata$p, _data$usageMetadata2, _data$usageMetadata$c, _data$usageMetadata3;
    const data = result.data;
    if (data.usageMetadata == null) {
      logger.error('Response did not contain usage metadata object');
      return null;
    }
    return {
      total_tokens: (_data$usageMetadata$t = (_data$usageMetadata = data.usageMetadata) === null || _data$usageMetadata === void 0 ? void 0 : _data$usageMetadata.totalTokenCount) !== null && _data$usageMetadata$t !== void 0 ? _data$usageMetadata$t : 0,
      prompt_tokens: (_data$usageMetadata$p = (_data$usageMetadata2 = data.usageMetadata) === null || _data$usageMetadata2 === void 0 ? void 0 : _data$usageMetadata2.promptTokenCount) !== null && _data$usageMetadata$p !== void 0 ? _data$usageMetadata$p : 0,
      completion_tokens: (_data$usageMetadata$c = (_data$usageMetadata3 = data.usageMetadata) === null || _data$usageMetadata3 === void 0 ? void 0 : _data$usageMetadata3.candidatesTokenCount) !== null && _data$usageMetadata$c !== void 0 ? _data$usageMetadata$c : 0,
      telemetry_metadata: telemetryMetadata
    };
  }

  // this is a non-streamed Bedrock response used by security solution
  if (actionTypeId === '.bedrock' && validatedParams.subAction === 'invokeAI') {
    try {
      const rData = result.data;
      if (typeof rData.message === 'string') {
        const {
          total,
          prompt,
          completion
        } = await (0, _get_token_count_from_bedrock_invoke.getTokenCountFromBedrockInvoke)({
          response: rData.message,
          body: JSON.stringify({
            prompt: validatedParams.subActionParams.messages[0].content
          }),
          usage: rData.usage
        });
        return {
          total_tokens: total,
          prompt_tokens: prompt,
          completion_tokens: completion,
          telemetry_metadata: telemetryMetadata
        };
      } else {
        logger.error('Response from Bedrock invoke response did not contain message string');
        return {
          total_tokens: 0,
          prompt_tokens: 0,
          completion_tokens: 0,
          telemetry_metadata: telemetryMetadata
        };
      }
    } catch (e) {
      logger.error('Failed to calculate tokens from Bedrock invoke response');
      logger.error(e);
      // silently fail and null is returned at bottom of function
    }
  }

  // BedrockRuntimeClient.send response used by chat model ActionsClientChatBedrockConverse
  if (actionTypeId === '.bedrock' && validatedParams.subAction === 'bedrockClientSend') {
    const {
      tokenStream,
      usage
    } = result.data;
    if (tokenStream) {
      const res = await (0, _get_token_count_from_bedrock_converse.getTokensFromBedrockConverseStream)(tokenStream, logger);
      return res;
    }
    if (usage) {
      return {
        total_tokens: usage.totalTokens,
        prompt_tokens: usage.inputTokens,
        completion_tokens: usage.outputTokens,
        telemetry_metadata: telemetryMetadata
      };
    } else {
      logger.error('Response from Bedrock converse API did not contain usage object');
      return null;
    }
  }
  if (actionTypeId === '.bedrock' && validatedParams.subAction === 'invokeAIRaw') {
    const results = result.data;
    if (results !== null && results !== void 0 && results.usage) {
      const {
        input_tokens: inputTokens = 0,
        output_tokens: outputTokens = 0
      } = results.usage;
      return {
        total_tokens: inputTokens + outputTokens,
        prompt_tokens: inputTokens,
        completion_tokens: outputTokens,
        telemetry_metadata: telemetryMetadata
      };
    } else {
      logger.error('Response from Bedrock converse API did not contain usage object');
      return null;
    }
  }
  return null;
};
exports.getGenAiTokenTracking = getGenAiTokenTracking;
const shouldTrackGenAiToken = actionTypeId => actionTypeId === '.gen-ai' || actionTypeId === '.bedrock' || actionTypeId === '.gemini' || actionTypeId === '.inference';
exports.shouldTrackGenAiToken = shouldTrackGenAiToken;
function hasTelemetryMetadata(obj) {
  return obj !== null && typeof obj === 'object' && 'telemetryMetadata' in obj;
}