"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleCSV = handleCSV;
var _util = require("../../util");
var _processors = require("../../util/processors");
var _unparseable_csv_error = require("../../lib/errors/unparseable_csv_error");
var _columns = require("./columns");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// We will only create the processor for the first MAX_CSV_COLUMNS columns.
const MAX_CSV_COLUMNS = 100;

// Converts CSV samples into JSON samples.
async function handleCSV({
  state,
  client
}) {
  const packageName = state.packageName;
  const dataStreamName = state.dataStreamName;
  const samples = state.logSamples;
  const temporaryColumns = (0, _columns.generateColumnNames)(Math.min((0, _columns.upperBoundForColumnCount)(samples), MAX_CSV_COLUMNS));
  const temporaryProcessor = (0, _processors.createCSVProcessor)('message', temporaryColumns);
  const {
    pipelineResults: tempResults,
    errors: tempErrors
  } = await (0, _util.createJSONInput)([temporaryProcessor], samples, client);
  if (tempErrors.length > 0) {
    throw new _unparseable_csv_error.UnparseableCSVFormatError(tempErrors);
  }
  const headerColumns = state.samplesFormat.header ? (0, _columns.columnsFromHeader)(temporaryColumns, tempResults[0]) : [];
  const llmProvidedColumns = (state.samplesFormat.columns || []).map(_columns.toSafeColumnName);
  const needColumns = (0, _columns.totalColumnCount)(temporaryColumns, tempResults);
  const columns = Array.from((0, _columns.yieldUniqueColumnNames)(needColumns, [llmProvidedColumns, headerColumns], temporaryColumns));
  const prefix = [packageName, dataStreamName];
  const prefixedColumns = (0, _columns.prefixColumns)(columns, prefix);
  const csvProcessor = (0, _processors.createCSVProcessor)('message', prefixedColumns);
  const csvHandlingProcessors = [csvProcessor];
  if (headerColumns.length > 0) {
    const dropValues = columns.reduce((acc, column, index) => {
      if (headerColumns[index] !== undefined) {
        acc[column] = String(headerColumns[index]);
      }
      return acc;
    }, {});
    const dropProcessor = (0, _processors.createDropProcessor)(dropValues, prefix, 'remove_csv_header', 'Remove the CSV header line by comparing the values');
    csvHandlingProcessors.push(dropProcessor);
  }
  const {
    pipelineResults: finalResults,
    errors: finalErrors
  } = await (0, _util.createJSONInput)(csvHandlingProcessors, samples, client);
  if (finalErrors.length > 0) {
    throw new _unparseable_csv_error.UnparseableCSVFormatError(finalErrors);
  }

  // Converts JSON Object into a string and parses it as a array of JSON strings
  const jsonSamples = finalResults.map(log => log[packageName]).map(log => log[dataStreamName]).map(log => JSON.stringify(log));
  return {
    jsonSamples,
    additionalProcessors: [...state.additionalProcessors, ...csvHandlingProcessors],
    lastExecutedChain: 'handleCSV'
  };
}