"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getMetricMetadata = void 0;
var _lodash = require("lodash");
var _common = require("@kbn/metrics-data-access-plugin/common");
var _constants = require("../../../../common/constants");
var _get_filtered_hosts = require("../../infra/lib/host/get_filtered_hosts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getMetricMetadata = async (framework, requestContext, sourceConfiguration, nodeId, nodeType, timeRange, infraMetricsClient) => {
  const fields = (0, _common.findInventoryFields)(nodeType);
  const metricQuery = {
    allow_no_indices: true,
    ignore_unavailable: true,
    index: sourceConfiguration.metricAlias,
    body: {
      query: {
        bool: {
          must_not: [{
            match: {
              'event.dataset': 'aws.ec2'
            }
          }],
          filter: [{
            match: {
              [fields.id]: nodeId
            }
          }, {
            range: {
              [_constants.TIMESTAMP_FIELD]: {
                gte: timeRange.from,
                lte: timeRange.to,
                format: 'epoch_millis'
              }
            }
          }]
        }
      },
      size: 0,
      aggs: {
        nodeName: {
          terms: {
            field: fields.name,
            size: 1
          }
        },
        metrics: {
          terms: {
            field: 'event.dataset',
            size: 1000
          }
        }
      }
    }
  };
  const response = await framework.callWithRequest(requestContext, 'search', metricQuery);
  const buckets = response.aggregations && response.aggregations.metrics ? response.aggregations.metrics.buckets : [];
  const res = {
    id: nodeId,
    name: (0, _lodash.get)(response, ['aggregations', 'nodeName', 'buckets', 0, 'key'], nodeId),
    buckets
  };
  if (nodeType === 'host') {
    const hasSystemIntegration = await (0, _get_filtered_hosts.getHasDataFromSystemIntegration)({
      infraMetricsClient,
      from: timeRange.from,
      to: timeRange.to,
      query: {
        match: {
          [fields.id]: nodeId
        }
      }
    });
    return {
      hasSystemIntegration,
      ...res
    };
  }
  return res;
};
exports.getMetricMetadata = getMetricMetadata;