"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createMetricThresholdExecutor = exports.WARNING_ACTIONS_ID = exports.WARNING_ACTIONS = exports.NO_DATA_ACTIONS_ID = exports.NO_DATA_ACTIONS = exports.FIRED_ACTIONS_ID = exports.FIRED_ACTIONS = void 0;
var _i18n = require("@kbn/i18n");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lodash = require("lodash");
var _common = require("@kbn/alerting-plugin/common");
var _server = require("@kbn/alerting-plugin/server");
var _common2 = require("@kbn/observability-plugin/common");
var _alertingRuleUtils = require("@kbn/alerting-rule-utils");
var _convert_legacy_outside_comparator = require("@kbn/observability-plugin/common/utils/convert_legacy_outside_comparator");
var _get_original_action_group = require("../../../utils/get_original_action_group");
var _metrics = require("../../../../common/alerting/metrics");
var _formatters = require("../../../../common/formatters");
var _messages = require("../common/messages");
var _utils = require("../common/utils");
var _get_values = require("../common/get_values");
var _evaluate_rule = require("./lib/evaluate_rule");
var _convert_strings_to_missing_groups_record = require("./lib/convert_strings_to_missing_groups_record");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
// no specific instance state used
// no specific instance state used

const FIRED_ACTIONS_ID = exports.FIRED_ACTIONS_ID = 'metrics.threshold.fired';
const WARNING_ACTIONS_ID = exports.WARNING_ACTIONS_ID = 'metrics.threshold.warning';
const NO_DATA_ACTIONS_ID = exports.NO_DATA_ACTIONS_ID = 'metrics.threshold.nodata';
// TODO: Refactor the executor code to have better flow-control with better
// reasoning of different state/conditions for improved maintainability
const createMetricThresholdExecutor = (libs, {
  alertsLocator,
  assetDetailsLocator,
  metricsExplorerLocator
}) => async options => {
  var _alertsClient$getReco;
  const startTime = Date.now();
  const {
    services,
    params,
    state,
    startedAt,
    executionId,
    spaceId,
    rule: {
      id: ruleId
    }
  } = options;
  const {
    criteria
  } = params;
  if (criteria.length === 0) throw new Error('Cannot execute an alert with 0 conditions');
  const groupBy = (0, _lodash.castArray)(params.groupBy);
  const logger = (0, _utils.createScopedLogger)(libs.logger, 'metricThresholdRule', {
    alertId: ruleId,
    executionId
  });
  const {
    alertsClient,
    savedObjectsClient
  } = services;
  if (!alertsClient) {
    throw new _server.AlertsClientError();
  }
  const alertReporter = async ({
    id,
    reason,
    actionGroup,
    context: contextWithoutAlertDetailsUrl,
    additionalContext,
    evaluationValues,
    groups,
    thresholds
  }) => {
    const {
      uuid
    } = alertsClient.report({
      id,
      actionGroup
    });
    alertsClient.setAlertData({
      id,
      payload: {
        [_ruleDataUtils.ALERT_REASON]: reason,
        [_ruleDataUtils.ALERT_EVALUATION_VALUES]: evaluationValues,
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: thresholds,
        [_ruleDataUtils.ALERT_GROUP]: groups,
        ...(0, _utils.flattenAdditionalContext)(additionalContext),
        ...(0, _alertingRuleUtils.getEcsGroups)(groups)
      },
      context: {
        ...contextWithoutAlertDetailsUrl,
        alertDetailsUrl: await (0, _common2.getAlertDetailsUrl)(libs.basePath, spaceId, uuid)
      }
    });
  };
  const {
    sourceId,
    alertOnNoData,
    alertOnGroupDisappear: _alertOnGroupDisappear
  } = params;
  if (!params.filterQuery && params.filterQueryText) {
    try {
      const {
        fromKueryExpression
      } = await Promise.resolve().then(() => _interopRequireWildcard(require('@kbn/es-query')));
      fromKueryExpression(params.filterQueryText);
    } catch (e) {
      const timestamp = startedAt.toISOString();
      const actionGroupId = FIRED_ACTIONS_ID; // Change this to an Error action group when able
      const reason = (0, _messages.buildInvalidQueryAlertReason)(params.filterQueryText);
      const alertContext = {
        alertState: _messages.stateToAlertMessage[_metrics.AlertStates.ERROR],
        group: _utils.UNGROUPED_FACTORY_KEY,
        metric: mapToConditionsLookup(criteria, c => c.metric),
        reason,
        timestamp,
        value: null,
        // TODO: Check if we need additionalContext here or not?
        viewInAppUrl: (0, _utils.getMetricsViewInAppUrlWithSpaceId)({
          timestamp,
          groupBy,
          assetDetailsLocator,
          metricsExplorerLocator
        })
      };
      await alertReporter({
        id: _utils.UNGROUPED_FACTORY_KEY,
        reason,
        actionGroup: actionGroupId,
        context: alertContext
      });
      return {
        state: {
          lastRunTimestamp: startedAt.valueOf(),
          missingGroups: [],
          groupBy,
          filterQuery: params.filterQuery
        }
      };
    }
  }

  // For backwards-compatibility, interpret undefined alertOnGroupDisappear as true
  const alertOnGroupDisappear = _alertOnGroupDisappear !== false;
  const source = await libs.sources.getSourceConfiguration(savedObjectsClient, sourceId || 'default');
  const config = source.configuration;
  const compositeSize = libs.configuration.alerting.metric_threshold.group_by_page_size;
  const filterQueryIsSame = (0, _lodash.isEqual)(state.filterQuery, params.filterQuery);
  const groupByIsSame = (0, _lodash.isEqual)(state.groupBy, params.groupBy);
  const previousMissingGroups = alertOnGroupDisappear && filterQueryIsSame && groupByIsSame && state.missingGroups ? state.missingGroups.filter(missingGroup =>
  // We use isTrackedAlert to remove missing groups that are untracked by the user
  typeof missingGroup === 'string' ? alertsClient.isTrackedAlert(missingGroup) : alertsClient.isTrackedAlert(missingGroup.key)) : [];
  const alertResults = await (0, _evaluate_rule.evaluateRule)(services.scopedClusterClient.asCurrentUser, params, config, compositeSize, alertOnGroupDisappear, logger, state.lastRunTimestamp, {
    end: startedAt.valueOf()
  }, (0, _convert_strings_to_missing_groups_record.convertStringsToMissingGroupsRecord)(previousMissingGroups));
  const resultGroupSet = new Set();
  for (const resultSet of alertResults) {
    for (const group of Object.keys(resultSet)) {
      resultGroupSet.add(group);
    }
  }
  const groupByKeysObjectMapping = (0, _utils.getGroupByObject)(params.groupBy, resultGroupSet);
  const groupByMapping = (0, _utils.getFormattedGroupBy)(params.groupBy, resultGroupSet);
  const groupArray = [...resultGroupSet];
  const nextMissingGroups = new Set();
  const hasGroups = !(0, _lodash.isEqual)(groupArray, [_utils.UNGROUPED_FACTORY_KEY]);
  let scheduledActionsCount = 0;

  // The key of `groupArray` is the alert instance ID.
  for (const group of groupArray) {
    // AND logic; all criteria must be across the threshold
    const shouldAlertFire = alertResults.every(result => {
      var _result$group;
      return (_result$group = result[group]) === null || _result$group === void 0 ? void 0 : _result$group.shouldFire;
    });
    const shouldAlertWarn = alertResults.every(result => {
      var _result$group2;
      return (_result$group2 = result[group]) === null || _result$group2 === void 0 ? void 0 : _result$group2.shouldWarn;
    });
    // AND logic; because we need to evaluate all criteria, if one of them reports no data then the
    // whole alert is in a No Data/Error state
    const isNoData = alertResults.some(result => {
      var _result$group3;
      return (_result$group3 = result[group]) === null || _result$group3 === void 0 ? void 0 : _result$group3.isNoData;
    });
    if (isNoData && group !== _utils.UNGROUPED_FACTORY_KEY) {
      nextMissingGroups.add({
        key: group,
        bucketKey: alertResults[0][group].bucketKey
      });
    }
    const nextState = isNoData ? _metrics.AlertStates.NO_DATA : shouldAlertFire ? _metrics.AlertStates.ALERT : shouldAlertWarn ? _metrics.AlertStates.WARNING : _metrics.AlertStates.OK;
    let reason;
    if (nextState === _metrics.AlertStates.ALERT || nextState === _metrics.AlertStates.WARNING) {
      reason = alertResults.map(result => (0, _messages.buildFiredAlertReason)({
        ...formatAlertResult({
          ...result[group],
          comparator: (0, _convert_legacy_outside_comparator.convertToBuiltInComparators)(result[group].comparator),
          warningComparator: result[group].comparator ? (0, _convert_legacy_outside_comparator.convertToBuiltInComparators)(result[group].comparator) : undefined
        }, nextState === _metrics.AlertStates.WARNING),
        group
      })).join('\n');
    }

    /* NO DATA STATE HANDLING
     *
     * - `alertOnNoData` does not indicate IF the alert's next state is No Data, but whether or not the user WANTS TO BE ALERTED
     *   if the state were No Data.
     * - `alertOnGroupDisappear`, on the other hand, determines whether or not it's possible to return a No Data state
     *   when a group disappears.
     *
     * This means we need to handle the possibility that `alertOnNoData` is false, but `alertOnGroupDisappear` is true
     *
     * nextState === NO_DATA would be true on both { '*': No Data } or, e.g. { 'a': No Data, 'b': OK, 'c': OK }, but if the user
     * has for some reason disabled `alertOnNoData` and left `alertOnGroupDisappear` enabled, they would only care about the latter
     * possibility. In this case, use hasGroups to determine whether to alert on a potential No Data state
     *
     * If `alertOnNoData` is true but `alertOnGroupDisappear` is false, we don't need to worry about the {a, b, c} possibility.
     * At this point in the function, a false `alertOnGroupDisappear` would already have prevented group 'a' from being evaluated at all.
     */
    if (alertOnNoData || alertOnGroupDisappear && hasGroups) {
      // In the previous line we've determined if the user is interested in No Data states, so only now do we actually
      // check to see if a No Data state has occurred
      if (nextState === _metrics.AlertStates.NO_DATA) {
        reason = alertResults.filter(result => {
          var _result$group4;
          return (_result$group4 = result[group]) === null || _result$group4 === void 0 ? void 0 : _result$group4.isNoData;
        }).map(result => (0, _messages.buildNoDataAlertReason)({
          ...result[group],
          group
        })).join('\n');
      }
    }
    if (reason) {
      var _alertResults$0$group, _additionalContext$ta;
      const timestamp = startedAt.toISOString();
      const actionGroupId = nextState === _metrics.AlertStates.OK ? _common.RecoveredActionGroup.id : nextState === _metrics.AlertStates.NO_DATA ? NO_DATA_ACTIONS_ID : nextState === _metrics.AlertStates.WARNING ? WARNING_ACTIONS_ID : FIRED_ACTIONS_ID;
      const additionalContext = (0, _utils.hasAdditionalContext)(params.groupBy, _utils.validGroupByForContext) ? alertResults && alertResults.length > 0 ? (_alertResults$0$group = alertResults[0][group].context) !== null && _alertResults$0$group !== void 0 ? _alertResults$0$group : {} : {} : {};
      additionalContext.tags = Array.from(new Set([...((_additionalContext$ta = additionalContext.tags) !== null && _additionalContext$ta !== void 0 ? _additionalContext$ta : []), ...options.rule.tags]));
      const evaluationValues = (0, _get_values.getEvaluationValues)(alertResults, group);
      const thresholds = (0, _get_values.getThresholds)(criteria);
      const groups = groupByMapping[group];
      const alertContext = {
        alertState: _messages.stateToAlertMessage[nextState],
        group,
        groupByKeys: groupByKeysObjectMapping[group],
        metric: mapToConditionsLookup(criteria, c => {
          if (c.aggType === 'count') {
            return 'count';
          }
          return c.metric;
        }),
        reason,
        threshold: mapToConditionsLookup(alertResults, (result, index) => {
          const evaluation = result[group];
          if (!evaluation) {
            return criteria[index].threshold;
          }
          return formatAlertResult({
            ...evaluation,
            comparator: (0, _convert_legacy_outside_comparator.convertToBuiltInComparators)(evaluation.comparator),
            warningComparator: evaluation.warningComparator ? (0, _convert_legacy_outside_comparator.convertToBuiltInComparators)(evaluation.warningComparator) : undefined
          }).threshold;
        }),
        timestamp,
        value: mapToConditionsLookup(alertResults, (result, index) => {
          const evaluation = result[group];
          if (!evaluation && criteria[index].aggType === 'count') {
            return 0;
          } else if (!evaluation) {
            return null;
          }
          return formatAlertResult({
            ...evaluation,
            comparator: (0, _convert_legacy_outside_comparator.convertToBuiltInComparators)(evaluation.comparator),
            warningComparator: evaluation.warningComparator ? (0, _convert_legacy_outside_comparator.convertToBuiltInComparators)(evaluation.warningComparator) : undefined
          }).currentValue;
        }),
        viewInAppUrl: (0, _utils.getMetricsViewInAppUrlWithSpaceId)({
          timestamp,
          groupBy,
          assetDetailsLocator,
          metricsExplorerLocator,
          additionalContext
        }),
        ...additionalContext
      };
      await alertReporter({
        id: `${group}`,
        reason,
        actionGroup: actionGroupId,
        context: alertContext,
        additionalContext,
        evaluationValues,
        groups,
        thresholds
      });
      scheduledActionsCount++;
    }
  }
  const recoveredAlerts = (_alertsClient$getReco = alertsClient === null || alertsClient === void 0 ? void 0 : alertsClient.getRecoveredAlerts()) !== null && _alertsClient$getReco !== void 0 ? _alertsClient$getReco : [];
  const groupByKeysObjectForRecovered = (0, _utils.getGroupByObject)(params.groupBy, new Set(recoveredAlerts.map(recoveredAlert => recoveredAlert.alert.getId())));
  for (const recoveredAlert of recoveredAlerts) {
    var _recoveredAlert$alert;
    const recoveredAlertId = recoveredAlert.alert.getId();
    const alertUuid = recoveredAlert.alert.getUuid();
    const timestamp = startedAt.toISOString();
    const indexedStartedAt = (_recoveredAlert$alert = recoveredAlert.alert.getStart()) !== null && _recoveredAlert$alert !== void 0 ? _recoveredAlert$alert : timestamp;
    const alertHits = recoveredAlert.hit;
    const additionalContext = (0, _utils.getContextForRecoveredAlerts)(alertHits);
    const originalActionGroup = (0, _get_original_action_group.getOriginalActionGroup)(alertHits);
    recoveredAlert.alert.setContext({
      alertDetailsUrl: await (0, _common2.getAlertDetailsUrl)(libs.basePath, spaceId, alertUuid),
      alertState: _messages.stateToAlertMessage[_metrics.AlertStates.OK],
      group: recoveredAlertId,
      groupByKeys: groupByKeysObjectForRecovered[recoveredAlertId],
      metric: mapToConditionsLookup(criteria, c => {
        if (criteria.aggType === 'count') {
          return 'count';
        }
        return c.metric;
      }),
      timestamp,
      threshold: mapToConditionsLookup(criteria, c => c.threshold),
      viewInAppUrl: (0, _utils.getMetricsViewInAppUrlWithSpaceId)({
        timestamp: indexedStartedAt,
        groupBy,
        assetDetailsLocator,
        metricsExplorerLocator,
        additionalContext
      }),
      originalAlertState: translateActionGroupToAlertState(originalActionGroup),
      originalAlertStateWasALERT: originalActionGroup === FIRED_ACTIONS.id,
      originalAlertStateWasWARNING: originalActionGroup === WARNING_ACTIONS.id,
      // eslint-disable-next-line @typescript-eslint/naming-convention
      originalAlertStateWasNO_DATA: originalActionGroup === NO_DATA_ACTIONS.id,
      ...additionalContext
    });
  }
  const stopTime = Date.now();
  logger.debug(`Scheduled ${scheduledActionsCount} actions in ${stopTime - startTime}ms`);
  return {
    state: {
      lastRunTimestamp: startedAt.valueOf(),
      missingGroups: [...nextMissingGroups],
      groupBy,
      filterQuery: params.filterQuery
    }
  };
};
exports.createMetricThresholdExecutor = createMetricThresholdExecutor;
const FIRED_ACTIONS = exports.FIRED_ACTIONS = {
  id: 'metrics.threshold.fired',
  name: _i18n.i18n.translate('xpack.infra.metrics.alerting.threshold.fired', {
    defaultMessage: 'Alert'
  })
};
const WARNING_ACTIONS = exports.WARNING_ACTIONS = {
  id: 'metrics.threshold.warning',
  name: _i18n.i18n.translate('xpack.infra.metrics.alerting.threshold.warning', {
    defaultMessage: 'Warning'
  })
};
const NO_DATA_ACTIONS = exports.NO_DATA_ACTIONS = {
  id: 'metrics.threshold.nodata',
  name: _i18n.i18n.translate('xpack.infra.metrics.alerting.threshold.nodata', {
    defaultMessage: 'No Data'
  })
};
const translateActionGroupToAlertState = actionGroupId => {
  if (actionGroupId === FIRED_ACTIONS.id) {
    return _messages.stateToAlertMessage[_metrics.AlertStates.ALERT];
  }
  if (actionGroupId === WARNING_ACTIONS.id) {
    return _messages.stateToAlertMessage[_metrics.AlertStates.WARNING];
  }
  if (actionGroupId === NO_DATA_ACTIONS.id) {
    return _messages.stateToAlertMessage[_metrics.AlertStates.NO_DATA];
  }
};
const mapToConditionsLookup = (list, mapFn) => list.map(mapFn).reduce((result, value, i) => {
  result[`condition${i}`] = value;
  return result;
}, {});
const formatAlertResult = (alertResult, useWarningThreshold) => {
  const {
    metric,
    currentValue,
    threshold,
    comparator,
    warningThreshold,
    warningComparator
  } = alertResult;
  const noDataValue = _i18n.i18n.translate('xpack.infra.metrics.alerting.threshold.noDataFormattedValue', {
    defaultMessage: '[NO DATA]'
  });
  const thresholdToFormat = useWarningThreshold ? warningThreshold : threshold;
  const comparatorToUse = useWarningThreshold ? warningComparator : comparator;
  if (metric.endsWith('.pct')) {
    const formatter = (0, _formatters.createFormatter)('percent');
    return {
      ...alertResult,
      currentValue: currentValue !== null && currentValue !== undefined ? formatter(currentValue) : noDataValue,
      threshold: Array.isArray(thresholdToFormat) ? thresholdToFormat.map(v => formatter(v)) : formatter(thresholdToFormat),
      comparator: (0, _convert_legacy_outside_comparator.convertToBuiltInComparators)(comparatorToUse)
    };
  }
  const formatter = (0, _formatters.createFormatter)('highPrecision');
  return {
    ...alertResult,
    currentValue: currentValue !== null && currentValue !== undefined ? formatter(currentValue) : noDataValue,
    threshold: Array.isArray(thresholdToFormat) ? thresholdToFormat.map(v => formatter(v)) : formatter(thresholdToFormat),
    comparator: comparatorToUse
  };
};