"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createInventoryMetricThresholdExecutor = exports.WARNING_ACTIONS_ID = exports.WARNING_ACTIONS = exports.FIRED_ACTIONS_ID = exports.FIRED_ACTIONS = void 0;
var _i18n = require("@kbn/i18n");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lodash = require("lodash");
var _server = require("@kbn/alerting-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _get_original_action_group = require("../../../utils/get_original_action_group");
var _metrics = require("../../../../common/alerting/metrics");
var _formatters = require("../../../../common/formatters");
var _get_custom_metric_label = require("../../../../common/formatters/get_custom_metric_label");
var _snapshot_metric_formats = require("../../../../common/formatters/snapshot_metric_formats");
var _snapshot_metric_i18n = require("../../../../common/snapshot_metric_i18n");
var _messages = require("../common/messages");
var _utils = require("../common/utils");
var _get_values = require("../common/get_values");
var _evaluate_condition = require("./evaluate_condition");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; } /*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
const FIRED_ACTIONS_ID = exports.FIRED_ACTIONS_ID = 'metrics.inventory_threshold.fired';
const WARNING_ACTIONS_ID = exports.WARNING_ACTIONS_ID = 'metrics.inventory_threshold.warning';

// no specific state used
// no specific state used
// no specific instance context used

const createInventoryMetricThresholdExecutor = (libs, {
  alertsLocator,
  assetDetailsLocator,
  inventoryLocator
}) => async options => {
  var _alertsClient$getReco;
  const {
    services,
    params,
    startedAt,
    executionId,
    spaceId,
    rule: {
      id: ruleId,
      tags: ruleTags
    },
    getTimeRange
  } = options;
  const startTime = Date.now();
  const {
    criteria,
    filterQuery,
    sourceId = 'default',
    nodeType,
    alertOnNoData
  } = params;
  if (criteria.length === 0) throw new Error('Cannot execute an alert with 0 conditions');
  const logger = (0, _utils.createScopedLogger)(libs.logger, 'inventoryRule', {
    alertId: ruleId,
    executionId
  });
  const esClient = services.scopedClusterClient.asCurrentUser;
  const {
    savedObjectsClient,
    alertsClient
  } = services;
  if (!alertsClient) {
    throw new _server.AlertsClientError();
  }
  if (!params.filterQuery && params.filterQueryText) {
    try {
      const {
        fromKueryExpression
      } = await Promise.resolve().then(() => _interopRequireWildcard(require('@kbn/es-query')));
      fromKueryExpression(params.filterQueryText);
    } catch (e) {
      logger.error(e.message);
      const actionGroup = FIRED_ACTIONS.id; // Change this to an Error action group when able,
      const reason = (0, _messages.buildInvalidQueryAlertReason)(params.filterQueryText);
      const {
        uuid,
        start
      } = alertsClient.report({
        id: _utils.UNGROUPED_FACTORY_KEY,
        actionGroup
      });
      const indexedStartedAt = start !== null && start !== void 0 ? start : startedAt.toISOString();
      alertsClient.setAlertData({
        id: _utils.UNGROUPED_FACTORY_KEY,
        payload: {
          [_ruleDataUtils.ALERT_REASON]: reason
        },
        context: {
          alertDetailsUrl: await (0, _common.getAlertDetailsUrl)(libs.basePath, spaceId, uuid),
          alertState: _messages.stateToAlertMessage[_metrics.AlertStates.ERROR],
          group: _utils.UNGROUPED_FACTORY_KEY,
          metric: mapToConditionsLookup(criteria, c => c.metric),
          reason,
          timestamp: startedAt.toISOString(),
          value: null,
          viewInAppUrl: (0, _utils.getInventoryViewInAppUrlWithSpaceId)({
            criteria,
            nodeType,
            timestamp: indexedStartedAt,
            assetDetailsLocator,
            inventoryLocator
          })
        }
      });
      return {
        state: {}
      };
    }
  }
  const source = await libs.sources.getSourceConfiguration(savedObjectsClient, sourceId);
  const [, {
    logsShared,
    logsDataAccess
  }] = await libs.getStartServices();
  const logSourcesService = logsDataAccess.services.logSourcesServiceFactory.getLogSourcesService(savedObjectsClient);
  const logQueryFields = await logsShared.logViews.getClient(savedObjectsClient, esClient, logSourcesService).getResolvedLogView({
    type: 'log-view-reference',
    logViewId: sourceId
  }).then(({
    indices
  }) => ({
    indexPattern: indices
  }), () => undefined);
  const compositeSize = libs.configuration.alerting.inventory_threshold.group_by_page_size;
  const {
    dateEnd
  } = getTimeRange();
  const results = await Promise.all(criteria.map(condition => (0, _evaluate_condition.evaluateCondition)({
    compositeSize,
    condition,
    esClient,
    executionTimestamp: new Date(dateEnd),
    filterQuery,
    logger,
    logQueryFields,
    nodeType,
    source
  })));
  let scheduledActionsCount = 0;
  const alertLimit = alertsClient.getAlertLimitValue();
  let hasReachedLimit = false;
  const inventoryItems = Object.keys((0, _lodash.first)(results));
  for (const group of inventoryItems) {
    if (scheduledActionsCount >= alertLimit) {
      // need to set this so that warning is displayed in the UI and in the logs
      hasReachedLimit = true;
      break; // once limit is reached, we break out of the loop and don't schedule any more alerts
    }
    // AND logic; all criteria must be across the threshold
    const shouldAlertFire = results.every(result => {
      var _result$group;
      return (_result$group = result[group]) === null || _result$group === void 0 ? void 0 : _result$group.shouldFire;
    });
    const shouldAlertWarn = results.every(result => {
      var _result$group2;
      return (_result$group2 = result[group]) === null || _result$group2 === void 0 ? void 0 : _result$group2.shouldWarn;
    });
    // AND logic; because we need to evaluate all criteria, if one of them reports no data then the
    // whole alert is in a No Data/Error state
    const isNoData = results.some(result => {
      var _result$group3;
      return (_result$group3 = result[group]) === null || _result$group3 === void 0 ? void 0 : _result$group3.isNoData;
    });
    const isError = results.some(result => {
      var _result$group4;
      return (_result$group4 = result[group]) === null || _result$group4 === void 0 ? void 0 : _result$group4.isError;
    });
    const nextState = isError ? _metrics.AlertStates.ERROR : isNoData ? _metrics.AlertStates.NO_DATA : shouldAlertFire ? _metrics.AlertStates.ALERT : shouldAlertWarn ? _metrics.AlertStates.WARNING : _metrics.AlertStates.OK;
    let reason;
    if (nextState === _metrics.AlertStates.ALERT || nextState === _metrics.AlertStates.WARNING) {
      reason = results.map(result => buildReasonWithVerboseMetricName({
        group,
        resultItem: result[group],
        buildReason: _messages.buildFiredAlertReason,
        useWarningThreshold: nextState === _metrics.AlertStates.WARNING,
        nodeType
      })).join('\n');
    }
    if (alertOnNoData) {
      if (nextState === _metrics.AlertStates.NO_DATA) {
        reason = results.filter(result => result[group].isNoData).map(result => buildReasonWithVerboseMetricName({
          group,
          resultItem: result[group],
          buildReason: _messages.buildNoDataAlertReason,
          nodeType
        })).join('\n');
      } else if (nextState === _metrics.AlertStates.ERROR) {
        reason = results.filter(result => result[group].isError).map(result => buildReasonWithVerboseMetricName({
          group,
          resultItem: result[group],
          buildReason: _messages.buildErrorAlertReason,
          nodeType
        })).join('\n');
      }
    }
    if (reason) {
      var _additionalContext$ta, _additionalContext$ho;
      const actionGroup = nextState === _metrics.AlertStates.WARNING ? WARNING_ACTIONS_ID : FIRED_ACTIONS_ID;
      const additionalContext = results && results.length > 0 ? results[0][group].context : {};
      additionalContext.tags = Array.from(new Set([...((_additionalContext$ta = additionalContext.tags) !== null && _additionalContext$ta !== void 0 ? _additionalContext$ta : []), ...ruleTags]));
      const evaluationValues = (0, _get_values.getEvaluationValues)(results, group);
      const thresholds = (0, _get_values.getThresholds)(criteria);
      const {
        uuid,
        start
      } = alertsClient.report({
        id: group,
        actionGroup
      });
      const indexedStartedAt = start !== null && start !== void 0 ? start : startedAt.toISOString();
      scheduledActionsCount++;
      const context = {
        alertDetailsUrl: await (0, _common.getAlertDetailsUrl)(libs.basePath, spaceId, uuid),
        alertState: _messages.stateToAlertMessage[nextState],
        group,
        reason,
        metric: mapToConditionsLookup(criteria, c => c.metric),
        timestamp: startedAt.toISOString(),
        threshold: mapToConditionsLookup(criteria, c => c.threshold),
        value: mapToConditionsLookup(results, result => formatMetric(result[group].metric, result[group].currentValue)),
        viewInAppUrl: (0, _utils.getInventoryViewInAppUrlWithSpaceId)({
          criteria,
          nodeType,
          timestamp: indexedStartedAt,
          hostName: additionalContext === null || additionalContext === void 0 ? void 0 : (_additionalContext$ho = additionalContext.host) === null || _additionalContext$ho === void 0 ? void 0 : _additionalContext$ho.name,
          assetDetailsLocator,
          inventoryLocator
        }),
        ...additionalContext
      };
      const payload = {
        [_ruleDataUtils.ALERT_REASON]: reason,
        [_ruleDataUtils.ALERT_EVALUATION_VALUES]: evaluationValues,
        [_ruleDataUtils.ALERT_EVALUATION_THRESHOLD]: thresholds,
        ...(0, _utils.flattenAdditionalContext)(additionalContext)
      };
      alertsClient.setAlertData({
        id: group,
        payload,
        context
      });
    }
  }
  alertsClient.setAlertLimitReached(hasReachedLimit);
  const recoveredAlerts = (_alertsClient$getReco = alertsClient.getRecoveredAlerts()) !== null && _alertsClient$getReco !== void 0 ? _alertsClient$getReco : [];
  for (const recoveredAlert of recoveredAlerts) {
    var _recoveredAlert$alert, _additionalContext$ho2;
    const recoveredAlertId = recoveredAlert.alert.getId();
    const indexedStartedAt = (_recoveredAlert$alert = recoveredAlert.alert.getStart()) !== null && _recoveredAlert$alert !== void 0 ? _recoveredAlert$alert : startedAt.toISOString();
    const alertUuid = recoveredAlert.alert.getUuid();
    const alertHits = recoveredAlert.hit;
    const additionalContext = (0, _utils.getContextForRecoveredAlerts)(alertHits);
    const originalActionGroup = (0, _get_original_action_group.getOriginalActionGroup)(alertHits);
    const recoveredContext = {
      alertDetailsUrl: await (0, _common.getAlertDetailsUrl)(libs.basePath, spaceId, alertUuid),
      alertState: _messages.stateToAlertMessage[_metrics.AlertStates.OK],
      group: recoveredAlertId,
      metric: mapToConditionsLookup(criteria, c => c.metric),
      threshold: mapToConditionsLookup(criteria, c => c.threshold),
      timestamp: startedAt.toISOString(),
      viewInAppUrl: (0, _utils.getInventoryViewInAppUrlWithSpaceId)({
        criteria,
        nodeType,
        timestamp: indexedStartedAt,
        hostName: additionalContext === null || additionalContext === void 0 ? void 0 : (_additionalContext$ho2 = additionalContext.host) === null || _additionalContext$ho2 === void 0 ? void 0 : _additionalContext$ho2.name,
        assetDetailsLocator,
        inventoryLocator
      }),
      originalAlertState: translateActionGroupToAlertState(originalActionGroup),
      originalAlertStateWasALERT: originalActionGroup === FIRED_ACTIONS_ID,
      originalAlertStateWasWARNING: originalActionGroup === WARNING_ACTIONS_ID,
      ...additionalContext
    };
    alertsClient.setAlertData({
      id: recoveredAlertId,
      context: recoveredContext
    });
  }
  const stopTime = Date.now();
  logger.debug(`Scheduled ${scheduledActionsCount} actions in ${stopTime - startTime}ms`);
  return {
    state: {}
  };
};
exports.createInventoryMetricThresholdExecutor = createInventoryMetricThresholdExecutor;
const formatThreshold = (metric, value) => {
  const metricFormatter = (0, _lodash.get)(_snapshot_metric_formats.METRIC_FORMATTERS, metric, _snapshot_metric_formats.METRIC_FORMATTERS.count);
  const formatter = (0, _formatters.createFormatter)(metricFormatter.formatter, metricFormatter.template);
  const threshold = Array.isArray(value) ? value.map(v => {
    if (metricFormatter.formatter === 'percent') {
      v = Number(v) / 100;
    }
    if (metricFormatter.formatter === 'bits') {
      v = Number(v) / 8;
    }
    return formatter(v);
  }) : value;
  return threshold;
};
const buildReasonWithVerboseMetricName = ({
  group,
  resultItem,
  buildReason,
  useWarningThreshold,
  nodeType
}) => {
  var _toMetricOpt;
  if (!resultItem) return '';
  const thresholdToFormat = useWarningThreshold ? resultItem.warningThreshold : resultItem.threshold;
  const resultWithVerboseMetricName = {
    ...resultItem,
    group,
    metric: ((_toMetricOpt = (0, _snapshot_metric_i18n.toMetricOpt)(resultItem.metric, nodeType)) === null || _toMetricOpt === void 0 ? void 0 : _toMetricOpt.text) || (resultItem.metric === 'custom' && resultItem.customMetric ? (0, _get_custom_metric_label.getCustomMetricLabel)(resultItem.customMetric) : resultItem.metric),
    currentValue: formatMetric(resultItem.metric, resultItem.currentValue),
    threshold: formatThreshold(resultItem.metric, thresholdToFormat),
    comparator: useWarningThreshold ? (0, _common.convertToBuiltInComparators)(resultItem.warningComparator) : (0, _common.convertToBuiltInComparators)(resultItem.comparator)
  };
  return buildReason(resultWithVerboseMetricName);
};
const mapToConditionsLookup = (list, mapFn) => list.map(mapFn).reduce((result, value, i) => {
  result[`condition${i}`] = value;
  return result;
}, {});
const FIRED_ACTIONS = exports.FIRED_ACTIONS = {
  id: FIRED_ACTIONS_ID,
  name: _i18n.i18n.translate('xpack.infra.metrics.alerting.inventory.threshold.fired', {
    defaultMessage: 'Alert'
  })
};
const WARNING_ACTIONS = exports.WARNING_ACTIONS = {
  id: WARNING_ACTIONS_ID,
  name: _i18n.i18n.translate('xpack.infra.metrics.alerting.threshold.warning', {
    defaultMessage: 'Warning'
  })
};
const translateActionGroupToAlertState = actionGroupId => {
  if (actionGroupId === FIRED_ACTIONS.id) {
    return _messages.stateToAlertMessage[_metrics.AlertStates.ALERT];
  }
  if (actionGroupId === WARNING_ACTIONS.id) {
    return _messages.stateToAlertMessage[_metrics.AlertStates.WARNING];
  }
};
const formatMetric = (metric, value) => {
  const metricFormatter = (0, _lodash.get)(_snapshot_metric_formats.METRIC_FORMATTERS, metric, _snapshot_metric_formats.METRIC_FORMATTERS.count);
  if (isNaN(value)) {
    return _i18n.i18n.translate('xpack.infra.metrics.alerting.inventory.noDataFormattedValue', {
      defaultMessage: '[NO DATA]'
    });
  }
  const formatter = (0, _formatters.createFormatter)(metricFormatter.formatter, metricFormatter.template);
  return formatter(value);
};