"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InferenceTaskErrorCode = exports.InferenceTaskError = void 0;
exports.createInferenceInternalError = createInferenceInternalError;
exports.createInferenceRequestAbortedError = createInferenceRequestAbortedError;
exports.createInferenceRequestError = createInferenceRequestError;
exports.isInferenceError = isInferenceError;
exports.isInferenceInternalError = isInferenceInternalError;
exports.isInferenceRequestAbortedError = isInferenceRequestAbortedError;
exports.isInferenceRequestError = isInferenceRequestError;
var _inference_task = require("./inference_task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/**
 * Enum for generic inference error codes.
 */
let InferenceTaskErrorCode = exports.InferenceTaskErrorCode = /*#__PURE__*/function (InferenceTaskErrorCode) {
  InferenceTaskErrorCode["internalError"] = "internalError";
  InferenceTaskErrorCode["requestError"] = "requestError";
  InferenceTaskErrorCode["abortedError"] = "requestAborted";
  return InferenceTaskErrorCode;
}({});
/**
 * Base class for all inference API errors.
 */
class InferenceTaskError extends Error {
  constructor(code, message, meta) {
    super(message);
    this.code = code;
    this.meta = meta;
  }
  toJSON() {
    return {
      type: _inference_task.InferenceTaskEventType.error,
      error: {
        code: this.code,
        message: this.message,
        meta: this.meta
      }
    };
  }
}

/**
 * Inference error thrown when an unexpected internal error occurs while handling the request.
 */

/**
 * Inference error thrown when the request was considered invalid.
 *
 * Some example of reasons for invalid requests would be:
 * - no connector matching the provided connectorId
 * - invalid connector type for the provided connectorId
 */

/**
 * Inference error thrown when the request was aborted.
 *
 * Request abortion occurs when providing an abort signal and firing it
 * before the call to the LLM completes.
 */
exports.InferenceTaskError = InferenceTaskError;
function createInferenceInternalError(message = 'An internal error occurred', meta) {
  return new InferenceTaskError(InferenceTaskErrorCode.internalError, message, meta !== null && meta !== void 0 ? meta : {});
}
function createInferenceRequestError(message, status) {
  return new InferenceTaskError(InferenceTaskErrorCode.requestError, message, {
    status
  });
}
function createInferenceRequestAbortedError() {
  return new InferenceTaskError(InferenceTaskErrorCode.abortedError, 'Request was aborted', {
    status: 499
  });
}

/**
 * Check if the given error is an {@link InferenceTaskError}
 */
function isInferenceError(error) {
  return error instanceof InferenceTaskError;
}

/**
 * Check if the given error is an {@link InferenceTaskInternalError}
 */
function isInferenceInternalError(error) {
  return isInferenceError(error) && error.code === InferenceTaskErrorCode.internalError;
}

/**
 * Check if the given error is an {@link InferenceTaskRequestError}
 */
function isInferenceRequestError(error) {
  return isInferenceError(error) && error.code === InferenceTaskErrorCode.requestError;
}

/**
 * Check if the given error is an {@link InferenceTaskAbortedError}
 */
function isInferenceRequestAbortedError(error) {
  return isInferenceError(error) && error.code === InferenceTaskErrorCode.abortedError;
}