"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CsvGenerator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _elasticsearch = require("@elastic/elasticsearch");
var _common = require("@kbn/data-plugin/common");
var _reportingCommon = require("@kbn/reporting-common");
var _constants = require("../constants");
var _get_export_settings = require("./lib/get_export_settings");
var _i18n_texts = require("./lib/i18n_texts");
var _max_size_string_builder = require("./lib/max_size_string_builder");
var _search_cursor_pit = require("./lib/search_cursor_pit");
var _search_cursor_scroll = require("./lib/search_cursor_scroll");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class CsvGenerator {
  constructor(job, config, taskInstanceFields, clients, dependencies, cancellationToken, logger, stream) {
    (0, _defineProperty2.default)(this, "csvContainsFormulas", false);
    (0, _defineProperty2.default)(this, "maxSizeReached", false);
    (0, _defineProperty2.default)(this, "csvRowCount", 0);
    this.job = job;
    this.config = config;
    this.taskInstanceFields = taskInstanceFields;
    this.clients = clients;
    this.dependencies = dependencies;
    this.cancellationToken = cancellationToken;
    this.logger = logger;
    this.stream = stream;
  }
  /*
   * Load field formats for each field in the list
   */
  getFormatters(table) {
    // initialize field formats
    const formatters = {};
    table.columns.forEach(c => {
      const fieldFormat = this.dependencies.fieldFormatsRegistry.deserialize(c.meta.params);
      formatters[c.id] = fieldFormat;
    });
    return formatters;
  }
  escapeValues(settings) {
    return value => {
      if (settings.checkForFormulas && (0, _common.cellHasFormulas)(value)) {
        this.csvContainsFormulas = true; // set warning if cell value has a formula
      }
      return settings.escapeValue(value);
    };
  }
  getColumnsFromTabify(table) {
    const columnIds = table.columns.map(c => c.id);
    columnIds.sort();
    return columnIds;
  }
  formatCellValues(formatters) {
    return ({
      column: tableColumn,
      data: dataTableCell
    }) => {
      let cell;
      // check truthiness to guard against _score, _type, etc
      if (tableColumn && dataTableCell) {
        try {
          cell = formatters[tableColumn].convert(dataTableCell);
        } catch (err) {
          this.logger.error(err);
          cell = '-';
        }
        const isIdField = tableColumn === '_id'; // _id field can not be formatted or mutated
        if (!isIdField) {
          try {
            // unwrap the value
            // expected values are a string of JSON where the value(s) is in an array
            // examples: "[""Jan 1, 2020 @ 04:00:00.000""]","[""username""]"
            cell = JSON.parse(cell);
          } catch (e) {
            // ignore
          }
        }

        // We have to strip singular array values out of their array wrapper,
        // So that the value appears the visually the same as seen in Discover
        if (Array.isArray(cell)) {
          cell = cell.map(c => typeof c === 'object' ? JSON.stringify(c) : c).join(', ');
        }

        // Check for object-type value (geoip)
        if (typeof cell === 'object') {
          cell = JSON.stringify(cell);
        }
        return cell;
      }
      return '-'; // Unknown field: it existed in searchSource but has no value in the result
    };
  }

  /*
   * Use the list of columns to generate the header row
   */
  generateHeader(columns, builder, settings, dataView) {
    this.logger.debug(`Building CSV header row`);
    const header = Array.from(columns).map(column => {
      const field = dataView === null || dataView === void 0 ? void 0 : dataView.fields.getByName(column);
      if (field && field.customLabel && field.customLabel !== column) {
        return `${field.customLabel} (${column})`;
      }
      return column;
    }).map(this.escapeValues(settings)).join(settings.separator) + '\n';
    if (!builder.tryAppend(header)) {
      return {
        content: '',
        maxSizeReached: true,
        warnings: []
      };
    }
  }

  /*
   * Format a Datatable into rows of CSV content
   */
  async generateRows(columns, table, builder, formatters, settings) {
    this.logger.debug(`Building ${table.rows.length} CSV data rows`);
    for (const dataTableRow of table.rows) {
      if (this.cancellationToken.isCancelled()) {
        break;
      }

      /*
       * Intrinsically, generating the rows is a synchronous process. Awaiting
       * on a setImmediate call here partitions what could be a very long and
       * CPU-intensive synchronous process into asynchronous processes. This
       * give NodeJS to process other asynchronous events that wait on the Event
       * Loop.
       *
       * See: https://nodejs.org/en/docs/guides/dont-block-the-event-loop/
       *
       * It's likely this creates a lot of context switching, and adds to the
       * time it would take to generate the CSV. There are alternatives to the
       * chosen performance solution:
       *
       * 1. Partition the synchronous process with fewer partitions, by using
       * the loop counter to call setImmediate only every N amount of rows.
       * Testing is required to see what the best N value for most data will
       * be.
       *
       * 2. Use a C++ add-on to generate the CSV using the Node Worker Pool
       * instead of using the Event Loop
       */
      await new Promise(setImmediate);
      const rowDefinition = [];
      const format = this.formatCellValues(formatters);
      const escape = this.escapeValues(settings);
      for (const column of columns) {
        rowDefinition.push(escape(format({
          column,
          data: dataTableRow[column]
        })));
      }
      if (!builder.tryAppend(rowDefinition.join(settings.separator) + '\n')) {
        this.logger.warn(`Max Size Reached after ${this.csvRowCount} rows.`);
        this.maxSizeReached = true;
        if (this.cancellationToken) {
          this.cancellationToken.cancel();
        }
        break;
      }
      this.csvRowCount++;
    }
  }
  async generateData() {
    var _this$job$columns, _reportingError;
    const logger = this.logger;
    const [settings, searchSource] = await Promise.all([(0, _get_export_settings.getExportSettings)(this.clients.uiSettings, this.taskInstanceFields, this.config, this.job.browserTimezone, logger), this.dependencies.searchSourceStart.create(this.job.searchSource)]);
    const {
      startedAt,
      retryAt
    } = this.taskInstanceFields;
    if (startedAt) {
      this.logger.debug(`Task started at: ${startedAt && (0, _moment.default)(startedAt).format()}.` + ` Can run until: ${retryAt && (0, _moment.default)(retryAt).format()}`);
    }
    const index = searchSource.getField('index');
    if (!index) {
      throw new Error(`The search must have a reference to an index pattern!`);
    }
    const {
      maxSizeBytes,
      bom,
      escapeFormulaValues,
      timezone
    } = settings;
    const indexPatternTitle = index.getIndexPattern();
    const builder = new _max_size_string_builder.MaxSizeStringBuilder(this.stream, (0, _reportingCommon.byteSizeValueToNumber)(maxSizeBytes), bom);
    const warnings = [];
    let first = true;
    let currentRecord = -1;
    let totalRecords;
    let reportingError;
    const abortController = new AbortController();
    this.cancellationToken.on(() => abortController.abort());

    // use a class to internalize the paging strategy
    let cursor;
    if (this.job.pagingStrategy === 'scroll') {
      // Optional strategy: scan-and-scroll
      cursor = new _search_cursor_scroll.SearchCursorScroll(indexPatternTitle, settings, this.clients, abortController, this.logger);
      logger.debug('Using search strategy: scroll');
    } else {
      // Default strategy: point-in-time
      cursor = new _search_cursor_pit.SearchCursorPit(indexPatternTitle, settings, this.clients, abortController, this.logger);
      logger.debug('Using search strategy: pit');
    }
    await cursor.initialize();

    // apply timezone from the job to all date field formatters
    try {
      index.fields.getByType('date').forEach(({
        name
      }) => {
        var _index$fieldFormatMap, _index$fieldFormatMap2;
        logger.debug(`Setting timezone on ${name}`);
        const format = {
          ...index.fieldFormatMap[name],
          id: ((_index$fieldFormatMap = index.fieldFormatMap[name]) === null || _index$fieldFormatMap === void 0 ? void 0 : _index$fieldFormatMap.id) || 'date',
          // allow id: date_nanos
          params: {
            ...((_index$fieldFormatMap2 = index.fieldFormatMap[name]) === null || _index$fieldFormatMap2 === void 0 ? void 0 : _index$fieldFormatMap2.params),
            timezone
          }
        };
        index.setFieldFormat(name, format);
      });
    } catch (err) {
      logger.error(err);
    }
    const columns = new Set((_this$job$columns = this.job.columns) !== null && _this$job$columns !== void 0 ? _this$job$columns : []);
    try {
      do {
        var _trackedTotal$value, _this$job$columns2;
        if (this.cancellationToken.isCancelled()) {
          break;
        }
        searchSource.setField('size', settings.scroll.size);
        let results;
        try {
          results = await cursor.getPage(searchSource);
        } catch (err) {
          this.logger.error(`CSV export search error: ${err}`);
          throw err;
        }
        if (!results) {
          logger.warn(`Search results are undefined!`);
          break;
        }
        const {
          total
        } = results.hits;
        const trackedTotal = total;
        const currentTotal = (_trackedTotal$value = trackedTotal === null || trackedTotal === void 0 ? void 0 : trackedTotal.value) !== null && _trackedTotal$value !== void 0 ? _trackedTotal$value : total;
        if (first) {
          // export stops when totalRecords have been accumulated (or the results have run out)
          totalRecords = currentTotal;
        }

        // use the most recently received cursor id for the next search request
        cursor.updateIdFromResults(results);

        // check for shard failures, log them and add a warning if found
        const {
          _shards: shards
        } = results;
        if (shards.failures) {
          shards.failures.forEach(({
            reason
          }) => {
            warnings.push(`Shard failure: ${JSON.stringify(reason)}`);
            logger.warn(JSON.stringify(reason));
          });
        }
        let table;
        try {
          table = (0, _common.tabifyDocs)(results, index, {
            shallow: true,
            includeIgnoredValues: true
          });
        } catch (err) {
          var _err$message;
          logger.error(err);
          warnings.push(_i18n_texts.i18nTexts.unknownError((_err$message = err === null || err === void 0 ? void 0 : err.message) !== null && _err$message !== void 0 ? _err$message : err));
        }
        if (!table) {
          break;
        }
        if (!((_this$job$columns2 = this.job.columns) !== null && _this$job$columns2 !== void 0 && _this$job$columns2.length)) {
          this.getColumnsFromTabify(table).forEach(column => columns.add(column));
        }
        if (first) {
          first = false;
          this.generateHeader(columns, builder, settings, index);
        }
        if (table.rows.length < 1) {
          break; // empty report with just the header
        }

        // FIXME: make tabifyDocs handle the formatting, to get the same formatting logic as Discover?
        const formatters = this.getFormatters(table);
        await this.generateRows(columns, table, builder, formatters, settings);

        // update iterator
        currentRecord += table.rows.length;
      } while (totalRecords != null && currentRecord < totalRecords - 1);

      // Add warnings to be logged
      if (this.csvContainsFormulas && escapeFormulaValues) {
        warnings.push(_i18n_texts.i18nTexts.escapedFormulaValuesMessage);
      }
    } catch (err) {
      logger.error(err);
      if (err instanceof _elasticsearch.errors.ResponseError) {
        var _err$statusCode;
        if ([401, 403].includes((_err$statusCode = err.statusCode) !== null && _err$statusCode !== void 0 ? _err$statusCode : 0)) {
          reportingError = new _reportingCommon.AuthenticationExpiredError();
          warnings.push(_i18n_texts.i18nTexts.authenticationError.partialResultsMessage);
        } else {
          var _err$statusCode2;
          warnings.push(_i18n_texts.i18nTexts.esErrorMessage((_err$statusCode2 = err.statusCode) !== null && _err$statusCode2 !== void 0 ? _err$statusCode2 : 0, String(err.body)));
        }
      } else {
        var _err$message2;
        warnings.push(_i18n_texts.i18nTexts.unknownError((_err$message2 = err === null || err === void 0 ? void 0 : err.message) !== null && _err$message2 !== void 0 ? _err$message2 : err));
      }
    } finally {
      try {
        await cursor.closeCursor();
      } catch (err) {
        logger.error(err);
        warnings.push(cursor.getUnableToCloseCursorMessage());
      }
    }
    logger.info(`Finished generating. Row count: ${this.csvRowCount}.`);
    if (!this.maxSizeReached && this.csvRowCount !== totalRecords) {
      var _totalRecords;
      logger.warn(`ES scroll returned ` + `${this.csvRowCount > ((_totalRecords = totalRecords) !== null && _totalRecords !== void 0 ? _totalRecords : 0) ? 'more' : 'fewer'} total hits than expected!`);
      logger.warn(`Search result total hits: ${totalRecords}. Row count: ${this.csvRowCount}`);
      if (totalRecords || totalRecords === 0) {
        warnings.push(_i18n_texts.i18nTexts.csvRowCountError({
          expected: totalRecords,
          received: this.csvRowCount
        }));
      } else {
        warnings.push(_i18n_texts.i18nTexts.csvRowCountIndeterminable({
          received: this.csvRowCount
        }));
      }
    }
    if (this.csvRowCount === 0) {
      if (warnings.length > 0) {
        /*
         * Add the errors into the CSV content. This makes error messages more
         * discoverable. When the export was automated or triggered by an API
         * call or is automated, the user doesn't necessarily go through the
         * Kibana UI to download the export and might not otherwise see the
         * error message.
         */
        logger.info('CSV export content was empty. Adding error messages to CSV export content.');
        // join the string array and putting double quotes around each item
        // add a leading newline so the first message is not treated as a header
        builder.tryAppend('\n"' + warnings.join('"\n"') + '"');
      } else {
        logger.info('CSV export content was empty. No error messages.');
      }
    }
    return {
      content_type: _constants.CONTENT_TYPE_CSV,
      csv_contains_formulas: this.csvContainsFormulas && !escapeFormulaValues,
      max_size_reached: this.maxSizeReached,
      metrics: {
        csv: {
          rows: this.csvRowCount
        }
      },
      warnings,
      error_code: (_reportingError = reportingError) === null || _reportingError === void 0 ? void 0 : _reportingError.code
    };
  }
}
exports.CsvGenerator = CsvGenerator;