"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cleanPreconfiguredOutputs = cleanPreconfiguredOutputs;
exports.createOrUpdatePreconfiguredOutputs = createOrUpdatePreconfiguredOutputs;
exports.ensurePreconfiguredOutputs = ensurePreconfiguredOutputs;
exports.getPreconfiguredOutputFromConfig = getPreconfiguredOutputFromConfig;
exports.hashSecret = hashSecret;
var _nodeCrypto = _interopRequireDefault(require("node:crypto"));
var _nodeUtil = _interopRequireDefault(require("node:util"));
var _lodash = require("lodash");
var _jsYaml = require("js-yaml");
var _services = require("../../../common/services");
var _constants = require("../../constants");
var _output = require("../output");
var _agent_policy = require("../agent_policy");
var _app_context = require("../app_context");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const pbkdf2Async = _nodeUtil.default.promisify(_nodeCrypto.default.pbkdf2);
function getPreconfiguredOutputFromConfig(config) {
  const {
    outputs: outputsOrUndefined
  } = config;
  const outputs = (outputsOrUndefined || []).concat([...(config !== null && config !== void 0 && config.agents.elasticsearch.hosts ? [{
    ..._constants.DEFAULT_OUTPUT,
    id: _constants.DEFAULT_OUTPUT_ID,
    hosts: config === null || config === void 0 ? void 0 : config.agents.elasticsearch.hosts,
    ca_sha256: config === null || config === void 0 ? void 0 : config.agents.elasticsearch.ca_sha256,
    ca_trusted_fingerprint: config === null || config === void 0 ? void 0 : config.agents.elasticsearch.ca_trusted_fingerprint,
    is_preconfigured: true
  }] : [])]);
  return outputs;
}
async function ensurePreconfiguredOutputs(soClient, esClient, outputs) {
  await createOrUpdatePreconfiguredOutputs(soClient, esClient, outputs);
  await cleanPreconfiguredOutputs(soClient, esClient, outputs);
}
async function createOrUpdatePreconfiguredOutputs(soClient, esClient, outputs) {
  const logger = _app_context.appContextService.getLogger();
  if (outputs.length === 0) {
    return;
  }
  const existingOutputs = await _output.outputService.bulkGet(soClient, outputs.map(({
    id
  }) => id), {
    ignoreNotFound: true
  });
  await Promise.all(outputs.map(async output => {
    var _outputData$ca_sha, _outputData$ca_truste, _outputData$ssl;
    const existingOutput = existingOutputs.find(o => o.id === output.id);
    const {
      id,
      config,
      ...outputData
    } = output;
    const configYaml = config ? (0, _jsYaml.safeDump)(config) : undefined;
    const data = {
      ...outputData,
      is_preconfigured: true,
      config_yaml: configYaml !== null && configYaml !== void 0 ? configYaml : null,
      // Set value to null to update these fields on update
      ca_sha256: (_outputData$ca_sha = outputData.ca_sha256) !== null && _outputData$ca_sha !== void 0 ? _outputData$ca_sha : null,
      ca_trusted_fingerprint: (_outputData$ca_truste = outputData.ca_trusted_fingerprint) !== null && _outputData$ca_truste !== void 0 ? _outputData$ca_truste : null,
      ssl: (_outputData$ssl = outputData.ssl) !== null && _outputData$ssl !== void 0 ? _outputData$ssl : null
    };
    if (!data.hosts || data.hosts.length === 0) {
      data.hosts = _output.outputService.getDefaultESHosts();
    }
    const isCreate = !existingOutput;

    // field in allow edit are not updated through preconfiguration
    if (!isCreate && output.allow_edit) {
      for (const key of output.allow_edit) {
        // @ts-expect-error
        data[key] = existingOutput[key];
      }
    }
    const isUpdateWithNewData = existingOutput && (await isPreconfiguredOutputDifferentFromCurrent(existingOutput, data));
    if (isCreate || isUpdateWithNewData) {
      const secretHashes = await hashSecrets(output);
      if (isCreate) {
        logger.debug(`Creating preconfigured output ${output.id}`);
        await _output.outputService.create(soClient, esClient, data, {
          id,
          fromPreconfiguration: true,
          secretHashes
        });
      } else if (isUpdateWithNewData) {
        logger.debug(`Updating preconfigured output ${output.id}`);
        await _output.outputService.update(soClient, esClient, id, data, {
          fromPreconfiguration: true,
          secretHashes
        });
        // Bump revision of all policies using that output
        if (outputData.is_default || outputData.is_default_monitoring) {
          await _agent_policy.agentPolicyService.bumpAllAgentPolicies(esClient);
        } else {
          await _agent_policy.agentPolicyService.bumpAllAgentPoliciesForOutput(esClient, id);
        }
      }
    }
  }));
}

// Values recommended by NodeJS documentation
const keyLength = 64;
const saltLength = 16;
const maxIteration = 100000;
async function hashSecret(secret) {
  const salt = _nodeCrypto.default.randomBytes(saltLength).toString('hex');
  const derivedKey = await pbkdf2Async(secret, salt, maxIteration, keyLength, 'sha512');
  return `${salt}:${derivedKey.toString('hex')}`;
}
async function verifySecret(hash, secret) {
  const [salt, key] = hash.split(':');
  const derivedKey = await pbkdf2Async(secret, salt, maxIteration, keyLength, 'sha512');
  const keyBuffer = Buffer.from(key, 'hex');
  if (keyBuffer.length !== derivedKey.length) {
    return false;
  }
  return _nodeCrypto.default.timingSafeEqual(Buffer.from(key, 'hex'), derivedKey);
}
async function hashSecrets(output) {
  if (output.type === 'kafka') {
    var _kafkaOutput$secrets, _kafkaOutput$secrets3, _kafkaOutput$secrets4;
    const kafkaOutput = output;
    if (typeof ((_kafkaOutput$secrets = kafkaOutput.secrets) === null || _kafkaOutput$secrets === void 0 ? void 0 : _kafkaOutput$secrets.password) === 'string') {
      var _kafkaOutput$secrets2;
      const password = await hashSecret((_kafkaOutput$secrets2 = kafkaOutput.secrets) === null || _kafkaOutput$secrets2 === void 0 ? void 0 : _kafkaOutput$secrets2.password);
      return {
        password
      };
    }
    if (typeof ((_kafkaOutput$secrets3 = kafkaOutput.secrets) === null || _kafkaOutput$secrets3 === void 0 ? void 0 : (_kafkaOutput$secrets4 = _kafkaOutput$secrets3.ssl) === null || _kafkaOutput$secrets4 === void 0 ? void 0 : _kafkaOutput$secrets4.key) === 'string') {
      var _kafkaOutput$secrets5, _kafkaOutput$secrets6;
      const key = await hashSecret((_kafkaOutput$secrets5 = kafkaOutput.secrets) === null || _kafkaOutput$secrets5 === void 0 ? void 0 : (_kafkaOutput$secrets6 = _kafkaOutput$secrets5.ssl) === null || _kafkaOutput$secrets6 === void 0 ? void 0 : _kafkaOutput$secrets6.key);
      return {
        ssl: {
          key
        }
      };
    }
  }
  if (output.type === 'logstash') {
    var _logstashOutput$secre, _logstashOutput$secre2;
    const logstashOutput = output;
    if (typeof ((_logstashOutput$secre = logstashOutput.secrets) === null || _logstashOutput$secre === void 0 ? void 0 : (_logstashOutput$secre2 = _logstashOutput$secre.ssl) === null || _logstashOutput$secre2 === void 0 ? void 0 : _logstashOutput$secre2.key) === 'string') {
      var _logstashOutput$secre3, _logstashOutput$secre4;
      const key = await hashSecret((_logstashOutput$secre3 = logstashOutput.secrets) === null || _logstashOutput$secre3 === void 0 ? void 0 : (_logstashOutput$secre4 = _logstashOutput$secre3.ssl) === null || _logstashOutput$secre4 === void 0 ? void 0 : _logstashOutput$secre4.key);
      return {
        ssl: {
          key
        }
      };
    }
  }
  if (output.type === 'remote_elasticsearch') {
    var _remoteESOutput$secre;
    const remoteESOutput = output;
    if (typeof ((_remoteESOutput$secre = remoteESOutput.secrets) === null || _remoteESOutput$secre === void 0 ? void 0 : _remoteESOutput$secre.service_token) === 'string') {
      var _remoteESOutput$secre2;
      const serviceToken = await hashSecret((_remoteESOutput$secre2 = remoteESOutput.secrets) === null || _remoteESOutput$secre2 === void 0 ? void 0 : _remoteESOutput$secre2.service_token);
      return {
        service_token: serviceToken
      };
    }
  }
  return undefined;
}
async function cleanPreconfiguredOutputs(soClient, esClient, outputs) {
  const existingOutputs = await _output.outputService.list(soClient);
  const existingPreconfiguredOutput = existingOutputs.items.filter(o => o.is_preconfigured === true);
  const logger = _app_context.appContextService.getLogger();
  for (const output of existingPreconfiguredOutput) {
    const hasBeenDelete = !outputs.find(({
      id
    }) => output.id === id);
    if (!hasBeenDelete) {
      continue;
    }
    if (output.is_default) {
      logger.info(`Updating default preconfigured output ${output.id} is no longer preconfigured`);
      await _output.outputService.update(soClient, esClient, output.id, {
        is_preconfigured: false
      }, {
        fromPreconfiguration: true
      });
    } else if (output.is_default_monitoring) {
      logger.info(`Updating default preconfigured output ${output.id} is no longer preconfigured`);
      await _output.outputService.update(soClient, esClient, output.id, {
        is_preconfigured: false
      }, {
        fromPreconfiguration: true
      });
    } else {
      logger.info(`Deleting preconfigured output ${output.id}`);
      await _output.outputService.delete(soClient, output.id, {
        fromPreconfiguration: true
      });
    }
  }
}
const hasHash = secret => {
  return !!secret && typeof secret !== 'string' && !!secret.hash;
};
async function isSecretDifferent(preconfiguredValue, existingSecret) {
  if (!existingSecret && preconfiguredValue) {
    return true;
  }
  if (!preconfiguredValue && existingSecret) {
    return true;
  }
  if (!preconfiguredValue && !existingSecret) {
    return false;
  }
  if (hasHash(existingSecret) && typeof preconfiguredValue === 'string') {
    // verifying the has tells us if the value has changed
    const hashIsVerified = await verifySecret(existingSecret.hash, preconfiguredValue);
    return !hashIsVerified;
  } else {
    // if there is no hash then the safest thing to do is assume the value has changed
    return true;
  }
}
async function isPreconfiguredOutputDifferentFromCurrent(existingOutput, preconfiguredOutput) {
  var _existingOutput$hosts, _existingOutput$allow, _preconfiguredOutput$7;
  const kafkaFieldsAreDifferent = async () => {
    var _preconfiguredOutput$, _existingOutput$secre, _preconfiguredOutput$2, _preconfiguredOutput$3, _existingOutput$secre2, _existingOutput$secre3;
    if (existingOutput.type !== 'kafka' || preconfiguredOutput.type !== 'kafka') {
      return false;
    }
    const passwordHashIsDifferent = await isSecretDifferent((_preconfiguredOutput$ = preconfiguredOutput.secrets) === null || _preconfiguredOutput$ === void 0 ? void 0 : _preconfiguredOutput$.password, (_existingOutput$secre = existingOutput.secrets) === null || _existingOutput$secre === void 0 ? void 0 : _existingOutput$secre.password);
    const sslKeyHashIsDifferent = await isSecretDifferent((_preconfiguredOutput$2 = preconfiguredOutput.secrets) === null || _preconfiguredOutput$2 === void 0 ? void 0 : (_preconfiguredOutput$3 = _preconfiguredOutput$2.ssl) === null || _preconfiguredOutput$3 === void 0 ? void 0 : _preconfiguredOutput$3.key, (_existingOutput$secre2 = existingOutput.secrets) === null || _existingOutput$secre2 === void 0 ? void 0 : (_existingOutput$secre3 = _existingOutput$secre2.ssl) === null || _existingOutput$secre3 === void 0 ? void 0 : _existingOutput$secre3.key);
    return (0, _utils.isDifferent)(existingOutput.client_id, preconfiguredOutput.client_id) || (0, _utils.isDifferent)(existingOutput.version, preconfiguredOutput.version) || (0, _utils.isDifferent)(existingOutput.key, preconfiguredOutput.key) || (0, _utils.isDifferent)(existingOutput.compression, preconfiguredOutput.compression) || (0, _utils.isDifferent)(existingOutput.compression_level, preconfiguredOutput.compression_level) || (0, _utils.isDifferent)(existingOutput.auth_type, preconfiguredOutput.auth_type) || (0, _utils.isDifferent)(existingOutput.connection_type, preconfiguredOutput.connection_type) || (0, _utils.isDifferent)(existingOutput.username, preconfiguredOutput.username) || (0, _utils.isDifferent)(existingOutput.password, preconfiguredOutput.password) || (0, _utils.isDifferent)(existingOutput.sasl, preconfiguredOutput.sasl) || (0, _utils.isDifferent)(existingOutput.partition, preconfiguredOutput.partition) || (0, _utils.isDifferent)(existingOutput.random, preconfiguredOutput.random) || (0, _utils.isDifferent)(existingOutput.round_robin, preconfiguredOutput.round_robin) || (0, _utils.isDifferent)(existingOutput.hash, preconfiguredOutput.hash) || (0, _utils.isDifferent)(existingOutput.topic, preconfiguredOutput.topic) || (0, _utils.isDifferent)(existingOutput.topics, preconfiguredOutput.topics) || (0, _utils.isDifferent)(existingOutput.headers, preconfiguredOutput.headers) || (0, _utils.isDifferent)(existingOutput.timeout, preconfiguredOutput.timeout) || (0, _utils.isDifferent)(existingOutput.broker_timeout, preconfiguredOutput.broker_timeout) || (0, _utils.isDifferent)(existingOutput.required_acks, preconfiguredOutput.required_acks) || passwordHashIsDifferent || sslKeyHashIsDifferent;
  };
  const logstashFieldsAreDifferent = async () => {
    var _preconfiguredOutput$4, _preconfiguredOutput$5, _existingOutput$secre4, _existingOutput$secre5;
    if (existingOutput.type !== 'logstash' || preconfiguredOutput.type !== 'logstash') {
      return false;
    }
    const sslKeyHashIsDifferent = await isSecretDifferent((_preconfiguredOutput$4 = preconfiguredOutput.secrets) === null || _preconfiguredOutput$4 === void 0 ? void 0 : (_preconfiguredOutput$5 = _preconfiguredOutput$4.ssl) === null || _preconfiguredOutput$5 === void 0 ? void 0 : _preconfiguredOutput$5.key, (_existingOutput$secre4 = existingOutput.secrets) === null || _existingOutput$secre4 === void 0 ? void 0 : (_existingOutput$secre5 = _existingOutput$secre4.ssl) === null || _existingOutput$secre5 === void 0 ? void 0 : _existingOutput$secre5.key);
    return sslKeyHashIsDifferent;
  };
  const remoteESFieldsAreDifferent = async () => {
    var _preconfiguredOutput$6, _existingOutput$secre6;
    if (existingOutput.type !== 'remote_elasticsearch' || preconfiguredOutput.type !== 'remote_elasticsearch') {
      return false;
    }
    const serviceTokenIsDifferent = await isSecretDifferent((_preconfiguredOutput$6 = preconfiguredOutput.secrets) === null || _preconfiguredOutput$6 === void 0 ? void 0 : _preconfiguredOutput$6.service_token, (_existingOutput$secre6 = existingOutput.secrets) === null || _existingOutput$secre6 === void 0 ? void 0 : _existingOutput$secre6.service_token);
    return serviceTokenIsDifferent;
  };
  return !existingOutput.is_preconfigured || (0, _utils.isDifferent)(existingOutput.is_default, preconfiguredOutput.is_default) || (0, _utils.isDifferent)(existingOutput.is_default_monitoring, preconfiguredOutput.is_default_monitoring) || (0, _utils.isDifferent)(existingOutput.name, preconfiguredOutput.name) || (0, _utils.isDifferent)(existingOutput.type, preconfiguredOutput.type) || preconfiguredOutput.hosts && !(0, _lodash.isEqual)((existingOutput === null || existingOutput === void 0 ? void 0 : existingOutput.type) === 'elasticsearch' ? (_existingOutput$hosts = existingOutput.hosts) === null || _existingOutput$hosts === void 0 ? void 0 : _existingOutput$hosts.map(_services.normalizeHostsForAgents) : existingOutput.hosts, preconfiguredOutput.type === 'elasticsearch' ? preconfiguredOutput.hosts.map(_services.normalizeHostsForAgents) : preconfiguredOutput.hosts) || (0, _utils.isDifferent)(preconfiguredOutput.ssl, existingOutput.ssl) || (0, _utils.isDifferent)(existingOutput.ca_sha256, preconfiguredOutput.ca_sha256) || (0, _utils.isDifferent)(existingOutput.ca_trusted_fingerprint, preconfiguredOutput.ca_trusted_fingerprint) || (0, _utils.isDifferent)(existingOutput.config_yaml, preconfiguredOutput.config_yaml) || (0, _utils.isDifferent)(existingOutput.proxy_id, preconfiguredOutput.proxy_id) || (0, _utils.isDifferent)((_existingOutput$allow = existingOutput.allow_edit) !== null && _existingOutput$allow !== void 0 ? _existingOutput$allow : [], (_preconfiguredOutput$7 = preconfiguredOutput.allow_edit) !== null && _preconfiguredOutput$7 !== void 0 ? _preconfiguredOutput$7 : []) || preconfiguredOutput.preset && (0, _utils.isDifferent)(existingOutput.preset, preconfiguredOutput.preset) || (0, _utils.isDifferent)(existingOutput.is_internal, preconfiguredOutput.is_internal) || (await kafkaFieldsAreDifferent()) || (await logstashFieldsAreDifferent()) || (await remoteESFieldsAreDifferent());
}