"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.TYPE = exports.FleetMetricsTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _task = require("@kbn/task-manager-plugin/server/task");
var _apmUtils = require("@kbn/apm-utils");
var _lodash = require("lodash");
var _app_context = require("../app_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'Fleet-Metrics-Task';
const VERSION = exports.VERSION = '1.1.1';
const TITLE = 'Fleet Metrics Task';
const TIMEOUT = '1m';
const SCOPE = ['fleet'];
const INTERVAL = '1m';
class FleetMetricsTask {
  constructor(taskManager, _fetchAgentMetrics) {
    (0, _defineProperty2.default)(this, "taskManager", void 0);
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, fetchAgentMetrics) => {
      if (!this.wasStarted) {
        _app_context.appContextService.getLogger().debug('[runTask()] Aborted. Task not started yet');
        return;
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        _app_context.appContextService.getLogger().info(`Outdated task version: Got [${taskInstance.id}] from task instance. Current version is [${this.taskId}]`);
        return (0, _task.getDeleteTaskRunResult)();
      }
      if (!this.esClient) {
        _app_context.appContextService.getLogger().debug('esClient not set, skipping Fleet metrics task');
        return;
      }
      _app_context.appContextService.getLogger().debug('Running Fleet metrics task');
      try {
        const agentMetrics = await fetchAgentMetrics();
        if (!agentMetrics) {
          return;
        }
        const {
          agents_per_version: agentsPerVersion,
          agents
        } = agentMetrics;
        const clusterInfo = await this.esClient.info();
        const getCommonFields = dataset => {
          var _clusterInfo$cluster_;
          return {
            data_stream: {
              dataset,
              type: 'metrics',
              namespace: 'default'
            },
            agent: {
              id: _app_context.appContextService.getKibanaInstanceId(),
              version: _app_context.appContextService.getKibanaVersion(),
              type: 'kibana'
            },
            cluster: {
              id: (_clusterInfo$cluster_ = clusterInfo === null || clusterInfo === void 0 ? void 0 : clusterInfo.cluster_uuid) !== null && _clusterInfo$cluster_ !== void 0 ? _clusterInfo$cluster_ : ''
            }
          };
        };
        const agentStatusDoc = {
          '@timestamp': new Date().toISOString(),
          ...getCommonFields('fleet_server.agent_status'),
          fleet: {
            agents: {
              total: agents.total_all_statuses,
              enrolled: agents.total_enrolled,
              unenrolled: agents.unenrolled,
              healthy: agents.healthy,
              offline: agents.offline,
              updating: agents.updating,
              unhealthy: agents.unhealthy,
              inactive: agents.inactive,
              upgrading_step: agentMetrics.upgrading_step,
              unhealthy_reason: agentMetrics.unhealthy_reason
            }
          }
        };
        _app_context.appContextService.getLogger().trace('Agent status metrics: ' + JSON.stringify(agentStatusDoc));
        await this.esClient.index({
          index: 'metrics-fleet_server.agent_status-default',
          body: agentStatusDoc,
          refresh: true
        });
        if (agentsPerVersion.length === 0) return;
        const operations = [];
        for (const byVersion of agentsPerVersion) {
          const agentVersionsDoc = {
            '@timestamp': new Date().toISOString(),
            ...getCommonFields('fleet_server.agent_versions'),
            fleet: {
              agent: {
                version: byVersion.version,
                count: byVersion.count
              }
            }
          };
          operations.push({
            create: {}
          }, agentVersionsDoc);
        }
        _app_context.appContextService.getLogger().trace('Agent versions metrics: ' + JSON.stringify(operations));
        const resp = await this.esClient.bulk({
          operations,
          refresh: true,
          index: 'metrics-fleet_server.agent_versions-default'
        });
        if (resp.errors) {
          const errors = (0, _lodash.uniq)(resp.items.filter(item => {
            var _item$create;
            return !!((_item$create = item.create) !== null && _item$create !== void 0 && _item$create.error);
          }).map(item => {
            var _item$create$error$re, _item$create2, _item$create2$error;
            return (_item$create$error$re = (_item$create2 = item.create) === null || _item$create2 === void 0 ? void 0 : (_item$create2$error = _item$create2.error) === null || _item$create2$error === void 0 ? void 0 : _item$create2$error.reason) !== null && _item$create$error$re !== void 0 ? _item$create$error$re : '';
          }));
          throw new Error(errors.join(', '));
        }
      } catch (error) {
        _app_context.appContextService.getLogger().warn('Error occurred while publishing Fleet metrics: ' + error);
      }
    });
    taskManager.registerTaskDefinitions({
      [TYPE]: {
        title: TITLE,
        timeout: TIMEOUT,
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return (0, _apmUtils.withSpan)({
                name: TYPE,
                type: 'metrics'
              }, () => this.runTask(taskInstance, () => _fetchAgentMetrics(this.abortController)));
            },
            cancel: async () => {
              this.abortController.abort('task timed out');
            }
          };
        }
      }
    });
  }
  get taskId() {
    return `${TYPE}:${VERSION}`;
  }
  async start(taskManager, esClient) {
    this.taskManager = taskManager;
    this.esClient = esClient;
    if (!taskManager) {
      _app_context.appContextService.getLogger().error('missing required service during start');
      return;
    }
    this.wasStarted = true;
    try {
      _app_context.appContextService.getLogger().info(`Task ${this.taskId} scheduled with interval ${INTERVAL}`);
      await this.taskManager.ensureScheduled({
        id: this.taskId,
        taskType: TYPE,
        schedule: {
          interval: INTERVAL
        },
        scope: SCOPE,
        state: {},
        params: {}
      });
    } catch (e) {
      _app_context.appContextService.getLogger().error(`Error scheduling task, received error: ${e}`);
    }
  }
}
exports.FleetMetricsTask = FleetMetricsTask;