"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.storedPackagePolicyToAgentInputs = exports.storedPackagePoliciesToAgentInputs = exports.mergeInputsOverrides = exports.getFullInputStreams = void 0;
var _lodash = require("lodash");
var _deepmerge = _interopRequireDefault(require("deepmerge"));
var _agentless_policy_helper = require("../../../common/services/agentless_policy_helper");
var _services = require("../../../common/services");
var _constants = require("../../constants");
var _registry = require("../epm/registry");
var _epm = require("../../../common/constants/epm");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isPolicyEnabled = packagePolicy => {
  return packagePolicy.enabled && packagePolicy.inputs && packagePolicy.inputs.length;
};
const storedPackagePolicyToAgentInputs = (packagePolicy, packageInfo, agentPolicyOutputId = _constants.DEFAULT_OUTPUT.name, agentPolicyNamespace, addFields) => {
  const fullInputs = [];
  if (!isPolicyEnabled(packagePolicy)) {
    return fullInputs;
  }

  // Marks to skip appending input information to package policy ID to make it unique if package is "limited":
  // this means that only one policy for the package can exist on the agent policy, so its ID is already unique
  const appendInputId = packageInfo && (0, _services.isPackageLimited)(packageInfo) ? false : true;
  packagePolicy.inputs.forEach(input => {
    if (!input.enabled) {
      return;
    }
    const inputId = appendInputId ? `${input.type}${input.policy_template ? `-${input.policy_template}-` : '-'}${packagePolicy.id}` : packagePolicy.id;
    const fullInput = {
      // @ts-ignore-next-line the following id is actually one level above the one in fullInputStream, but the linter thinks it gets overwritten
      id: inputId,
      revision: packagePolicy.revision,
      name: packagePolicy.name,
      type: input.type,
      // @ts-ignore-next-line
      data_stream: {
        namespace: (packagePolicy === null || packagePolicy === void 0 ? void 0 : packagePolicy.namespace) || agentPolicyNamespace || 'default' // custom namespace has precedence on agent policy's one
      },
      use_output: packagePolicy.output_id || agentPolicyOutputId,
      package_policy_id: packagePolicy.id,
      ...getFullInputStreams(input)
    };
    if (addFields && !_epm.GLOBAL_DATA_TAG_EXCLUDED_INPUTS.has(fullInput.type)) {
      fullInput.processors = [addFields];
    }

    // deeply merge the input.config values with the full policy input
    (0, _lodash.merge)(fullInput, Object.entries(input.config || {}).reduce((acc, [key, {
      value
    }]) => {
      acc[key] = value;
      return acc;
    }, {}));
    if (packagePolicy.package) {
      fullInput.meta = {
        package: {
          name: packagePolicy.package.name,
          version: packagePolicy.package.version
        }
      };
    }
    const fullInputWithOverrides = mergeInputsOverrides(packagePolicy, fullInput);
    fullInputs.push(fullInputWithOverrides);
  });
  return fullInputs;
};
exports.storedPackagePolicyToAgentInputs = storedPackagePolicyToAgentInputs;
const mergeInputsOverrides = (packagePolicy, fullInput) => {
  var _packagePolicy$overri;
  // check if there are inputs overrides and merge them
  if (packagePolicy !== null && packagePolicy !== void 0 && (_packagePolicy$overri = packagePolicy.overrides) !== null && _packagePolicy$overri !== void 0 && _packagePolicy$overri.inputs) {
    const overrideInputs = packagePolicy.overrides.inputs;
    const keys = Object.keys(overrideInputs);
    if (keys.length > 0 && fullInput.id === keys[0]) {
      return (0, _deepmerge.default)(fullInput, overrideInputs[keys[0]]);
    }
  }
  return fullInput;
};
exports.mergeInputsOverrides = mergeInputsOverrides;
const getFullInputStreams = (input, allStreamEnabled = false, streamsOriginalIdsMap) => {
  return {
    ...(input.compiled_input || {}),
    ...(input.streams.length ? {
      streams: input.streams.filter(stream => stream.enabled || allStreamEnabled).map(stream => {
        const streamId = stream.id;
        const fullStream = {
          id: streamId,
          data_stream: stream.data_stream,
          ...stream.compiled_stream,
          ...Object.entries(stream.config || {}).reduce((acc, [key, {
            value
          }]) => {
            acc[key] = value;
            return acc;
          }, {})
        };
        streamsOriginalIdsMap === null || streamsOriginalIdsMap === void 0 ? void 0 : streamsOriginalIdsMap.set(fullStream.id, streamId);
        return fullStream;
      })
    } : {})
  };
};
exports.getFullInputStreams = getFullInputStreams;
const storedPackagePoliciesToAgentInputs = async (packagePolicies, packageInfoCache, agentPolicyOutputId = _constants.DEFAULT_OUTPUT.name, agentPolicyNamespace, globalDataTags) => {
  const fullInputs = [];
  for (const packagePolicy of packagePolicies) {
    if (!isPolicyEnabled(packagePolicy)) {
      continue;
    }
    const packageInfo = packagePolicy.package ? packageInfoCache.get((0, _registry.pkgToPkgKey)(packagePolicy.package)) : undefined;
    const filteredGlobalDataTags = filterGlobalDataTags(globalDataTags, packageInfo);
    const addFields = filteredGlobalDataTags && filteredGlobalDataTags.length > 0 ? globalDataTagsToAddFields(filteredGlobalDataTags) : undefined;
    fullInputs.push(...storedPackagePolicyToAgentInputs(packagePolicy, packageInfo, agentPolicyOutputId, agentPolicyNamespace, addFields));
  }
  return fullInputs;
};
exports.storedPackagePoliciesToAgentInputs = storedPackagePoliciesToAgentInputs;
const globalDataTagsToAddFields = tags => {
  const fields = {};
  tags.forEach(tag => {
    fields[tag.name] = tag.value;
  });
  return {
    add_fields: {
      target: '',
      fields
    }
  };
};
const filterGlobalDataTags = (globalDataTags, packageInfo) => {
  if (!globalDataTags) {
    return globalDataTags;
  }
  const agentlessGlobalDataTags = (0, _agentless_policy_helper.getAgentlessGlobalDataTags)(packageInfo);
  if (!agentlessGlobalDataTags) {
    return globalDataTags;
  }
  return globalDataTags.filter(globalDataTag => {
    return !agentlessGlobalDataTags.some(({
      name,
      value
    }) => name === globalDataTag.name && value === globalDataTag.value);
  });
};