"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateFleetConfig = generateFleetConfig;
exports.getFullAgentPolicy = getFullAgentPolicy;
exports.getFullMonitoringSettings = getFullMonitoringSettings;
exports.transformOutputToFullPolicyOutput = transformOutputToFullPolicyOutput;
var _jsYaml = require("js-yaml");
var _deepmerge = _interopRequireDefault(require("deepmerge"));
var _saferLodashSet = require("@kbn/safer-lodash-set");
var _output_helpers = require("../../../common/services/output_helpers");
var _agent_policy = require("../agent_policy");
var _constants = require("../../../common/constants");
var _form_settings = require("../form_settings");
var _packages = require("../epm/packages");
var _registry = require("../epm/registry");
var _app_context = require("../app_context");
var _secrets = require("../secrets");
var _monitoring_permissions = require("./monitoring_permissions");
var _ = require(".");
var _package_policies_to_agent_permissions = require("./package_policies_to_agent_permissions");
var _related_saved_objects = require("./related_saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/* eslint-disable @typescript-eslint/naming-convention */

async function fetchAgentPolicy(soClient, id) {
  try {
    return await _agent_policy.agentPolicyService.get(soClient, id);
  } catch (err) {
    if (!err.isBoom || err.output.statusCode !== 404) {
      throw err;
    }
  }
  return null;
}
async function getFullAgentPolicy(soClient, id, options) {
  var _options$standalone, _options$agentPolicy, _agentPolicy, _agentPolicy$monitori, _agentPolicy$monitori2, _agentPolicy$monitori3, _agentPolicy$monitori4, _agentPolicy$monitori5, _agentPolicy$monitori6;
  const standalone = (_options$standalone = options === null || options === void 0 ? void 0 : options.standalone) !== null && _options$standalone !== void 0 ? _options$standalone : false;
  let agentPolicy;
  if (options !== null && options !== void 0 && (_options$agentPolicy = options.agentPolicy) !== null && _options$agentPolicy !== void 0 && _options$agentPolicy.package_policies) {
    agentPolicy = options.agentPolicy;
  } else {
    agentPolicy = await fetchAgentPolicy(soClient, id);
  }
  if (!agentPolicy) {
    return null;
  }
  const {
    outputs,
    proxies,
    dataOutput,
    fleetServerHosts,
    monitoringOutput,
    downloadSourceUri,
    downloadSourceProxyUri
  } = await (0, _related_saved_objects.fetchRelatedSavedObjects)(soClient, agentPolicy);
  // Build up an in-memory object for looking up Package Info, so we don't have
  // call `getPackageInfo` for every single policy, which incurs performance costs
  const packageInfoCache = new Map();
  for (const policy of agentPolicy.package_policies) {
    if (!policy.package || packageInfoCache.has((0, _registry.pkgToPkgKey)(policy.package))) {
      continue;
    }

    // Prime the cache w/ just the package key - we'll fetch all the package
    // info concurrently below
    packageInfoCache.set((0, _registry.pkgToPkgKey)(policy.package), {});
  }

  // Fetch all package info concurrently
  await Promise.all(Array.from(packageInfoCache.keys()).map(async pkgKey => {
    const {
      pkgName,
      pkgVersion
    } = (0, _registry.splitPkgKey)(pkgKey);
    const packageInfo = await (0, _packages.getPackageInfo)({
      savedObjectsClient: soClient,
      pkgName,
      pkgVersion
    });
    packageInfoCache.set(pkgKey, packageInfo);
  }));
  const inputs = (await (0, _.storedPackagePoliciesToAgentInputs)(agentPolicy.package_policies, packageInfoCache, getOutputIdForAgentPolicy(dataOutput), agentPolicy.namespace, agentPolicy.global_data_tags)).map(input => {
    // fix output id for default output
    const output = outputs.find(({
      id: outputId
    }) => input.use_output === outputId);
    if (output) {
      input.use_output = getOutputIdForAgentPolicy(output);
    }
    return input;
  });
  const features = (agentPolicy.agent_features || []).reduce((acc, {
    name,
    ...featureConfig
  }) => {
    acc[name] = featureConfig;
    return acc;
  }, {});
  const outputSecretReferences = outputs.flatMap(output => (0, _secrets.getOutputSecretReferences)(output));
  const packagePolicySecretReferences = (((_agentPolicy = agentPolicy) === null || _agentPolicy === void 0 ? void 0 : _agentPolicy.package_policies) || []).flatMap(policy => policy.secret_references || []);
  const fullAgentPolicy = {
    id: agentPolicy.id,
    outputs: {
      ...outputs.reduce((acc, output) => {
        acc[getOutputIdForAgentPolicy(output)] = transformOutputToFullPolicyOutput(output, output.proxy_id ? proxies.find(proxy => output.proxy_id === proxy.id) : undefined, standalone);
        return acc;
      }, {})
    },
    inputs,
    secret_references: [...outputSecretReferences, ...packagePolicySecretReferences],
    revision: agentPolicy.revision,
    agent: {
      download: {
        sourceURI: downloadSourceUri,
        ...(downloadSourceProxyUri ? {
          proxy_url: downloadSourceProxyUri
        } : {})
      },
      monitoring: getFullMonitoringSettings(agentPolicy, monitoringOutput),
      features,
      protection: {
        enabled: agentPolicy.is_protected,
        uninstall_token_hash: '',
        signing_key: ''
      }
    },
    signed: {
      data: '',
      signature: ''
    }
  };
  if (agentPolicy.space_ids) {
    fullAgentPolicy.namespaces = agentPolicy.space_ids;
  }
  const packagePoliciesByOutputId = Object.keys(fullAgentPolicy.outputs).reduce((acc, outputId) => {
    acc[outputId] = [];
    return acc;
  }, {});
  (agentPolicy.package_policies || []).forEach(packagePolicy => {
    const packagePolicyDataOutput = packagePolicy.output_id ? outputs.find(output => output.id === packagePolicy.output_id) : undefined;
    if (packagePolicyDataOutput) {
      packagePoliciesByOutputId[getOutputIdForAgentPolicy(packagePolicyDataOutput)].push(packagePolicy);
    } else {
      packagePoliciesByOutputId[getOutputIdForAgentPolicy(dataOutput)].push(packagePolicy);
    }
  });
  const dataPermissionsByOutputId = Object.keys(fullAgentPolicy.outputs).reduce((acc, outputId) => {
    acc[outputId] = {};
    return acc;
  }, {});
  for (const [outputId, packagePolicies] of Object.entries(packagePoliciesByOutputId)) {
    const dataPermissions = await (0, _package_policies_to_agent_permissions.storedPackagePoliciesToAgentPermissions)(packageInfoCache, agentPolicy.namespace, packagePolicies);
    dataPermissionsByOutputId[outputId] = {
      _elastic_agent_checks: {
        cluster: _package_policies_to_agent_permissions.DEFAULT_CLUSTER_PERMISSIONS
      },
      ...(dataPermissions || {})
    };
  }
  const monitoringPermissions = await (0, _monitoring_permissions.getMonitoringPermissions)(soClient, {
    logs: (_agentPolicy$monitori = (_agentPolicy$monitori2 = agentPolicy.monitoring_enabled) === null || _agentPolicy$monitori2 === void 0 ? void 0 : _agentPolicy$monitori2.includes(_constants.dataTypes.Logs)) !== null && _agentPolicy$monitori !== void 0 ? _agentPolicy$monitori : false,
    metrics: (_agentPolicy$monitori3 = (_agentPolicy$monitori4 = agentPolicy.monitoring_enabled) === null || _agentPolicy$monitori4 === void 0 ? void 0 : _agentPolicy$monitori4.includes(_constants.dataTypes.Metrics)) !== null && _agentPolicy$monitori3 !== void 0 ? _agentPolicy$monitori3 : false,
    traces: (_agentPolicy$monitori5 = (_agentPolicy$monitori6 = agentPolicy.monitoring_enabled) === null || _agentPolicy$monitori6 === void 0 ? void 0 : _agentPolicy$monitori6.includes(_constants.dataTypes.Traces)) !== null && _agentPolicy$monitori5 !== void 0 ? _agentPolicy$monitori5 : false
  }, agentPolicy.namespace);
  monitoringPermissions._elastic_agent_checks = {
    cluster: _package_policies_to_agent_permissions.DEFAULT_CLUSTER_PERMISSIONS
  };

  // Only add permissions if output.type is "elasticsearch"
  fullAgentPolicy.output_permissions = Object.keys(fullAgentPolicy.outputs).reduce((outputPermissions, outputId) => {
    const output = fullAgentPolicy.outputs[outputId];
    if (output && (output.type === _constants.outputType.Elasticsearch || output.type === _constants.outputType.RemoteElasticsearch)) {
      const permissions = {};
      if (outputId === getOutputIdForAgentPolicy(monitoringOutput)) {
        Object.assign(permissions, monitoringPermissions);
      }
      if (outputId === getOutputIdForAgentPolicy(dataOutput) || packagePoliciesByOutputId[outputId].length > 0) {
        Object.assign(permissions, dataPermissionsByOutputId[outputId]);
      }
      outputPermissions[outputId] = permissions;
    }
    return outputPermissions;
  }, {});

  // only add fleet server hosts if not in standalone
  if (!standalone && fleetServerHosts) {
    fullAgentPolicy.fleet = generateFleetConfig(fleetServerHosts, proxies);
  }
  const settingsValues = (0, _form_settings.getSettingsValuesForAgentPolicy)('AGENT_POLICY_ADVANCED_SETTINGS', agentPolicy);
  Object.entries(settingsValues).forEach(([settingsKey, settingValue]) => {
    (0, _saferLodashSet.set)(fullAgentPolicy, settingsKey, settingValue);
  });

  // populate protection and signed properties
  const messageSigningService = _app_context.appContextService.getMessageSigningService();
  if (messageSigningService && fullAgentPolicy.agent) {
    var _await$appContextServ, _appContextService$ge;
    const publicKey = await messageSigningService.getPublicKey();
    const tokenHash = (_await$appContextServ = await ((_appContextService$ge = _app_context.appContextService.getUninstallTokenService()) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.getHashedTokenForPolicyId(fullAgentPolicy.id))) !== null && _await$appContextServ !== void 0 ? _await$appContextServ : '';
    fullAgentPolicy.agent.protection = {
      enabled: agentPolicy.is_protected,
      uninstall_token_hash: tokenHash,
      signing_key: publicKey
    };
    const dataToSign = {
      id: fullAgentPolicy.id,
      agent: {
        features,
        protection: fullAgentPolicy.agent.protection
      },
      inputs: inputs.map(({
        id: inputId,
        name,
        revision,
        type
      }) => ({
        id: inputId,
        name,
        revision,
        type
      }))
    };
    const {
      data: signedData,
      signature
    } = await messageSigningService.sign(dataToSign);
    fullAgentPolicy.signed = {
      data: signedData.toString('base64'),
      signature
    };
  }
  if (agentPolicy.overrides) {
    return (0, _deepmerge.default)(fullAgentPolicy, agentPolicy.overrides);
  }
  return fullAgentPolicy;
}
function generateFleetConfig(fleetServerHosts, proxies) {
  const config = {
    hosts: fleetServerHosts.host_urls
  };
  const fleetServerHostproxy = fleetServerHosts.proxy_id ? proxies.find(proxy => proxy.id === fleetServerHosts.proxy_id) : null;
  if (fleetServerHostproxy) {
    config.proxy_url = fleetServerHostproxy.url;
    if (fleetServerHostproxy.proxy_headers) {
      config.proxy_headers = fleetServerHostproxy.proxy_headers;
    }
    if (fleetServerHostproxy.certificate_authorities || fleetServerHostproxy.certificate || fleetServerHostproxy.certificate_key) {
      config.ssl = {
        renegotiation: 'never',
        verification_mode: '',
        ...(fleetServerHostproxy.certificate_authorities && {
          certificate_authorities: [fleetServerHostproxy.certificate_authorities]
        }),
        ...(fleetServerHostproxy.certificate && {
          certificate: fleetServerHostproxy.certificate
        }),
        ...(fleetServerHostproxy.certificate_key && {
          key: fleetServerHostproxy.certificate_key
        })
      };
    }
  }
  return config;
}
function transformOutputToFullPolicyOutput(output, proxy, standalone = false) {
  var _configJs$shipper;
  const {
    config_yaml,
    type,
    hosts,
    ca_sha256,
    ca_trusted_fingerprint,
    ssl,
    shipper,
    secrets,
    preset
  } = output;
  const configJs = config_yaml ? (0, _jsYaml.safeLoad)(config_yaml) : {};

  // build logic to read config_yaml and transform it with the new shipper data
  const isShipperDisabled = !(configJs !== null && configJs !== void 0 && configJs.shipper) || (configJs === null || configJs === void 0 ? void 0 : (_configJs$shipper = configJs.shipper) === null || _configJs$shipper === void 0 ? void 0 : _configJs$shipper.enabled) === false;
  let shipperDiskQueueData = {};
  let generalShipperData;
  let kafkaData = {};
  if (type === _constants.outputType.Kafka) {
    var _topics$filter, _topics$filter$;
    const {
      client_id,
      version,
      key,
      compression,
      compression_level,
      username,
      password,
      sasl,
      partition,
      random,
      round_robin,
      hash,
      topic,
      topics,
      headers,
      timeout,
      broker_timeout,
      required_acks
    } = output;
    const kafkaTopic = topic ? topic : topics === null || topics === void 0 ? void 0 : (_topics$filter = topics.filter(t => !t.when)) === null || _topics$filter === void 0 ? void 0 : (_topics$filter$ = _topics$filter[0]) === null || _topics$filter$ === void 0 ? void 0 : _topics$filter$.topic;
    const transformPartition = () => {
      if (!partition) return {};
      switch (partition) {
        case 'random':
          return {
            random: {
              ...(random !== null && random !== void 0 && random.group_events ? {
                group_events: random.group_events
              } : {
                group_events: 1
              })
            }
          };
        case 'round_robin':
          return {
            round_robin: {
              ...(round_robin !== null && round_robin !== void 0 && round_robin.group_events ? {
                group_events: round_robin.group_events
              } : {
                group_events: 1
              })
            }
          };
        case 'hash':
        default:
          return {
            hash: {
              ...(hash !== null && hash !== void 0 && hash.hash ? {
                hash: hash.hash
              } : {
                hash: ''
              })
            }
          };
      }
    };

    /* eslint-enable @typescript-eslint/naming-convention */
    kafkaData = {
      client_id,
      version,
      key,
      compression,
      ...(compression === _constants.kafkaCompressionType.Gzip ? {
        compression_level
      } : {}),
      ...(username ? {
        username
      } : {}),
      ...(password ? {
        password
      } : {}),
      ...(sasl ? {
        sasl
      } : {}),
      partition: transformPartition(),
      topic: kafkaTopic,
      headers: (headers !== null && headers !== void 0 ? headers : []).filter(item => item.key !== '' || item.value !== ''),
      timeout,
      broker_timeout,
      required_acks
    };
  }
  if (shipper) {
    if (!isShipperDisabled) {
      shipperDiskQueueData = buildShipperQueueData(shipper);
    }
    /* eslint-disable @typescript-eslint/naming-convention */
    const {
      loadbalance,
      compression_level,
      queue_flush_timeout,
      max_batch_bytes,
      mem_queue_events
    } = shipper;
    /* eslint-enable @typescript-eslint/naming-convention */

    generalShipperData = {
      loadbalance,
      compression_level,
      queue_flush_timeout,
      max_batch_bytes,
      mem_queue_events
    };
  }
  const newOutput = {
    ...configJs,
    ...shipperDiskQueueData,
    type,
    hosts,
    ...kafkaData,
    ...(!isShipperDisabled ? generalShipperData : {}),
    ...(ca_sha256 ? {
      ca_sha256
    } : {}),
    ...(secrets ? {
      secrets
    } : {}),
    ...(ca_trusted_fingerprint ? {
      'ssl.ca_trusted_fingerprint': ca_trusted_fingerprint
    } : {})
  };
  if (ssl) {
    newOutput.ssl = {
      ...ssl,
      // ssl coming from preconfig
      ...newOutput.ssl // ssl coming from config_yaml
    };
  }
  if (proxy) {
    newOutput.proxy_url = proxy.url;
    if (proxy.proxy_headers) {
      newOutput.proxy_headers = proxy.proxy_headers;
    }
    if (proxy.certificate_authorities) {
      if (!newOutput.ssl) {
        newOutput.ssl = {};
      }
      if (!newOutput.ssl.certificate_authorities) {
        newOutput.ssl.certificate_authorities = [];
      }
      newOutput.ssl.certificate_authorities.push(proxy.certificate_authorities);
    }
    if (proxy.certificate) {
      if (!newOutput.ssl) {
        newOutput.ssl = {};
      }
      newOutput.ssl.certificate = proxy.certificate;
    }
    if (proxy.certificate_key) {
      if (!newOutput.ssl) {
        newOutput.ssl = {};
      }
      newOutput.ssl.key = proxy.certificate_key;
    }
  }
  if (output.type === _constants.outputType.Elasticsearch && standalone) {
    // adding a place_holder as API_KEY
    newOutput.api_key = '${API_KEY}';
  }
  if (output.type === _constants.outputType.RemoteElasticsearch) {
    newOutput.service_token = output.service_token;
  }
  if ((0, _output_helpers.outputTypeSupportPresets)(output.type)) {
    newOutput.preset = preset !== null && preset !== void 0 ? preset : (0, _output_helpers.getDefaultPresetForEsOutput)(config_yaml !== null && config_yaml !== void 0 ? config_yaml : '', _jsYaml.safeLoad);
  }
  return newOutput;
}
function getFullMonitoringSettings(agentPolicy, monitoringOutput) {
  var _agentPolicy$monitori7, _agentPolicy$monitori8, _agentPolicy$monitori9, _agentPolicy$monitori10;
  // Set base beats monitoring settings
  const monitoring = {
    enabled: Boolean(agentPolicy.monitoring_enabled && agentPolicy.monitoring_enabled.length > 0 || ((_agentPolicy$monitori7 = agentPolicy.monitoring_http) === null || _agentPolicy$monitori7 === void 0 ? void 0 : _agentPolicy$monitori7.enabled) || agentPolicy.keep_monitoring_alive),
    logs: false,
    metrics: false,
    traces: false
  };

  // If the agent policy has monitoring enabled for at least one of "logs", "metrics", or "traces"
  // generate a monitoring config for the resulting compiled agent policy
  if (agentPolicy.monitoring_enabled && agentPolicy.monitoring_enabled.length > 0) {
    monitoring.namespace = agentPolicy.namespace;
    monitoring.use_output = getOutputIdForAgentPolicy(monitoringOutput);
    monitoring.logs = agentPolicy.monitoring_enabled.includes(_constants.dataTypes.Logs);
    monitoring.metrics = agentPolicy.monitoring_enabled.includes(_constants.dataTypes.Metrics);
    monitoring.traces = agentPolicy.monitoring_enabled.includes(_constants.dataTypes.Traces);
  }
  if (agentPolicy.monitoring_pprof_enabled !== undefined) {
    monitoring.pprof = {
      enabled: agentPolicy.monitoring_pprof_enabled
    };
  }

  // Conditionally set http monitoring settings
  if ((_agentPolicy$monitori8 = agentPolicy.monitoring_http) !== null && _agentPolicy$monitori8 !== void 0 && _agentPolicy$monitori8.enabled) {
    monitoring.http = {
      enabled: agentPolicy.monitoring_http.enabled,
      ...(agentPolicy.monitoring_http.host && {
        host: agentPolicy.monitoring_http.host
      }),
      ...(agentPolicy.monitoring_http.port && {
        port: agentPolicy.monitoring_http.port
      })
    };
  }

  // Conditionally set diagnostics monitoring settings
  if ((_agentPolicy$monitori9 = agentPolicy.monitoring_diagnostics) !== null && _agentPolicy$monitori9 !== void 0 && _agentPolicy$monitori9.limit || (_agentPolicy$monitori10 = agentPolicy.monitoring_diagnostics) !== null && _agentPolicy$monitori10 !== void 0 && _agentPolicy$monitori10.uploader) {
    monitoring.diagnostics = {};
    if (agentPolicy.monitoring_diagnostics.limit && (agentPolicy.monitoring_diagnostics.limit.interval || typeof agentPolicy.monitoring_diagnostics.limit.burst === 'number')) {
      monitoring.diagnostics.limit = {
        ...(agentPolicy.monitoring_diagnostics.limit.interval && {
          interval: agentPolicy.monitoring_diagnostics.limit.interval
        }),
        ...(typeof agentPolicy.monitoring_diagnostics.limit.burst === 'number' && {
          burst: agentPolicy.monitoring_diagnostics.limit.burst
        })
      };
    }
    if (agentPolicy.monitoring_diagnostics.uploader && (typeof agentPolicy.monitoring_diagnostics.uploader.max_retries === 'number' || agentPolicy.monitoring_diagnostics.uploader.init_dur || agentPolicy.monitoring_diagnostics.uploader.max_dur)) {
      monitoring.diagnostics.uploader = {
        ...(typeof agentPolicy.monitoring_diagnostics.uploader.max_retries === 'number' && {
          max_retries: agentPolicy.monitoring_diagnostics.uploader.max_retries
        }),
        ...(agentPolicy.monitoring_diagnostics.uploader.init_dur && {
          init_dur: agentPolicy.monitoring_diagnostics.uploader.init_dur
        }),
        ...(agentPolicy.monitoring_diagnostics.uploader.max_dur && {
          max_dur: agentPolicy.monitoring_diagnostics.uploader.max_dur
        })
      };
    }
  }
  return monitoring;
}

/**
 * Get id used in full agent policy (sent to the agents)
 * we use "default" for the default policy to avoid breaking changes
 */
function getOutputIdForAgentPolicy(output) {
  if (output.is_default && output.type === _constants.outputType.Elasticsearch) {
    return _constants.DEFAULT_OUTPUT.name;
  }
  return output.id;
}

/* eslint-disable @typescript-eslint/naming-convention */
function buildShipperQueueData(shipper) {
  const {
    disk_queue_enabled,
    disk_queue_path,
    disk_queue_max_size,
    disk_queue_compression_enabled
  } = shipper;
  if (!disk_queue_enabled) return {};
  return {
    shipper: {
      queue: {
        disk: {
          path: disk_queue_path,
          max_size: disk_queue_max_size,
          use_compression: disk_queue_compression_enabled
        }
      }
    }
  };
}
/* eslint-enable @typescript-eslint/naming-convention */