"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerDeployAgentPoliciesTask = registerDeployAgentPoliciesTask;
exports.scheduleDeployAgentPoliciesTask = scheduleDeployAgentPoliciesTask;
var _uuid = require("uuid");
var _common = require("@kbn/spaces-plugin/common");
var _ = require("..");
var _cache = require("../epm/packages/cache");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'fleet:deploy_agent_policies';
function registerDeployAgentPoliciesTask(taskManagerSetup) {
  taskManagerSetup.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: 'Fleet Deploy policies',
      timeout: '5m',
      maxAttempts: 3,
      createTaskRunner: ({
        taskInstance
      }) => {
        const agentPolicyIdsWithSpace = taskInstance.params.agentPolicyIdsWithSpace;
        let cancelled = false;
        return {
          async run() {
            if (!agentPolicyIdsWithSpace.length) {
              return;
            }
            _.appContextService.getLogger().debug(`Deploying ${agentPolicyIdsWithSpace.length} policies`);
            const agentPoliciesIdsIndexedBySpace = agentPolicyIdsWithSpace.reduce((acc, {
              id,
              spaceId = _common.DEFAULT_SPACE_ID
            }) => {
              if (!acc[spaceId]) {
                acc[spaceId] = [];
              }
              acc[spaceId].push(id);
              return acc;
            }, {});
            await (0, _cache.runWithCache)(async () => {
              for (const [spaceId, agentPolicyIds] of Object.entries(agentPoliciesIdsIndexedBySpace)) {
                if (cancelled) {
                  throw new Error('Task has been cancelled');
                }
                await _.agentPolicyService.deployPolicies(_.appContextService.getInternalUserSOClientForSpaceId(spaceId), agentPolicyIds);
              }
            });
          },
          async cancel() {
            cancelled = true;
          }
        };
      }
    }
  });
}
async function scheduleDeployAgentPoliciesTask(taskManagerStart, agentPolicyIdsWithSpace) {
  if (!agentPolicyIdsWithSpace.length) {
    return;
  }
  await taskManagerStart.ensureScheduled({
    id: `${TASK_TYPE}:${(0, _uuid.v4)()}`,
    scope: ['fleet'],
    params: {
      agentPolicyIdsWithSpace
    },
    taskType: TASK_TYPE,
    runAt: new Date(Date.now() + 3 * 1000),
    state: {}
  });
}