"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUpdateTags = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _hooks = require("../../../../hooks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useUpdateTags = () => {
  const {
    notifications
  } = (0, _hooks.useStartServices)();
  const wrapRequest = (0, _react.useCallback)(async (requestFn, onSuccess, successMessage, errorMessage) => {
    try {
      const res = await requestFn();
      if (res.error) {
        throw res.error;
      }
      const hasCompleted = !res.data.actionId;
      const message = successMessage !== null && successMessage !== void 0 ? successMessage : _i18n.i18n.translate('xpack.fleet.updateAgentTags.successNotificationTitle', {
        defaultMessage: 'Tag(s) updated'
      });
      notifications.toasts.addSuccess(message);
      onSuccess(hasCompleted);
    } catch (error) {
      const errorTitle = errorMessage !== null && errorMessage !== void 0 ? errorMessage : _i18n.i18n.translate('xpack.fleet.updateAgentTags.errorNotificationTitle', {
        defaultMessage: 'Tag(s) update failed'
      });
      notifications.toasts.addError(error, {
        title: errorTitle
      });
    }
  }, [notifications.toasts]);
  const updateTags = (0, _react.useCallback)(async (agentId, newTags, onSuccess, successMessage, errorMessage) => {
    await wrapRequest(async () => await (0, _hooks.sendPutAgentTagsUpdate)(agentId, {
      tags: newTags
    }), onSuccess, successMessage, errorMessage);
  }, [wrapRequest]);
  const bulkUpdateTags = (0, _react.useCallback)(async (agents, tagsToAdd, tagsToRemove, onSuccess, successMessage, errorMessage) => {
    await wrapRequest(async () => await (0, _hooks.sendPostBulkAgentTagsUpdate)({
      agents,
      tagsToAdd,
      tagsToRemove,
      includeInactive: true
    }), onSuccess, successMessage, errorMessage);
  }, [wrapRequest]);
  return {
    updateTags,
    bulkUpdateTags
  };
};
exports.useUpdateTags = useUpdateTags;