"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getById = getById;
exports.validateFileNameExtension = validateFileNameExtension;
exports.validateMimeType = validateMimeType;
var _server = require("@kbn/core/server");
var _mimeTypes = _interopRequireDefault(require("mime-types"));
var _file_service = require("../../file_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * A helper that given an ID will return a file or map errors to an http response.
 */
async function getById(fileService, id, _fileKind) {
  let result;
  try {
    result = await fileService.getById({
      id
    });
  } catch (e) {
    let error;
    if (e instanceof _file_service.errors.FileNotFoundError) {
      error = _server.kibanaResponseFactory.notFound({
        body: {
          message: e.message
        }
      });
    } else {
      error = _server.kibanaResponseFactory.custom({
        statusCode: 500,
        body: {
          message: e.message
        }
      });
    }
    return {
      error
    };
  }
  return {
    result
  };
}

/**
 * Validate file kind restrictions on a provided MIME type
 * @param mimeType The MIME type to validate
 * @param fileKind The file kind definition that may contain restrictions
 * @returns `undefined` if the MIME type is valid or there are no restrictions.
 */
function validateMimeType(mimeType, fileKind) {
  if (!mimeType || !fileKind) {
    return;
  }
  const allowedMimeTypes = fileKind.allowedMimeTypes;
  if (!allowedMimeTypes || allowedMimeTypes.length === 0) {
    return;
  }
  if (!allowedMimeTypes.includes(mimeType)) {
    return _server.kibanaResponseFactory.badRequest({
      body: {
        message: `File type is not supported`
      }
    });
  }
}

/**
 * Validate file name extension matches the file's MIME type
 * @param fileName The file name to validate
 * @param file
 * @returns `undefined` if the extension matches the MIME type or if no MIME type is provided.
 */
function validateFileNameExtension(fileName, file) {
  if (!fileName || !file || !file.data.mimeType) {
    return;
  }
  const fileMimeType = file.data.mimeType.trim();
  if (!fileMimeType) {
    return;
  }

  // Extract file extension (handle cases with multiple dots)
  const lastDotIndex = fileName.lastIndexOf('.');
  if (lastDotIndex === -1) {
    // No extension found - this might be intentional for some file types
    return;
  }
  const fileExtension = fileName.substring(lastDotIndex + 1).toLowerCase();
  if (!fileExtension) {
    return;
  }
  const expectedExtensions = _mimeTypes.default.extensions[fileMimeType];
  if (expectedExtensions && !expectedExtensions.includes(fileExtension)) {
    return _server.kibanaResponseFactory.badRequest({
      body: {
        message: `File extension does not match file type`
      }
    });
  }
}